
# deploys Ceph radosgw as an Apache FastCGI application
class ceph::radosgw (
  $rgw_id                           = 'radosgw.gateway',
  $rgw_user                         = $::ceph::params::user_httpd,
  $use_ssl                          = $::ceph::use_ssl,
  $primary_mon                      = $::ceph::primary_mon,

  # RadosGW settings
  $rgw_host                         = $::ceph::rgw_host,
  $rgw_ip                           = $::ceph::rgw_ip,
  $rgw_port                         = $::ceph::rgw_port,
  $swift_endpoint_port              = $::ceph::swift_endpoint_port,
  $rgw_keyring_path                 = $::ceph::rgw_keyring_path,
  $rgw_socket_path                  = $::ceph::rgw_socket_path,
  $rgw_frontends                    = $::ceph::rgw_frontends,
  $rgw_log_file                     = $::ceph::rgw_log_file,
  $rgw_data                         = $::ceph::rgw_data,
  $rgw_dns_name                     = $::ceph::rgw_dns_name,
  $rgw_print_continue               = $::ceph::rgw_print_continue,

  #rgw Keystone settings
  $rgw_use_pki                      = $::ceph::rgw_use_pki,
  $rgw_use_keystone                 = $::ceph::rgw_use_keystone,
  $rgw_keystone_url                 = $::ceph::rgw_keystone_url,
  $rgw_keystone_admin_token         = $::ceph::rgw_keystone_admin_token,
  $rgw_keystone_token_cache_size    = $::ceph::rgw_keystone_token_cache_size,
  $rgw_keystone_accepted_roles      = $::ceph::rgw_keystone_accepted_roles,
  $rgw_keystone_revocation_interval = $::ceph::rgw_keystone_revocation_interval,
  $rgw_s3_auth_use_keystone         = $::ceph::rgw_s3_auth_use_keystone,
  $rgw_nss_db_path                  = $::ceph::rgw_nss_db_path,
  $rgw_large_pool_name              = $::ceph::rgw_large_pool_name,
  $rgw_large_pool_pg_nums           = $::ceph::rgw_large_pool_pg_nums,

  $rgw_init_timeout                 = $::ceph::rgw_init_timeout,

  #rgw Log settings
  $use_syslog                       = $::ceph::use_syslog,
  $syslog_facility                  = $::ceph::syslog_log_facility,
  $syslog_level                     = $::ceph::syslog_log_level,
) {

  $keyring_path     = "/etc/ceph/keyring.${rgw_id}"
  $radosgw_auth_key = "client.${rgw_id}"
  $dir_httpd_root   = '/var/www/radosgw'
  $dir_httpd_log    = $::ceph::params::dir_httpd_log

  package { [
    $::ceph::params::package_radosgw,
    $::ceph::params::package_fastcgi,
    $::ceph::params::package_libnss,
  ]:
    ensure  => 'installed',
  }

  # check out httpd package/service is defined
  if !defined(Package['httpd']) {
    package { 'httpd':
      ensure => 'installed',
      name   => $::ceph::params::package_httpd,
    }
  }

  if !defined(Service['httpd']) {
    service { 'httpd':
      ensure => 'running',
      name   => $::ceph::params::service_httpd,
      enable => true,
    }
  }

  firewall {'012 RadosGW allow':
    chain  => 'INPUT',
    dport  => [ $rgw_port, $swift_endpoint_port ],
    proto  => 'tcp',
    action => accept,
  }

  # All files need to be owned by the rgw / http user.
  File {
    owner => $rgw_user,
    group => $rgw_user,
  }

  ceph_conf {
    "client.${rgw_id}/host":               value => $rgw_host;
    "client.${rgw_id}/keyring":            value => $keyring_path;
    "client.${rgw_id}/rgw_socket_path":    value => $rgw_socket_path;
    "client.${rgw_id}/rgw_frontends":      value => $rgw_frontends;
    "client.${rgw_id}/user":               value => $rgw_user;
    "client.${rgw_id}/rgw_data":           value => $rgw_data;
    "client.${rgw_id}/rgw_dns_name":       value => $rgw_dns_name;
    "client.${rgw_id}/rgw_print_continue": value => $rgw_print_continue;
    "client.${rgw_id}/rgw_init_timeout":   value => $rgw_init_timeout;
  }

  if ($use_ssl) {

    $httpd_ssl = $::ceph::params::dir_httpd_ssl
    exec {'copy OpenSSL certificates':
      command => "scp -r ${rgw_nss_db_path}/* ${primary_mon}:${rgw_nss_db_path} && \
                  ssh ${primary_mon} '/etc/init.d/radosgw restart'",
    }
    exec {"generate SSL certificate on ${name}":
      command => "openssl req -x509 -nodes -days 365 -newkey rsa:2048 -keyout ${httpd_ssl}apache.key -out ${httpd_ssl}apache.crt -subj '/C=RU/ST=Russia/L=Saratov/O=Fuel/OU=CA/CN=localhost'",
      returns => [0,1],
    }
  }

  if ($rgw_use_keystone) {

    ceph_conf {
      "client.${rgw_id}/rgw_keystone_url":                 value => $rgw_keystone_url;
      "client.${rgw_id}/rgw_keystone_admin_token":         value => $rgw_keystone_admin_token;
      "client.${rgw_id}/rgw_keystone_accepted_roles":      value => $rgw_keystone_accepted_roles;
      "client.${rgw_id}/rgw_keystone_token_cache_size":    value => $rgw_keystone_token_cache_size;
      "client.${rgw_id}/rgw_keystone_revocation_interval": value => $rgw_keystone_revocation_interval;
    }

    if ($rgw_s3_auth_use_keystone) {
      ceph_conf {
        "client.${rgw_id}/rgw_s3_auth_use_keystone": value => $rgw_s3_auth_use_keystone;
      }
    } else {
      ceph_conf {
        "client.${rgw_id}/rgw_s3_auth_use_keystone": ensure => 'absent'
      }
    }

    if ($rgw_use_pki) {

      ceph_conf {
        "client.${rgw_id}/nss db path": value => $rgw_nss_db_path;
      }

      # This creates the signing certs used by radosgw to check cert revocation
      #   status from keystone
      exec {'create nss db signing certs':
        command => "openssl x509 -in /etc/keystone/ssl/certs/ca.pem -pubkey | \
          certutil -d ${rgw_nss_db_path} -A -n ca -t 'TCu,Cu,Tuw' && \
          openssl x509 -in /etc/keystone/ssl/certs/signing_cert.pem -pubkey | \
          certutil -A -d ${rgw_nss_db_path} -n signing_cert -t 'P,P,P'",
        user    => $rgw_user,
      }

      Exec["ceph-create-radosgw-keyring-on ${name}"] ->
      Exec['create nss db signing certs']

    } #END rgw_use_pki
  } #END rgw_use_keystone

  if ($::osfamily == 'Debian'){

    file {'/etc/apache2/sites-enabled/rgw.conf':
      ensure => link,
      target => "${::ceph::params::dir_httpd_sites}/rgw.conf",
      owner   => 'root',
      group   => 'root',
    }

    Package[$::ceph::params::package_fastcgi] ->
    File["${::ceph::params::dir_httpd_sites}/rgw.conf"] ->
    File['/etc/apache2/sites-enabled/rgw.conf'] ~>
    Service<| title == 'httpd' |>

  } #END osfamily Debian

  if ! $use_syslog {
    ceph_conf {
      "client.${rgw_id}/log_file":      value => $rgw_log_file;
      "client.${rgw_id}/log_to_syslog": value => $use_syslog;
    }
    file { $rgw_log_file:
      ensure => present,
      mode   => '0755',
    }
  } else {
    file { $rgw_log_file:
      ensure => absent,
    }
  }

  file { [
    $::ceph::params::dir_httpd_ssl,
    "${rgw_data}/ceph-${rgw_id}",
    $rgw_data,
    $dir_httpd_root,
    $rgw_nss_db_path,
  ]:
    ensure  => 'directory',
    mode    => '0755',
    recurse => true,
  }

  file { "${::ceph::params::dir_httpd_sites}/rgw.conf":
    content => template('ceph/rgw.conf.erb'),
    owner   => 'root',
    group   => 'root',
  }

  file { "${dir_httpd_root}/s3gw.fcgi":
    content => template('ceph/s3gw.fcgi.erb'),
    mode    => '0755',
  }

  file {"${::ceph::params::dir_httpd_sites}/fastcgi.conf":
    content => template('ceph/fastcgi.conf.erb'),
    mode    => '0755',
    owner   => 'root',
    group   => 'root',
  }

  exec { "ceph create ${radosgw_auth_key}":
    command => "ceph auth get-or-create ${radosgw_auth_key} osd 'allow rwx' mon 'allow rw'",
    unless  => "ceph auth list | fgrep -qx ${radosgw_auth_key}",
  }

  exec { "Populate ${radosgw_auth_key} keyring":
    command => "ceph auth get-or-create ${radosgw_auth_key} > ${keyring_path}",
    creates => $keyring_path,
  }

  exec { "Create ${rgw_large_pool_name} pool":
    command => "ceph -n ${radosgw_auth_key} osd pool create ${rgw_large_pool_name} ${rgw_large_pool_pg_nums} ${rgw_large_pool_pg_nums}",
    unless  => "rados lspools | grep '^${rgw_large_pool_name}$'",
  }

  file { $keyring_path: mode => '0640', }

  file {"${rgw_data}/ceph-${rgw_id}/done":
    ensure => present,
    mode   => '0644',
  }

  Ceph_conf <||> ->
  Package<| title == 'httpd' |> ->
  Package[ [
    $::ceph::params::package_radosgw,
    $::ceph::params::package_fastcgi,
    $::ceph::params::package_libnss,
  ] ] ->
  File[ [
    "${::ceph::params::dir_httpd_sites}/rgw.conf",
    "${::ceph::params::dir_httpd_sites}/fastcgi.conf",
    "${dir_httpd_root}/s3gw.fcgi",
    $::ceph::params::dir_httpd_ssl,
    "${rgw_data}/ceph-${rgw_id}",
    $rgw_data,
    $dir_httpd_root,
    $rgw_nss_db_path,
    $rgw_log_file,
  ] ] ->
  Exec["ceph create ${radosgw_auth_key}"] ->
  Exec["Populate ${radosgw_auth_key} keyring"] ->
  File["${rgw_data}/ceph-${rgw_id}/done"] ->
  File[$keyring_path] ->
  Exec["Create ${rgw_large_pool_name} pool"] ->
  Firewall['012 RadosGW allow'] ~>
  Service <| title == 'httpd' |>
}
