# Copyright (c) 2013 Mirantis Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import abc

from oslo_config import cfg
from oslo_log import log as logging
import six

from blazar.db import api as db_api

LOG = logging.getLogger(__name__)
CONF = cfg.CONF


@six.add_metaclass(abc.ABCMeta)
class BasePlugin(object):

    resource_type = 'none'
    title = None
    description = None
    monitor = None

    def get_plugin_opts(self):
        """Plugin can expose some options that should be specified in conf file

        For example:

            def get_plugin_opts(self):
            return [
                cfg.StrOpt('mandatory-conf', required=True),
                cfg.StrOpt('optional_conf', default="42"),
            ]
        """
        return []

    def setup(self, conf):
        """Plugin initialization

        :param conf: plugin-specific configurations
        """
        pass

    def to_dict(self):
        return {
            'resource_type': self.resource_type,
            'title': self.title,
            'description': self.description,
        }

    @abc.abstractmethod
    def reserve_resource(self, reservation_id, values):
        """Reserve resource."""
        pass

    def update_reservation(self, reservation_id, values):
        """Update reservation."""
        reservation_values = {
            'resource_id': values['resource_id']
        }
        db_api.reservation_update(reservation_id, reservation_values)

    @abc.abstractmethod
    def on_end(self, resource_id):
        """Delete resource."""
        pass

    @abc.abstractmethod
    def on_start(self, resource_id):
        """Wake up resource."""
        pass

    def before_end(self, resource_id):
        """Take actions before the end of a lease"""
        pass

    def heal_reservations(self, failed_resources, interval_begin,
                          interval_end):
        """Heal reservations which suffer from resource failures.

        :param: failed_resources: failed resources
        :param: interval_begin: start date of the period to heal.
        :param: interval_end: end date of the period to heal.
        :return: a dictionary of {reservation id: flags to update}
                 e.g. {'de27786d-bd96-46bb-8363-19c13b2c6657':
                       {'missing_resources': True}}
        """
        raise NotImplementedError


@six.add_metaclass(abc.ABCMeta)
class BaseMonitorPlugin():
    """Base class of monitor plugin."""
    @abc.abstractmethod
    def is_notification_enabled(self):
        """Check if the notification monitor is enabled."""
        pass

    @abc.abstractmethod
    def get_notification_event_types(self):
        """Get a list of event types of messages to handle."""
        pass

    @abc.abstractmethod
    def get_notification_topics(self):
        """Get a list of topics of notification to subscribe to."""
        pass

    @abc.abstractmethod
    def notification_callback(self, event_type, payload):
        """Handle a notification message.

        It is used as a callback of a notification based resource monitor.

        :param event_type: an event type of a notification.
        :param payload: a payload of a notification.
        :return: a dictionary of {reservation id: flags to update}
                 e.g. {'de27786d-bd96-46bb-8363-19c13b2c6657':
                       {'missing_resources': True}}
        """
        pass

    @abc.abstractmethod
    def is_polling_enabled(self):
        """Check if the polling monitor is enabled."""
        pass

    @abc.abstractmethod
    def get_polling_interval(self):
        """Get an interval of polling in seconds."""
        pass

    @abc.abstractmethod
    def poll(self):
        """Check health of resources.

        :return: a dictionary of {reservation id: flags to update}
                 e.g. {'de27786d-bd96-46bb-8363-19c13b2c6657':
                       {'missing_resources': True}}
        """
        pass

    @abc.abstractmethod
    def get_healing_interval(self):
        """Get interval of reservation healing in minutes."""
        pass

    @abc.abstractmethod
    def heal(self):
        """Heal suffering reservations.

        :return: a dictionary of {reservation id: flags to update}
        """
