#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import functools

import novaclient
from novaclient import api_versions
from novaclient import client as nova_client
import openstack
from oslo_config import cfg
from oslo_log import log

from ceilometer import keystone_client

SERVICE_OPTS = [
    cfg.StrOpt('nova',
               default='compute',
               help='Nova service type.'),
]

LOG = log.getLogger(__name__)


def logged(func):

    @functools.wraps(func)
    def with_logging(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            LOG.exception(e)
            raise

    return with_logging


class Client:
    """A client which gets information via python-novaclient."""

    def __init__(self, conf):
        """Initialize a nova client object."""
        creds = conf.service_credentials

        ks_session = keystone_client.get_session(conf)

        self.nova_client = nova_client.Client(
            version=api_versions.APIVersion('2.1'),
            session=ks_session,

            # nova adapter options
            region_name=creds.region_name,
            endpoint_type=creds.interface,
            service_type=conf.service_types.nova)

        self.image_client = openstack.connection.Connection(
            image_api_version='2',
            session=ks_session,
            region_name=creds.region_name,
            image_interface=creds.interface)

    def _with_flavor_and_image(self, instances):
        flavor_cache = {}
        image_cache = {}
        for instance in instances:
            self._with_flavor(instance, flavor_cache)
            self._with_image(instance, image_cache)

        return instances

    def _with_flavor(self, instance, cache):
        fid = instance.flavor['id']
        if fid in cache:
            flavor = cache.get(fid)
        else:
            try:
                flavor = self.nova_client.flavors.get(fid)
            except novaclient.exceptions.NotFound:
                flavor = None
            cache[fid] = flavor

        attr_defaults = [('name', 'unknown-id-%s' % fid),
                         ('vcpus', 0), ('ram', 0), ('disk', 0),
                         ('ephemeral', 0)]

        for attr, default in attr_defaults:
            if not flavor:
                instance.flavor[attr] = default
                continue
            instance.flavor[attr] = getattr(flavor, attr, default)

    def _with_image(self, instance, cache):
        try:
            iid = instance.image['id']
        except TypeError:
            instance.image = None
            instance.kernel_id = None
            instance.ramdisk_id = None
            return

        if iid in cache:
            image = cache.get(iid)
        else:
            try:
                image = self.image_client.image.get_image(iid)
            except openstack.exceptions.NotFoundException:
                image = None
            cache[iid] = image

        attr_defaults = [('kernel_id', None),
                         ('ramdisk_id', None)]

        instance.image['name'] = (
            getattr(image, 'name') if image else 'unknown-id-%s' % iid)

        for attr, default in attr_defaults:
            ameta = getattr(image, attr, default) if image else default
            setattr(instance, attr, ameta)

        if image:
            image_meta = {"base_image_ref": iid}
            # Notifications and libvirt XML metadata return all
            # image_meta values as strings. Do the same here.
            image_meta.update((k, str(v))
                              for k, v in image.items()
                              if k not in ('id', 'name', 'metadata',
                                           'kernel_id', 'ramdisk_id'))
        else:
            image_meta = {}
        instance.image_meta = image_meta

    @logged
    def instance_get_all_by_host(self, hostname, since=None):
        """Returns list of instances on particular host.

        If since is supplied, it will return the instances changed since that
        datetime. since should be in ISO Format '%Y-%m-%dT%H:%M:%SZ'
        """
        search_opts = {'host': hostname, 'all_tenants': True}
        if since:
            search_opts['changes-since'] = since
        return self._with_flavor_and_image(self.nova_client.servers.list(
            detailed=True,
            search_opts=search_opts))
