/*
 * Decompiled with CFR 0.152.
 */
package com.googlesource.gerrit.plugins.deleteproject.fs;

import com.google.gerrit.extensions.annotations.PluginName;
import com.google.gerrit.extensions.api.changes.NotifyHandling;
import com.google.gerrit.extensions.events.ProjectDeletedListener;
import com.google.gerrit.extensions.registration.DynamicSet;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.server.config.PluginConfigFactory;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.gerrit.server.project.ProjectResource;
import com.google.inject.Inject;
import com.googlesource.gerrit.plugins.deleteproject.CannotDeleteProjectException;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import org.eclipse.jgit.errors.RepositoryNotFoundException;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilesystemDeleteHandler {
    private static final Logger log = LoggerFactory.getLogger(FilesystemDeleteHandler.class);
    private final GitRepositoryManager repoManager;
    private final DynamicSet<ProjectDeletedListener> deletedListener;
    private final PluginConfigFactory cfgFactory;
    private final String pluginName;

    @Inject
    public FilesystemDeleteHandler(GitRepositoryManager repoManager, DynamicSet<ProjectDeletedListener> deletedListener, PluginConfigFactory cfgFactory, @PluginName String pluginName) {
        this.repoManager = repoManager;
        this.deletedListener = deletedListener;
        this.cfgFactory = cfgFactory;
        this.pluginName = pluginName;
    }

    public void delete(Project project, boolean preserveGitRepository) throws IOException, RepositoryNotFoundException {
        Repository repository = this.repoManager.openRepository(project.getNameKey());
        File repoFile = repository.getDirectory();
        this.cleanCache(repository);
        if (!preserveGitRepository) {
            this.deleteGitRepository(project.getNameKey(), repoFile);
        }
    }

    public void assertCanDelete(ProjectResource rsrc, boolean preserveGitRepository) throws CannotDeleteProjectException {
        if (!preserveGitRepository && !this.cfgFactory.getFromGerritConfig(this.pluginName).getBoolean("allowDeletionOfReposWithTags", true)) {
            this.assertHasNoTags(rsrc);
        }
    }

    private void assertHasNoTags(ProjectResource rsrc) throws CannotDeleteProjectException {
        try (Repository repo = this.repoManager.openRepository(rsrc.getNameKey());){
            if (!repo.getRefDatabase().getRefs("refs/tags/").isEmpty()) {
                throw new CannotDeleteProjectException(String.format("Project %s has tags", rsrc.getName()));
            }
        }
        catch (IOException e) {
            throw new CannotDeleteProjectException(e);
        }
    }

    private void deleteGitRepository(final Project.NameKey project, File repoFile) throws IOException {
        Path basePath = this.getBasePath(repoFile.toPath(), project);
        Path trash = this.moveToTrash(repoFile.toPath(), basePath, project);
        boolean ok = false;
        try {
            this.recursiveDelete(trash);
            ok = true;
        }
        catch (IOException e) {
            log.warn("Error trying to delete " + trash, (Throwable)e);
        }
        if (ok) {
            try {
                this.recursiveDeleteParent(repoFile.getParentFile(), basePath.toFile());
            }
            catch (IOException e) {
                log.warn("Couldn't delete (empty) parents of " + repoFile, (Throwable)e);
            }
        }
        ProjectDeletedListener.Event event = new ProjectDeletedListener.Event(){

            public String getProjectName() {
                return project.get();
            }

            public NotifyHandling getNotify() {
                return NotifyHandling.NONE;
            }
        };
        for (ProjectDeletedListener l : this.deletedListener) {
            try {
                l.onProjectDeleted(event);
            }
            catch (RuntimeException e) {
                log.warn("Failure in ProjectDeletedListener", (Throwable)e);
            }
        }
    }

    private Path getBasePath(Path repo, Project.NameKey project) {
        Path projectPath = Paths.get(project.get(), new String[0]);
        return repo.getRoot().resolve(repo.subpath(0, repo.getNameCount() - projectPath.getNameCount()));
    }

    private Path moveToTrash(Path directory, Path basePath, Project.NameKey nameKey) throws IOException {
        Path trashRepo = basePath.resolve(nameKey.get() + "." + System.currentTimeMillis() + ".%deleted%.git");
        return Files.move(directory, trashRepo, StandardCopyOption.ATOMIC_MOVE);
    }

    private void cleanCache(Repository repository) {
        repository.close();
        RepositoryCache.close((Repository)repository);
    }

    private void recursiveDelete(Path file) throws IOException {
        Files.walkFileTree(file, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException e) throws IOException {
                if (e != null) {
                    throw e;
                }
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private void recursiveDeleteParent(File file, File until) throws IOException {
        if (file.equals(until)) {
            return;
        }
        if (file.listFiles().length == 0) {
            File parent = file.getParentFile();
            Files.delete(file.toPath());
            this.recursiveDeleteParent(parent, until);
        }
    }
}

