# Copyright (C) 2012 Hewlett-Packard Development Company, L.P.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""
Client side of the volume backup RPC API.
"""


from oslo_config import cfg
from oslo_log import log as logging

from cinder import rpc


CONF = cfg.CONF
LOG = logging.getLogger(__name__)


class BackupAPI(rpc.RPCAPI):
    """Client side of the volume rpc API.

    API version history:

        1.0 - Initial version.
        1.1 - Changed methods to accept backup objects instead of IDs.
        1.2 - A version that got in by mistake (without breaking anything).
        1.3 - Dummy version bump to mark start of having cinder-backup service
              decoupled from cinder-volume.
    """

    RPC_API_VERSION = '1.3'
    TOPIC = CONF.backup_topic
    BINARY = 'cinder-backup'

    def create_backup(self, ctxt, backup):
        LOG.debug("create_backup in rpcapi backup_id %s", backup.id)
        cctxt = self.client.prepare(server=backup.host, version='1.1')
        cctxt.cast(ctxt, 'create_backup', backup=backup)

    def restore_backup(self, ctxt, volume_host, backup, volume_id):
        LOG.debug("restore_backup in rpcapi backup_id %s", backup.id)
        cctxt = self.client.prepare(server=volume_host, version='1.1')
        cctxt.cast(ctxt, 'restore_backup', backup=backup,
                   volume_id=volume_id)

    def delete_backup(self, ctxt, backup):
        LOG.debug("delete_backup  rpcapi backup_id %s", backup.id)
        cctxt = self.client.prepare(server=backup.host, version='1.1')
        cctxt.cast(ctxt, 'delete_backup', backup=backup)

    def export_record(self, ctxt, backup):
        LOG.debug("export_record in rpcapi backup_id %(id)s "
                  "on host %(host)s.",
                  {'id': backup.id,
                   'host': backup.host})
        cctxt = self.client.prepare(server=backup.host, version='1.1')
        return cctxt.call(ctxt, 'export_record', backup=backup)

    def import_record(self,
                      ctxt,
                      host,
                      backup,
                      backup_service,
                      backup_url,
                      backup_hosts):
        LOG.debug("import_record rpcapi backup id %(id)s "
                  "on host %(host)s for backup_url %(url)s.",
                  {'id': backup.id,
                   'host': host,
                   'url': backup_url})
        cctxt = self.client.prepare(server=host, version='1.1')
        cctxt.cast(ctxt, 'import_record',
                   backup=backup,
                   backup_service=backup_service,
                   backup_url=backup_url,
                   backup_hosts=backup_hosts)

    def reset_status(self, ctxt, backup, status):
        LOG.debug("reset_status in rpcapi backup_id %(id)s "
                  "on host %(host)s.",
                  {'id': backup.id,
                   'host': backup.host})
        cctxt = self.client.prepare(server=backup.host, version='1.1')
        return cctxt.cast(ctxt, 'reset_status', backup=backup, status=status)

    def check_support_to_force_delete(self, ctxt, host):
        LOG.debug("Check if backup driver supports force delete "
                  "on host %(host)s.", {'host': host})
        cctxt = self.client.prepare(server=host, version='1.1')
        return cctxt.call(ctxt, 'check_support_to_force_delete')
