#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Client side of the rock RPC API."""

from oslo.config import cfg
from oslo import messaging

from rock import exception
from rock.i18n import _, _LW
from rock.objects import base as objects_base
from rock.openstack.common import jsonutils
from rock import rpc

CONF = cfg.CONF

rpcapi_opts = [
    cfg.StrOpt('rock_topic',
               default='rock_api',
               help='The topic rock nodes listen on'),
]
CONF.register_opts(rpcapi_opts)

rpcapi_cap_opt = cfg.StrOpt('rock',
        help='Set a version cap for messages sent to conductor services')
CONF.register_opt(rpcapi_cap_opt, 'upgrade_levels')

def _rock_host(host, instance):
    '''Get the destination host for a message.

    :param host: explicit host to send the message to.
    :param instance: If an explicit host was not specified, use
                     instance['host']

    :returns: A host
    '''
    if host:
        return host
    if not instance:
        raise exception.RockException(_('No compute host specified'))
    if not instance['host']:
        raise exception.RockException(_('Unable to find host for '
                                        'Instance %s') % instance['uuid'])
    return instance['host']


class RockAPI(object):
    """Client side of the rock RPC API

    API version history:

    * 1.0 - Initial version.

    """

    VERSION_ALIASES = {
        'init': '1.0',
    }

    def __init__(self):
        super(RockAPI, self).__init__()
        target = messaging.Target(topic=CONF.rock_topic, version='1.0')
        version_cap = self.VERSION_ALIASES.get(CONF.upgrade_levels.rock,
                                               CONF.upgrade_levels.rock)
        serializer = objects_base.RockObjectSerializer()
        self.client = rpc.get_client(target,
                                     version_cap=version_cap,
                                     serializer=serializer)

    def allocate_accelerator(self, ctxt, instance, accelerator_request):
        version = '1.0'
        cctxt = self.client.prepare(version=version)
        return cctxt.call(ctxt, 'allocate_accelerator',
                          instance=instance, accelerator_request=accelerator_request)

    def detach_accelerator(self, ctxt, instance, pci_address):
        version = '1.0'
        cctxt = self.client.prepare(version=version)
        cctxt.cast(ctxt, 'detach_accelerator',
                   instance=instance, pci_address=pci_address)
