#    Copyright 2013 IBM Corp.
#    Copyright 2013 Red Hat, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Client side of the conductor RPC API."""

from oslo.config import cfg
from oslo import messaging

from rock.objects import base as objects_base
from rock.openstack.common import jsonutils
from rock import rpc

CONF = cfg.CONF

rpcapi_cap_opt = cfg.StrOpt('conductor',
        help='Set a version cap for messages sent to conductor services')
CONF.register_opt(rpcapi_cap_opt, 'upgrade_levels')


class ConductorAPI(object):
    """Client side of the conductor RPC API

    API version history:

    * 1.0 - Initial version.

    """

    VERSION_ALIASES = {
        'init': '1.0',
    }

    def __init__(self):
        super(ConductorAPI, self).__init__()
        target = messaging.Target(topic=CONF.conductor.topic)
        version_cap = self.VERSION_ALIASES.get(CONF.upgrade_levels.conductor,
                                               CONF.upgrade_levels.conductor)
        serializer = objects_base.RockObjectSerializer()
        self.client = rpc.get_client(target,
                                     version_cap=version_cap,
                                     serializer=serializer)

    def service_get_all_by(self, context, topic=None, host=None, binary=None):
        cctxt = self.client.prepare(version='1.0')
        return cctxt.call(context, 'service_get_all_by',
                          topic=topic, host=host, binary=binary)

    def service_create(self, context, values):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'service_create', values=values)

    def service_destroy(self, context, service_id):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'service_destroy', service_id=service_id)

    def compute_node_create(self, context, values):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'compute_node_create', values=values)

    def compute_node_update(self, context, node, values):
        node_p = jsonutils.to_primitive(node)
        cctxt = self.client.prepare()
        return cctxt.call(context, 'compute_node_update',
                          node=node_p, values=values)

    def compute_node_delete(self, context, node):
        node_p = jsonutils.to_primitive(node)
        cctxt = self.client.prepare()
        return cctxt.call(context, 'compute_node_delete', node=node_p)

    def service_update(self, context, service, values):
        service_p = jsonutils.to_primitive(service)
        cctxt = self.client.prepare()
        return cctxt.call(context, 'service_update',
                          service=service_p, values=values)

    def task_log_get(self, context, task_name, begin, end, host, state=None):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'task_log_get',
                          task_name=task_name, begin=begin, end=end,
                          host=host, state=state)

    def task_log_begin_task(self, context, task_name, begin, end, host,
                            task_items=None, message=None):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'task_log_begin_task',
                          task_name=task_name,
                          begin=begin, end=end, host=host,
                          task_items=task_items, message=message)

    def task_log_end_task(self, context, task_name, begin, end, host, errors,
                          message=None):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'task_log_end_task',
                          task_name=task_name, begin=begin, end=end,
                          host=host, errors=errors, message=message)

    def object_class_action(self, context, objname, objmethod, objver,
                            args, kwargs):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'object_class_action',
                          objname=objname, objmethod=objmethod,
                          objver=objver, args=args, kwargs=kwargs)

    def object_action(self, context, objinst, objmethod, args, kwargs):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'object_action', objinst=objinst,
                          objmethod=objmethod, args=args, kwargs=kwargs)

    def object_backport(self, context, objinst, target_version):
        cctxt = self.client.prepare()
        return cctxt.call(context, 'object_backport', objinst=objinst,
                          target_version=target_version)

    def accelerator_filter(self, context, host_name, acc_specs):
        """ rock filter for Nova
        host_name: str of destination host name
        acc_specs: request of accelerator. format like: {'AES:pps':'2000', 'IPSEC:Num':'1'} """
        cctxt = self.client.prepare()
        return cctxt.call(context, 'accelerator_filter', host_name=host_name,
                          acc_specs=acc_specs)
         
                          
