# Copyright (c) 2013 Intel, Inc.
# Copyright (c) 2013 OpenStack Foundation
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import copy
import time

from rock import exception
from rock.common import dict_util
from rock.i18n import _LE
from rock.openstack.common import jsonutils
from rock.openstack.common import log as logging
from rock.device import driver as rock_driver

LOG = logging.getLogger(__name__)

class RockDeviceStats(object):

    """Rock accelerators summary information.

    According to the ROCK spec, a ROCK physical function can have up to
    16 accelerator virtual functions, thus the number of assignable accelerator
    functions in a cloud can be big. The scheduler needs to know all accelerator
    availability information in order to determine which compute hosts can
    support a Rock request. Passing individual virtual accelerator information
    to the scheduler does not scale, so we provide summary information.

    Usually the virtual functions provided by a host Rock device have the same
    value for most properties, like accelerator_type, capability_unit and class type.
    The Rock stats class summarizes this information for the scheduler.

    The rock stats information is maintained exclusively by compute node
    resource tracker and updated to database. The scheduler fetches the
    information and selects the compute node accordingly. If a comptue
    node is selected, the resource tracker allocates the accelerators to the
    instance and updates the accelerator stats information.

    This summary information will be helpful for cloud management also.
    """

    pool_keys = ['num', 'pps', 'bps']

    def __init__(self, stats=None):
        super(RockDeviceStats, self).__init__()
        self.driver = rock_driver.load_rock_driver()
        self.pools = {}
        if not stats:
            for driver_element in self.driver:
                dev_type = driver_element.get_device_type()
                self.pools[dev_type] = driver_element.get_total_resource()
        else :
            self.pools = jsonutils.loads(stats)
        self.capability = None

    def _has_capability(self, pools, accelerator):
        return all(all(pools.get(atom_type).get(prop) >= accelerator.get(atom_type).get(prop)
                       for prop in self.pool_keys)
                   for atom_type in accelerator.get('acc_capability') if atom_type != 'count')

    def has_capability(self, cap_req):
        # LOG.info("pools ==> %s, request ==> %s" % (self.pools, cap_req))
        return cmp(self.pools, cap_req)

    def allocate_accelerator(self, acc_list):

        # 0. check request accelerator list
        # for acc in acc_list:
        #     if acc['function_type'] != 'type-VF' or acc['status'] != 'claimed':
        #         LOG.warn("Remove invalid request accelerator : %s" % acc)
        #         acc_list.remove(acc)

        # 1. check accelerator capability
        req_cap_dict = {}
        for acc in acc_list:
            req_cap_dict[acc['device_type']] = req_cap_dict[acc['device_type']] + (
            acc["acc_capability"],) if req_cap_dict.has_key(acc['device_type']) else (acc["acc_capability"],)
        req_cap_total = {}
        for k, v in req_cap_dict.items():
            req_cap_total[k] = dict_util._dict_sum(v)
        if not self.has_capability(req_cap_total):
            raise exception.AcceleratorCapInsufficient(pool=self.pools, req_total=req_cap_total,
                                                       req_list='\n'.join(acc_list))

        # 2. check accelerator state and device state
        for acc in acc_list:
            dev_driver = rock_driver.get_rock_driver(acc['device_type'])
            if not dev_driver:
                device_type = acc['device_type']
                raise exception.AcceleratorDriverFailed(dev_type=device_type)
            if not dev_driver.get_resource_state(acc):
                LOG.error("Get accelerator[%s] state error." % acc)
                acc_id = acc['id']
                raise exception.AcceleratorStateFailed(accelerator_id=acc_id)

        # 3. subtract capability from pool
        for accelerator in acc_list:
            device_type = accelerator['device_type']
            acc_capability = acc["acc_capability"]
            LOG.info("pools ====> %s, subtract ::::> %s" % (self.pools[device_type], acc_capability))
            self.pools[device_type] = dict_util._dict_subtract(self.pools[device_type], acc_capability)
            # LOG.info("pools ====> %s" % (self.pools[device_type]))

        # 4. attach accelerator with driver
        try:
            for acc in acc_list:
                dev_driver = rock_driver.get_rock_driver(acc['device_type'])
                dev_driver.set_vf_resource(acc)
        except:
            LOG.error("Set accelerator capability exception")
            self.detach_accelerator(acc_list)

    def attach_accelerator(self, acc):
        """Add a accelerator to the rock device."""
        device_type = acc['device_type']
        acc_capability = acc["acc_capability"]
        if not isinstance(acc_capability, dict):
            acc_capability = jsonutils.loads(acc_capability)
        # Check state of device and accelerator
        dev_driver = rock_driver.get_rock_driver(device_type)
        if not dev_driver:
            raise exception.AcceleratorDriverFailed(dev_type=device_type)
        if not dev_driver.get_resource_state(acc):
            LOG.error("Get accelerator[%s] state error." % acc)
            raise exception.AcceleratorStateFailed(accelerator_id=acc['id'])
        # subtract allocated accelerator capability from cache
        self.pools[device_type] = dict_util._dict_subtract(self.pools[device_type], acc_capability)
        # Set visual function capability using device driver
        dev_driver.set_vf_resource(acc)

    def detach_accelerator(self, acc):
        """Remove one device from the rock device."""
        dev_type = acc['device_type']
        if 'accelerators' in self.pools[dev_type] and acc in self.pools[dev_type]['accelerators']:
            self.pools[dev_type]['accelerators'].remove(acc)
        else:
            return
        acc_sub_type = acc['acc_sub_type']
        capability = acc['acc_capability']

        pools_sub_type = self.pools[dev_type][acc_sub_type]
        self.pools[dev_type]['free_vf_num'] += 1
        pools_sub_type['free_vf_num'] += 1
        for atom_type in capability:
            if atom_type == 'count':
                continue
            for atom_cap_key in capability[atom_type]:
                if atom_cap_key == 'num':
                    continue
                pools_sub_type[atom_type][atom_cap_key] += capability[atom_type][atom_cap_key]
        LOG.info("Accelerator[%s] capability pools == %s " % (acc_sub_type, pools_sub_type))
        LOG.info("Accelerator capability in memory == %s " % self.pools[dev_type][acc_sub_type])

    def free_accelerator(self, acc_list):
        LOG.info("accelerator list =======> %s " % acc_list)
        #  Give capability back to pool
        for accelerator in acc_list:
            if accelerator['function_type'] == 'type-VF' and accelerator['status'] in ['claimed', 'allocated']:
                device_type = accelerator['device_type']
                acc_capability = accelerator["acc_capability"]
                LOG.info("pools before add ====> %s, Add capability ::::> %s" % (self.pools[device_type], acc_capability))
                self.pools[device_type] = dict_util.dict_acc_back(self.pools[device_type], acc_capability)
                dev_driver = rock_driver.get_rock_driver(device_type)
                dev_driver.release_resource(accelerator)
                LOG.info("pools after added ====> %s" % (self.pools[device_type]))
            else:
                LOG.warn("Remove invalid [release] request accelerator : %s" % accelerator)

    @property
    def get_stat_dict(self):
        # 'accelerators' shouldn't be part of stats
        pools = {}
        LOG.info(type(self.pools))
        for key in self.pools.keys():
            tmp = dict((k, v) for k, v in self.pools[key].iteritems() if k != 'accelerators')
            pools[key] = tmp
        return pools

    def clear(self):
        """Clear all the stats maintained."""
        self.pools = {}

    def syn_pools(self):
        stat_cap = {}
        tmp = {}
        for driver_element in self.driver:
            dev_type = driver_element.get_device_type()
            new_info = driver_element.get_total_resource()
            tmp[dev_type] = dict((k, v) for k, v in self.pools[dev_type].iteritems() if k != 'accelerators')
            if not self.capability:
                self.capability = {}
                self.capability[dev_type] = new_info
            elif not dev_type in self.capability.keys():
                self.capability[dev_type] = new_info
            else:
                cache_cap, new_cap, stat_cap[dev_type] = calculate_dict(self.capability[dev_type], new_info, tmp[dev_type])
        if stat_cap:
            return stat_cap
        else:
            return self.capability

def calculate_dict(exist_value_dict, new_value_dict, stat_dict):
    new_key_set = set(new_value_dict.keys())
    exist_key_set = set(exist_value_dict.keys())
    for extra_key in new_key_set - exist_key_set:
        exist_value_dict[extra_key] = new_value_dict[extra_key]
        stat_dict[extra_key] = new_value_dict[extra_key]
        LOG.info('Add    key-[%s] from existed accelerator value-[%s] and statistic value-[%s]' % (
            extra_key, new_value_dict[extra_key], new_value_dict[extra_key]))
    # LOG.info('new_key_set - exist_key_set ==> %s \nexist_key_set - new_key_set ==> %s' %(new_key_set - exist_key_set, exist_key_set - new_key_set))
    for remove_key in exist_key_set - new_key_set:
        remove_stat = stat_dict.pop(remove_key)
        remove_exist = exist_value_dict.pop(remove_key)
        LOG.warn('Remove key-[%s] from existed accelerator value-[%s] and statistic value-[%s]' % (
            remove_key, remove_exist, remove_stat))
    # LOG.info('======old>====%s'%exist_value_dict)
    # LOG.info('======new>====%s'%new_value_dict)
    # LOG.info('======tmp>====%s'%stat_dict)
    # synchronize capability from old exist stat value to new hardware driver value
    for exist_key, exist_value in exist_value_dict.iteritems():
        new_exist = new_value_dict[exist_key]
        # LOG.info('key:  %s, exist_value: %s, new_value: %s, equal:%s' % (exist_key, exist_value, new_exist, new_exist == exist_value))
        # the same value, not changed
        if new_exist == exist_value:
            continue
        # dict value, iterate calculate
        if isinstance(new_exist, dict) and isinstance(exist_value, dict):
            exist_value, new_exist, stat_dict[exist_key] = calculate_dict(exist_value, new_exist, stat_dict[exist_key])
        # string value, change old existed value to new value
        elif isinstance(new_exist, str) and isinstance(exist_value, str):
            stat_dict[exist_key] = new_exist
            exist_value_dict[exist_key] = new_exist
        # number value(int or float), add stat value with the growth quantity from existed value to new value
        elif isinstance(new_exist, (int, float)) and isinstance(exist_value, (int,float)):
            stat_dict[exist_key] += new_exist - exist_value
            exist_value_dict[exist_key] = new_exist
        else:
            LOG.warn(
                'Existed accelerator dict : %s ,key: %s -> [type:%s] value:%s \n'
                ' not match with \n'
                '   New  accelerator dict : %s ,key: %s -> [type:%s] value:%s ' % (
                exist_value_dict, exist_key, type(exist_value), exist_value, new_value_dict, exist_key, type(new_exist),
                new_exist))
    return exist_value_dict, new_value_dict, stat_dict