# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Starter script for Rock Agent."""

import os
import sys
import traceback

curPath = os.path.abspath(os.path.dirname(__file__))
rootPath = os.path.split(curPath)[0]
sys.path.append(rootPath)

from oslo.config import cfg

from rock.conductor import rpcapi as conductor_rpcapi
from rock import config
import rock.db.api
from rock import exception
from rock.i18n import _
from rock import objects
from rock.objects import base as objects_base
from rock.openstack.common import log as logging
from rock.openstack.common.report import guru_meditation_report as gmr
from rock import service
from rock import utils
from rock import version

CONF = cfg.CONF
CONF.import_opt('agent_topic', 'rock.agent.rpcapi')
CONF.import_opt('use_local', 'rock.conductor.api', group='conductor')


def block_db_access():
    class NoDB(object):
        def __getattr__(self, attr):
            return self

        def __call__(self, *args, **kwargs):
            stacktrace = "".join(traceback.format_stack())
            LOG = logging.getLogger('rock.agent')
            LOG.error(_('No db access allowed in rock-agent: %s'),
                      stacktrace)
            raise exception.DBNotAllowed('rock-agent')

    rock.db.api.IMPL = NoDB()


def main():
    config.parse_args(sys.argv)
    logging.setup('rock')
    utils.monkey_patch()
    objects.register_all()

    gmr.TextGuruMeditation.setup_autorun(version)

    if not CONF.conductor.use_local:
        block_db_access()
        objects_base.RockObject.indirection_api = \
            conductor_rpcapi.ConductorAPI()

    server = service.Service.create(binary='rock-agent',
                                    topic=CONF.agent_topic,
                                    db_allowed=CONF.conductor.use_local)
    service.serve(server)
    service.wait()
