# Copyright (c) 2013 Intel, Inc.
# Copyright (c) 2012 OpenStack Foundation
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

# Service function of accelerator device
# We must validate status of accelerator device first when using service,
# and change status immediately when operate the accelerator device successfully
# The status of accelerator device as follows:
#   available:accelerator is found and can be used, that means free
#   claimed  :accelerator is claimed to be used by a [vm] instance, but not allocate
#             to the instance, it's a mid-status between available and allocated,
#             usually lasts a very short time.
#   allocated:accelerator is allocated to a [vm] instance.
#   removed  :accelerator has been found before, but now can't find it
# The status flow chart is:
#   [available]  ----claim---->  [claimed]  ----allocate---->  [allocated]
#      \                             |                               /
#       \--------<------free---------+--------------<---------------/

import functools

from rock import exception
from rock.openstack.common import jsonutils
from rock.openstack.common import log as logging

LOG = logging.getLogger(__name__)

def validate_accelerator_status(accelerator_status=None):
    """Decorator to validate status of accelerator before changing it."""

    if accelerator_status is not None and not isinstance(accelerator_status, set):
        accelerator_status = set(accelerator_status)

    def outer(f):
        @functools.wraps(f)
        def inner(acc_accelerator, instance=None):
            if acc_accelerator['status'] not in accelerator_status:
                raise exception.AcceleratorDeviceInvalidStatus(
                    compute_node_id=acc_accelerator.compute_node_id,
                    address=acc_accelerator.address, status=acc_accelerator.status,
                    hopestatus=accelerator_status)
            if instance:
                return f(acc_accelerator, instance)
            else:
                return f(acc_accelerator)
        return inner
    return outer

@validate_accelerator_status(accelerator_status=['available'])
def claim(acc_accelerator, instance = None):
    acc_accelerator.status = 'claimed'
    if not instance:
        acc_accelerator.instance_uuid = instance['uuid']

@validate_accelerator_status(accelerator_status=['available', 'claimed'])
def allocate(acc_accelerator, instance = None):
    if instance and acc_accelerator.status == 'claimed' and acc_accelerator.instance_uuid != instance['uuid']:
        raise exception.AcceleratorDeviceInvalidOwner(
            compute_node_id=acc_accelerator.compute_node_id,
            address=acc_accelerator.address, owner=acc_accelerator.instance_uuid,
            hopeowner=instance['uuid'])
    acc_accelerator.status = 'allocated'
    if instance:
        acc_accelerator.instance_uuid = instance['uuid']

@validate_accelerator_status(accelerator_status=['available'])
def remove(acc_accelerator):
    acc_accelerator.status = 'removed'
    acc_accelerator.instance_uuid = None

@validate_accelerator_status(accelerator_status=['claimed', 'allocated'])
def free(acc_accelerator, instance=None):
    if instance and acc_accelerator.instance_uuid != instance['uuid']:
        raise exception.AcceleratorDeviceInvalidOwner(
            compute_node_id=acc_accelerator.compute_node_id,
            address=acc_accelerator.address, owner=acc_accelerator.instance_uuid,
            hopeowner=instance['uuid'])
    old_status = acc_accelerator.status
    acc_accelerator.status = 'available'
    acc_accelerator.instance_uuid = None
    if old_status == 'allocated' and instance:
        existed = next((dev for dev in instance['rock_devices']
            if dev.id == acc_accelerator.id))
        if isinstance(instance, dict):
            instance['rock_devices'].remove(existed)
        else:
            instance.rock_devices.objects.remove(existed)


def update_accelerator(acc_accelerator, acc_dict):
    """Sync the property from accelerator dictionary to accelerator object.

    The resource tracker running on compute node updates the available accelerator
    information periodically.To avoid meaningless syncs with the database, update
    the accelerator object only when the property changed.
    """
    no_changes = ('status', 'instance_uuid', 'address', 'id', 'extra_info', 'belong_pf_id')
    map(lambda x: acc_dict.pop(x, None),
        [key for key in no_changes])

    for k, v in acc_dict.items():
        if ('acc_capability' == k):
            temp_capability = {}
            for key, element in v.items():
                temp_capability[key] = element
                if isinstance(element, str):
                    temp_capability[key] = element
                else:
                    temp_capability[key] = jsonutils.dumps(element)
            acc_accelerator[k] = temp_capability
        elif k in acc_accelerator.fields.keys():
            acc_accelerator[k] = v
        else:
            extra_info = acc_accelerator.extra_info
            extra_info.update({k: v})
            acc_accelerator.extra_info = extra_info
