# Copyright 2012 OpenStack Foundation
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from migrate.changeset import UniqueConstraint
from migrate import ForeignKeyConstraint
from sqlalchemy import Boolean, BigInteger, Column, DateTime, Enum, Float
from sqlalchemy import dialects
from sqlalchemy import ForeignKey, Index, Integer, MetaData, String, Table
from sqlalchemy import Text
from sqlalchemy.types import NullType

from rock.i18n import _
from rock.openstack.common import log as logging

LOG = logging.getLogger(__name__)


# Note on the autoincrement flag: this is defaulted for primary key columns
# of integral type, so is no longer set explicitly in such cases.

# NOTE(dprince): This wrapper allows us to easily match the Folsom MySQL
# Schema. In Folsom we created tables as latin1 and converted them to utf8
# later. This conversion causes some of the Text columns on MySQL to get
# created as mediumtext instead of just text.
def MediumText():
    return Text().with_variant(dialects.mysql.MEDIUMTEXT(), 'mysql')


def Inet():
    return String(length=43).with_variant(dialects.postgresql.INET(),
                  'postgresql')


def InetSmall():
    return String(length=39).with_variant(dialects.postgresql.INET(),
                  'postgresql')


def _create_shadow_tables(migrate_engine):
    meta = MetaData(migrate_engine)
    meta.reflect(migrate_engine)
    table_names = meta.tables.keys()

    meta.bind = migrate_engine

    for table_name in table_names:
        table = Table(table_name, meta, autoload=True)

        columns = []
        for column in table.columns:
            column_copy = None
            # NOTE(boris-42): BigInteger is not supported by sqlite, so
            #                 after copy it will have NullType, other
            #                 types that are used in rock are supported by
            #                 sqlite.
            if isinstance(column.type, NullType):
                column_copy = Column(column.name, BigInteger(), default=0)
            if table_name == 'instances' and column.name == 'locked_by':
                enum = Enum('owner', 'admin',
                            name='instances0locked_by'.upper())
                column_copy = Column(column.name, enum)
            else:
                column_copy = column.copy()
            columns.append(column_copy)

        shadow_table_name = 'shadow_' + table_name
        shadow_table = Table(shadow_table_name, meta, *columns,
                             mysql_engine='InnoDB')
        try:
            shadow_table.create(checkfirst=True)
        except Exception:
            LOG.info(repr(shadow_table))
            LOG.exception(_('Exception while creating table.'))
            raise


def upgrade(migrate_engine):
    meta = MetaData()
    meta.bind = migrate_engine

    compute_nodes = Table('compute_nodes', meta,
        Column('created_at', DateTime),
        Column('updated_at', DateTime),
        Column('deleted_at', DateTime),
        Column('id', Integer, primary_key=True, nullable=False),
        Column('service_id', Integer, nullable=False),
        Column('hypervisor_hostname', String(length=255)),
        Column('deleted', Integer),
        Column('host_ip', InetSmall()),
        Column('accelerator_stats', Text, nullable=True),
        Column('mac', String(length=32)),                      # ACC_BOARD mac . use to remote access
        mysql_engine='InnoDB',
        mysql_charset='utf8'
    )

    task_log = Table('task_log', meta,
        Column('created_at', DateTime),
        Column('updated_at', DateTime),
        Column('deleted_at', DateTime),
        Column('id', Integer, primary_key=True, nullable=False),
        Column('task_name', String(length=255), nullable=False),
        Column('state', String(length=255), nullable=False),
        Column('host', String(length=255), nullable=False),
        Column('period_beginning', DateTime, nullable=False),
        Column('period_ending', DateTime, nullable=False),
        Column('message', String(length=255), nullable=False),
        Column('task_items', Integer),
        Column('errors', Integer),
        Column('deleted', Integer),
        mysql_engine='InnoDB',
        mysql_charset='utf8'
    )

    services = Table('services', meta,
        Column('created_at', DateTime),
        Column('updated_at', DateTime),
        Column('deleted_at', DateTime),
        Column('id', Integer, primary_key=True, nullable=False),
        Column('host', String(length=255)),
        Column('binary', String(length=255)),
        Column('topic', String(length=255)),
        Column('report_count', Integer, nullable=False),
        Column('disabled', Boolean),
        Column('deleted', Integer),
        Column('disabled_reason', String(length=255)),
        mysql_engine='InnoDB',
        mysql_charset='utf8'
    )

    accelerators = Table('accelerators',meta,
        Column('created_at', DateTime),
        Column('updated_at', DateTime),
        Column('deleted_at', DateTime),
        Column('deleted', DateTime),
        Column('id', Integer, primary_key=True, nullable=False),
        Column('belong_pf_id', Integer),
        Column('vf_number', Integer),
        Column('dev_id', String(length=255)),
        Column('mac', String(length=32)),
        Column('device_type', String(length=12), nullable=False),
        Column('acc_type', String(length=8), nullable=False),
        Column('acc_sub_type', String(length=8), nullable=False),
        Column('acc_capability', Text),
        Column('queue_num', Integer),
        Column('queue_type', String(length=8)),
        Column('address', String(length=12), nullable=False),
        Column('vendor_id', String(length=4)),
        Column('product_id', String(length=4)),
        Column('function_type', String(length=8), nullable=False),
        Column('remotable', Integer),
        Column('remote_mac', String(length=32)),
        Column('status', String(length=36), nullable=False),
        Column('numa_node', Integer),
        Column('instance_uuid', String(length=36)),
        Column('compute_node_id', Integer, nullable=False),
        Column('belong_network', String(length=255)),
        Column('extra_info', Text),
        Column('request_id', String(length=36)),
        Column('config_flag', Integer),
        mysql_engine='InnoDB',
        mysql_charset='utf8'
    )

    # create all tables
    tables = [compute_nodes, services, task_log]

    for table in tables:
        try:
            table.create()
        except Exception:
            LOG.info(repr(table))
            LOG.exception(_('Exception while creating table.'))
            raise

    # task log unique constraint
    task_log_uc = "uniq_task_log0task_name0host0period_beginning0period_ending"
    task_log_cols = ('task_name', 'host', 'period_beginning', 'period_ending')
    uc = UniqueConstraint(*task_log_cols, table=task_log, name=task_log_uc)
    uc.create()

    # services
    UniqueConstraint('host', 'topic', 'deleted',
                     table=services,
                     name='uniq_services0host0topic0deleted').create()
    UniqueConstraint('host', 'binary', 'deleted',
                     table=services,
                     name='uniq_services0host0binary0deleted').create()


    # MySQL specific indexes
    if migrate_engine.name == 'mysql':
        # mysql-specific index by leftmost 100 chars.  (mysql gets angry if the
        # index key length is too long.)
        sql = ("create index migrations_by_host_nodes_and_status_idx ON "
               "migrations (deleted, source_compute(100), dest_compute(100), "
               "source_node(100), dest_node(100), status)")
        migrate_engine.execute(sql)


    MYSQL_INDEX_SKIPS = [
        # we create this one manually for MySQL above
    ]

        # Common indexes (indexes we apply to all databases)
    # NOTE: order specific for MySQL diff support
    common_indexes = [
    ]

    for index in common_indexes:
        if migrate_engine.name == 'mysql' and \
            index.name in MYSQL_INDEX_SKIPS:
            continue
        else:
            index.create(migrate_engine)

    fkeys = [
              [[compute_nodes.c.service_id],
                  [services.c.id],
                  'fk_compute_nodes_service_id'],
            ]

    for fkey_pair in fkeys:
        if migrate_engine.name == 'mysql':
            # For MySQL we name our fkeys explicitly so they match Havana
            fkey = ForeignKeyConstraint(columns=fkey_pair[0],
                                   refcolumns=fkey_pair[1],
                                   name=fkey_pair[2])
            fkey.create()
        elif migrate_engine.name == 'postgresql':
            # PostgreSQL names things like it wants (correct and compatible!)
            fkey = ForeignKeyConstraint(columns=fkey_pair[0],
                                   refcolumns=fkey_pair[1])
            fkey.create()

    if migrate_engine.name == 'mysql':
        # In Folsom we explicitly converted migrate_version to UTF8.
        migrate_engine.execute(
            'ALTER TABLE migrate_version CONVERT TO CHARACTER SET utf8')
        # Set default DB charset to UTF8.
        migrate_engine.execute(
            'ALTER DATABASE %s DEFAULT CHARACTER SET utf8' %
            migrate_engine.url.database)

    _create_shadow_tables(migrate_engine)


def downgrade(migrate_engine):
    raise NotImplementedError('Downgrade from Kilo is unsupported.')
