# Copyright (c) 2011 X.commerce, a business unit of eBay Inc.
# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Implementation of SQLAlchemy backend."""

import collections
import copy
import datetime
import functools
import sys
import threading
import time
import uuid

from oslo.config import cfg
from oslo.db import exception as db_exc
from oslo.db.sqlalchemy import session as db_session
from oslo.db.sqlalchemy import utils as sqlalchemyutils
import six
from sqlalchemy import and_
from sqlalchemy import Boolean
from sqlalchemy.exc import NoSuchTableError
from sqlalchemy import Integer
from sqlalchemy import MetaData
from sqlalchemy import or_
from sqlalchemy.orm import contains_eager
from sqlalchemy.orm import joinedload
from sqlalchemy.orm import joinedload_all
from sqlalchemy.orm import noload
from sqlalchemy.orm import undefer
from sqlalchemy.schema import Table
from sqlalchemy import sql
from sqlalchemy.sql.expression import asc
from sqlalchemy.sql.expression import desc
from sqlalchemy.sql import false
from sqlalchemy.sql import func
from sqlalchemy.sql import null
from sqlalchemy.sql import true
from sqlalchemy import String
from rock.openstack.common import jsonutils

import rock.context
from rock.common import dict_util
from rock.db.sqlalchemy import models
from rock import exception
from rock.i18n import _
from rock.openstack.common import excutils
from rock.openstack.common import log as logging
from rock.openstack.common import timeutils
from rock.openstack.common import uuidutils

db_opts = [
    cfg.StrOpt('osapi_compute_unique_server_name_scope',
               default='',
               help='When set, compute API will consider duplicate hostnames '
                    'invalid within the specified scope, regardless of case. '
                    'Should be empty, "project" or "global".'),
]

CONF = cfg.CONF
CONF.register_opts(db_opts)
CONF.import_opt('agent_topic', 'rock.agent.rpcapi')

LOG = logging.getLogger(__name__)


_ENGINE_FACADE = None
_LOCK = threading.Lock()


def _create_facade_lazily():
    global _LOCK, _ENGINE_FACADE
    if _ENGINE_FACADE is None:
        with _LOCK:
            if _ENGINE_FACADE is None:
                _ENGINE_FACADE = db_session.EngineFacade.from_config(CONF)
    return _ENGINE_FACADE


def get_engine(use_slave=False):
    facade = _create_facade_lazily()
    return facade.get_engine(use_slave=use_slave)


def get_session(use_slave=False, **kwargs):
    facade = _create_facade_lazily()
    return facade.get_session(use_slave=use_slave, **kwargs)


_SHADOW_TABLE_PREFIX = 'shadow_'
_DEFAULT_QUOTA_NAME = 'default'
PER_PROJECT_QUOTAS = ['fixed_ips', 'floating_ips', 'networks']


def get_backend():
    """The backend is this module itself."""
    return sys.modules[__name__]


def require_admin_context(f):
    """Decorator to require admin request context.

    The first argument to the wrapped function must be the context.

    """

    @functools.wraps(f)
    def wrapper(*args, **kwargs):
        rock.context.require_admin_context(args[0])
        return f(*args, **kwargs)
    return wrapper


def require_context(f):
    """Decorator to require *any* user or admin context.

    This does no authorization for user or project access matching, see
    :py:func:`rock.context.authorize_project_context` and
    :py:func:`rock.context.authorize_user_context`.

    The first argument to the wrapped function must be the context.

    """

    @functools.wraps(f)
    def wrapper(*args, **kwargs):
        rock.context.require_context(args[0])
        return f(*args, **kwargs)
    return wrapper

def _retry_on_deadlock(f):
    """Decorator to retry a DB API call if Deadlock was received."""
    @functools.wraps(f)
    def wrapped(*args, **kwargs):
        while True:
            try:
                return f(*args, **kwargs)
            except db_exc.DBDeadlock:
                LOG.warn(_("Deadlock detected when running "
                           "'%(func_name)s': Retrying..."),
                           dict(func_name=f.__name__))
                # Retry!
                time.sleep(0.5)
                continue
    functools.update_wrapper(wrapped, f)
    return wrapped


def model_query(context, model, *args, **kwargs):
    """Query helper that accounts for context's `read_deleted` field.

    :param context: context to query under
    :param use_slave: If true, use slave_connection
    :param session: if present, the session to use
    :param read_deleted: if present, overrides context's read_deleted field.
    :param project_only: if present and context is user-type, then restrict
            query to match the context's project_id. If set to 'allow_none',
            restriction includes project_id = None.
    :param base_model: Where model_query is passed a "model" parameter which is
            not a subclass of RockBase, we should pass an extra base_model
            parameter that is a subclass of RockBase and corresponds to the
            model parameter.
    """

    use_slave = kwargs.get('use_slave') or False
    if CONF.database.slave_connection == '':
        use_slave = False

    session = kwargs.get('session') or get_session(use_slave=use_slave)
    read_deleted = kwargs.get('read_deleted') or context.read_deleted
    project_only = kwargs.get('project_only', False)

    def issubclassof_rock_base(obj):
        return isinstance(obj, type) and issubclass(obj, models.RockBase)

    base_model = model
    if not issubclassof_rock_base(base_model):
        base_model = kwargs.get('base_model', None)
        if not issubclassof_rock_base(base_model):
            raise Exception(_("model or base_model parameter should be "
                              "subclass of RockBase"))

    query = session.query(model, *args)

    default_deleted_value = base_model.__mapper__.c.deleted.default.arg
    if read_deleted == 'no':
        query = query.filter(base_model.deleted == default_deleted_value)
    elif read_deleted == 'yes':
        pass  # omit the filter to include deleted and active
    elif read_deleted == 'only':
        query = query.filter(base_model.deleted != default_deleted_value)
    else:
        raise Exception(_("Unrecognized read_deleted value '%s'")
                            % read_deleted)

    if rock.context.is_user_context(context) and project_only:
        if project_only == 'allow_none':
            query = query.\
                filter(or_(base_model.project_id == context.project_id,
                           base_model.project_id == null()))
        else:
            query = query.filter_by(project_id=context.project_id)

    return query


def exact_filter(query, model, filters, legal_keys):
    """Applies exact match filtering to a query.

    Returns the updated query.  Modifies filters argument to remove
    filters consumed.

    :param query: query to apply filters to
    :param model: model object the query applies to, for IN-style
                  filtering
    :param filters: dictionary of filters; values that are lists,
                    tuples, sets, or frozensets cause an 'IN' test to
                    be performed, while exact matching ('==' operator)
                    is used for other values
    :param legal_keys: list of keys to apply exact filtering to
    """

    filter_dict = {}

    # Walk through all the keys
    for key in legal_keys:
        # Skip ones we're not filtering on
        if key not in filters:
            continue

        # OK, filtering on this key; what value do we search for?
        value = filters.pop(key)

        if key in ('metadata', 'system_metadata'):
            column_attr = getattr(model, key)
            if isinstance(value, list):
                for item in value:
                    for k, v in item.iteritems():
                        query = query.filter(column_attr.any(key=k))
                        query = query.filter(column_attr.any(value=v))

            else:
                for k, v in value.iteritems():
                    query = query.filter(column_attr.any(key=k))
                    query = query.filter(column_attr.any(value=v))
        elif isinstance(value, (list, tuple, set, frozenset)):
            # Looking for values in a list; apply to query directly
            column_attr = getattr(model, key)
            query = query.filter(column_attr.in_(value))
        else:
            # OK, simple exact match; save for later
            filter_dict[key] = value

    # Apply simple exact matches
    if filter_dict:
        query = query.filter_by(**filter_dict)

    return query


def convert_objects_related_datetimes(values, *datetime_keys):
    for key in datetime_keys:
        if key in values and values[key]:
            if isinstance(values[key], six.string_types):
                values[key] = timeutils.parse_strtime(values[key])
            # NOTE(danms): Strip UTC timezones from datetimes, since they're
            # stored that way in the database
            values[key] = values[key].replace(tzinfo=None)
    return values

###################


def constraint(**conditions):
    return Constraint(conditions)


def equal_any(*values):
    return EqualityCondition(values)


def not_equal(*values):
    return InequalityCondition(values)


class Constraint(object):

    def __init__(self, conditions):
        self.conditions = conditions

    def apply(self, model, query):
        for key, condition in self.conditions.iteritems():
            for clause in condition.clauses(getattr(model, key)):
                query = query.filter(clause)
        return query


class EqualityCondition(object):

    def __init__(self, values):
        self.values = values

    def clauses(self, field):
        # method signature requires us to return an iterable even if for OR
        # operator this will actually be a single clause
        return [or_(*[field == value for value in self.values])]


class InequalityCondition(object):

    def __init__(self, values):
        self.values = values

    def clauses(self, field):
        return [field != value for value in self.values]


###################


@require_admin_context
def service_destroy(context, service_id):
    session = get_session()
    with session.begin():
        count = model_query(context, models.Service, session=session).\
                    filter_by(id=service_id).\
                    soft_delete(synchronize_session=False)

        if count == 0:
            raise exception.ServiceNotFound(service_id=service_id)

        model_query(context, models.ComputeNode, session=session).\
                    filter_by(service_id=service_id).\
                    soft_delete(synchronize_session=False)


def _service_get(context, service_id, with_compute_node=True, session=None,
                 use_slave=False):
    query = model_query(context, models.Service, session=session,
                        use_slave=use_slave).\
                     filter_by(id=service_id)

    if with_compute_node:
        query = query.options(joinedload('compute_node'))

    result = query.first()
    if not result:
        raise exception.ServiceNotFound(service_id=service_id)

    return result


@require_admin_context
def service_get(context, service_id, with_compute_node=False,
                use_slave=False):
    return _service_get(context, service_id,
                        with_compute_node=with_compute_node,
                        use_slave=use_slave)


@require_admin_context
def service_get_all(context, disabled=None):
    query = model_query(context, models.Service)

    if disabled is not None:
        query = query.filter_by(disabled=disabled)

    return query.all()


@require_admin_context
def service_get_all_by_topic(context, topic):
    return model_query(context, models.Service, read_deleted="no").\
                filter_by(disabled=False).\
                filter_by(topic=topic).\
                all()


@require_admin_context
def service_get_by_host_and_topic(context, host, topic):
    return model_query(context, models.Service, read_deleted="no").\
                filter_by(disabled=False).\
                filter_by(host=host).\
                filter_by(topic=topic).\
                first()


@require_admin_context
def service_get_all_by_host(context, host):
    return model_query(context, models.Service, read_deleted="no").\
                filter_by(host=host).\
                all()


@require_admin_context
def service_get_by_compute_host(context, host, use_slave=False):
    result = model_query(context, models.Service, read_deleted="no",
                         use_slave=use_slave).\
                options(joinedload('compute_node')).\
                filter_by(host=host).\
                filter_by(topic=CONF.agent_topic).\
                first()

    if not result:
        raise exception.ComputeHostNotFound(host=host)

    return result


@require_admin_context
def service_get_by_args(context, host, binary):
    result = model_query(context, models.Service).\
                     filter_by(host=host).\
                     filter_by(binary=binary).\
                     first()

    if not result:
        raise exception.HostBinaryNotFound(host=host, binary=binary)

    return result


@require_admin_context
def service_create(context, values):
    service_ref = models.Service()
    service_ref.update(values)
    if not CONF.enable_new_services:
        service_ref.disabled = True
    try:
        service_ref.save()
    except db_exc.DBDuplicateEntry as e:
        if 'binary' in e.columns:
            raise exception.ServiceBinaryExists(host=values.get('host'),
                        binary=values.get('binary'))
        raise exception.ServiceTopicExists(host=values.get('host'),
                        topic=values.get('topic'))
    return service_ref


@require_admin_context
@_retry_on_deadlock
def service_update(context, service_id, values):
    session = get_session()
    with session.begin():
        service_ref = _service_get(context, service_id,
                                   with_compute_node=False, session=session)
        service_ref.update(values)

    return service_ref


###################

def compute_node_get(context, compute_id):
    return _compute_node_get(context, compute_id)


def _compute_node_get(context, compute_id, session=None):
    result = model_query(context, models.ComputeNode, session=session).\
            filter_by(id=compute_id).\
            options(joinedload('service')).\
            first()

    if not result:
        raise exception.ComputeHostNotFound(host=compute_id)

    return result


@require_admin_context
def compute_node_get_by_service_id(context, service_id):
    result = model_query(context, models.ComputeNode, read_deleted='no').\
        filter_by(service_id=service_id).\
        first()

    if not result:
        raise exception.ServiceNotFound(service_id=service_id)

    return result


@require_admin_context
def compute_node_get_all(context, no_date_fields):

    # NOTE(msdubov): Using lower-level 'select' queries and joining the tables
    #                manually here allows to gain 3x speed-up and to have 5x
    #                less network load / memory usage compared to the sqla ORM.

    engine = get_engine()

    # Retrieve ComputeNode, Service
    compute_node = models.ComputeNode.__table__
    service = models.Service.__table__

    with engine.begin() as conn:
        redundant_columns = set(['deleted_at', 'created_at', 'updated_at',
                                 'deleted']) if no_date_fields else set([])

        def filter_columns(table):
            return [c for c in table.c if c.name not in redundant_columns]

        compute_node_query = sql.select(filter_columns(compute_node)).\
                                where(compute_node.c.deleted == 0).\
                                order_by(compute_node.c.service_id)
        compute_node_rows = conn.execute(compute_node_query).fetchall()

        service_query = sql.select(filter_columns(service)).\
                            where((service.c.deleted == 0) &
                                  (service.c.binary == 'rock-agent')).\
                            order_by(service.c.id)
        service_rows = conn.execute(service_query).fetchall()

    # Join ComputeNode & Service manually.
    services = {}
    for proxy in service_rows:
        services[proxy['id']] = dict(proxy.items())

    compute_nodes = []
    for proxy in compute_node_rows:
        node = dict(proxy.items())
        node['service'] = services.get(proxy['service_id'])

        compute_nodes.append(node)

    return compute_nodes


@require_admin_context
def compute_node_search_by_hypervisor(context, hypervisor_match):
    field = models.ComputeNode.hypervisor_hostname
    return model_query(context, models.ComputeNode).\
            options(joinedload('service')).\
            filter(field.like('%%%s%%' % hypervisor_match)).\
            all()


@require_admin_context
def compute_node_create(context, values):
    """Creates a new ComputeNode and populates the capacity fields
    with the most recent data.
    """
    datetime_keys = ('created_at', 'deleted_at', 'updated_at')
    convert_objects_related_datetimes(values, *datetime_keys)

    compute_node_ref = models.ComputeNode()
    compute_node_ref.update(values)
    compute_node_ref.save()

    return compute_node_ref


@require_admin_context
@_retry_on_deadlock
def compute_node_update(context, compute_id, values):
    """Updates the ComputeNode record with the most recent data."""

    session = get_session()
    with session.begin():
        compute_ref = _compute_node_get(context, compute_id, session=session)
        # Always update this, even if there's going to be no other
        # changes in data.  This ensures that we invalidate the
        # scheduler cache of compute node data in case of races.
        values['updated_at'] = timeutils.utcnow()
        datetime_keys = ('created_at', 'deleted_at', 'updated_at')
        convert_objects_related_datetimes(values, *datetime_keys)
        compute_ref.update(values)

    return compute_ref


@require_admin_context
def compute_node_delete(context, compute_id):
    """Delete a ComputeNode record."""
    session = get_session()
    with session.begin():
        result = model_query(context, models.ComputeNode, session=session).\
                 filter_by(id=compute_id).\
                 soft_delete(synchronize_session=False)

        if not result:
            raise exception.ComputeHostNotFound(host=compute_id)


###################


def _task_log_get_query(context, task_name, period_beginning,
                        period_ending, host=None, state=None, session=None):
    query = model_query(context, models.TaskLog, session=session).\
                     filter_by(task_name=task_name).\
                     filter_by(period_beginning=period_beginning).\
                     filter_by(period_ending=period_ending)
    if host is not None:
        query = query.filter_by(host=host)
    if state is not None:
        query = query.filter_by(state=state)
    return query


@require_admin_context
def task_log_get(context, task_name, period_beginning, period_ending, host,
                 state=None):
    return _task_log_get_query(context, task_name, period_beginning,
                               period_ending, host, state).first()


@require_admin_context
def task_log_get_all(context, task_name, period_beginning, period_ending,
                     host=None, state=None):
    return _task_log_get_query(context, task_name, period_beginning,
                               period_ending, host, state).all()


@require_admin_context
def task_log_begin_task(context, task_name, period_beginning, period_ending,
                        host, task_items=None, message=None):

    task = models.TaskLog()
    task.task_name = task_name
    task.period_beginning = period_beginning
    task.period_ending = period_ending
    task.host = host
    task.state = "RUNNING"
    if message:
        task.message = message
    if task_items:
        task.task_items = task_items
    try:
        task.save()
    except db_exc.DBDuplicateEntry:
        raise exception.TaskAlreadyRunning(task_name=task_name, host=host)


@require_admin_context
def task_log_end_task(context, task_name, period_beginning, period_ending,
                      host, errors, message=None):
    values = dict(state="DONE", errors=errors)
    if message:
        values["message"] = message

    session = get_session()
    with session.begin():
        rows = _task_log_get_query(context, task_name, period_beginning,
                                       period_ending, host, session=session).\
                        update(values)
        if rows == 0:
            # It's not running!
            raise exception.TaskNotRunning(task_name=task_name, host=host)


###################


@require_admin_context
def accelerator_create(context, values):
    accelerator_ref = models.Accelerator()
    accelerator_ref.update(values)
    accelerator_ref.save()
    return accelerator_ref

@require_context
def accelerator_get_all_by_instance_uuid(context, instance_uuid):
    return model_query(context, models.Accelerator).\
                       filter_by(status='allocated').\
                       filter_by(instance_uuid=instance_uuid).\
                       all()

@require_admin_context
def accelerator_get_by_id(context, id):
    accelerator_ref = model_query(context, models.Accelerator).\
                        filter_by(id=id).\
                        first()
    if not accelerator_ref:
        raise exception.AcceleratorNotFoundById(id=id)
    return accelerator_ref

@require_admin_context
def accelerator_get_by_address(context, address):
    accelerator_ref = model_query(context, models.Accelerator).\
                         filter_by(address=address).\
                         first()
    if not accelerator_ref:
        raise exception.AcceleratorNotFoundByAddress(address=address)
    return accelerator_ref

@require_admin_context
def accelerator_get_all_by_node(context, node_id):
    return model_query(context, models.Accelerator).\
                       filter_by(compute_node_id=node_id).\
                       all()

def _accelertor_get(context, id, with_compute_node=True, session=None,
                 use_slave=False):
    query = model_query(context, models.Accelerator, session=session,
                        use_slave=use_slave).\
                     filter_by(id=id)

    if with_compute_node:
        query = query.options(joinedload('compute_node'))

    result = query.first()
    if not result:
        raise exception.AcceleratorNotFound(id=id)

    return result

def _instance_accdevs_get_multi(context, instance_uuids, session=None):
    return model_query(context, models.Accelerator, session=session).\
        filter_by(status='allocated').\
        filter(models.Accelerator.instance_uuid.in_(instance_uuids))

@require_admin_context
def accelerator_update(context, node_id, address, values):
    session = get_session()
    with session.begin():
        device = model_query(context, models.Accelerator, session=session,
                             read_deleted="no").\
                        filter_by(compute_node_id=node_id).\
                        filter_by(address=address).\
                        first()
        if not device:
            device = models.Accelerator()
        device.update(values)
        session.add(device)
    return device

@require_admin_context
def accelerator_delete_by_id(context, id):
    session = get_session()
    with session.begin():
        session.query(models.Accelerator).\
        filter_by(id=id).\
        update({'deleted': True,
            'deleted_at': timeutils.utcnow(),
             'updated_at': timeutils.utcnow()})

def accelerator_update_by_uuid(context, instance_uuid, values):
    return model_query(context, models.Accelerator).\
        filter_by(instance_uuid=instance_uuid).\
        update(values)

@require_admin_context
def accelerator_delete_by_node_id(context, compute_node_id):
    session = get_session()
    with session.begin():
        session.query(models.Accelerator).\
        filter_by(compute_node_id=compute_node_id).\
        update({'deleted': True,
            'deleted_at': timeutils.utcnow(),
             'updated_at': timeutils.utcnow()})

@require_admin_context
def accelerator_destroy(context, node_id, address):
    result = model_query(context, models.Accelerator).\
                         filter_by(compute_node_id=node_id).\
                         filter_by(address=address).\
                         soft_delete()
    if not result:
        raise exception.AcceleratorNotFound(node_id=node_id, address=address)

def instance_claim(context, instance, acc_list):
    LOG.info("=============> sql -----------")
    session = get_session()
    instance_uuid = instance['uuid']
    if instance_uuid is None:
        raise exception.InstanceUUIDIsNone()

    node_name = instance['node']
    if node_name is None:
        raise exception.InstanceNodeNameIsNone(instance_uuid=instance_uuid)

    host = instance['host']
    if host is None:
        raise exception.InstanceHostIsNone(instance_uuid=instance_uuid)

    with session.begin():
        # get cpmpute_node_id by nodename and host
        result = model_query(context, models.ComputeNode, session=session).\
            filter_by(hypervisor_hostname=node_name).\
            options(joinedload('service')).\
            filter(models.Service.host==host).\
            first()
        if not result:
            raise exception.ComputeNodeNotFoundByNode(host=host, node_name=node_name)


        update_info = {}
        update_acc_board_pf_list = []
        update_vf_list = []
        for acc in acc_list:
            acc_type = acc.get('acc_type')
            remote_disable = acc.get('remote_disable')  #reserve
            capabilitys = acc.get('acc_capability',{})
            dev_type = acc.get('device_type')     #reserve
            numa_node = acc.get('numa_node')   #reserve

            if dev_type is not None:
                # reserve
                # specify device type if ACC_BOARD or others
                pass

            if acc_type is None:
                raise exception.AccTypeNotFound()
            # LOG.info("accelerator id::%s, acc_type ==> %s" % (result.id, acc_type))
            # get available vf list in current compute node
            vf_rows = session.query(models.Accelerator).\
                filter_by(compute_node_id=result.id).\
                filter_by(function_type='type-VF').\
                filter_by(acc_type=acc_type).\
                filter_by(status='available').\
                all()

            # LOG.info("accelerator id::%s, vf_rows ==> %s" % (result.id, vf_rows))
            # test session should manual call rollback after exception ??????
            if not vf_rows:
                raise exception.GetAvailableSubTypeVFFailed(compute_node_id=result.id,
                                                            acc_sub_type=acc_type)
            belong_pf_info = {}
            # get available pf info
            for vf_row in vf_rows:
                # LOG.info("vf_row.belong_pf_id ====> %s" % vf_row.belong_pf_id)
                if vf_row.belong_pf_id not in belong_pf_info.keys():
                    belong_pf_info[vf_row.belong_pf_id] = {}
                    belong_pf_info[vf_row.belong_pf_id]['count'] = 1
                    if belong_pf_info[vf_row.belong_pf_id].has_key("vf_ids"):
                        belong_pf_info[vf_row.belong_pf_id]['vf_ids'].append(vf_row.id)
                    else:
                        belong_pf_info[vf_row.belong_pf_id]['vf_ids'] = [vf_row.id]
                else:
                    # LOG.info("belong_pf_info[vf_row.belong_pf_id] ====> %s" % belong_pf_info[vf_row.belong_pf_id])
                    belong_pf_info[vf_row.belong_pf_id]['count'] += 1
            # LOG.info("belong_pf_info ====> %s" % belong_pf_info)

            # get pf that has available vf
            pf_rows = session.query(models.Accelerator).\
                filter_by(compute_node_id=result.id).\
                filter_by(function_type='type-PF').\
                filter(models.Accelerator.id.in_(belong_pf_info.keys())).\
                all()

            # 'with session.begin()'.  session should rollback after exception ?
            if not pf_rows:
                raise exception.GetAvailableSubTypePFFailed(compute_node_id=result.id,
                                                            acc_sub_type=acc_type)

            acc_board_vfs = []
            acc_board_pf_info = {}
            common_pf_info = {}
            max_common_vf_num = 0
            max_common_vf_num_pf_id = 0
            max_common_vf_num_vf_id = 0
            board_pf_primary_id = 0
            board_vf_id = 0
            for pf_row in pf_rows:
                # LOG.info("pf_row ====> %s" % pf_row)
                pf_id = pf_row.get('id')
                if pf_row.get('device_type') == 'ACC_BOARD':
                    if pf_row.get('acc_capability') is None:
                        acc_board_vfs.append(belong_pf_info.get(pf_id).get('vf_ids'))
                    else:
                        # acc_board primary pf
                        acc_board_pf_info['id'] = pf_id
                        acc_board_pf_info['acc_capability'] = \
                            jsonutils.loads(pf_row.get('acc_capability'))
                        acc_board_pf_info['vf_ids'] = \
                            belong_pf_info.get(pf_id).get('vf_ids')
                else:
                    common_pf_info[pf_id] = {}
                    common_pf_info[pf_id]['count'] = belong_pf_info.get(pf_id).get('count')
                    common_pf_info[pf_id]['vf_ids'] = belong_pf_info.get(pf_id).get('vf_ids')
                    if max_common_vf_num < common_pf_info.get(pf_id).get('count'):
                        max_common_vf_num_pf_id = pf_id
                        max_common_vf_num_vf_id = common_pf_info.get(pf_id).get('vf_ids')[0]
                        max_common_vf_num = common_pf_info.get(pf_id).get('count')

            # LOG.info("acc_board_pf_info ====> %s" % acc_board_pf_info)
            if acc_board_pf_info:
                acc_board_pf_info['vf_ids'].append(acc_board_vfs)
                board_pf_primary_id = acc_board_pf_info.get('id')
                board_vf_id = acc_board_pf_info.get('vf_ids')[0]
                acc_board_pf_caps = acc_board_pf_info.get('acc_capability')

            selected_pf_id = 0
            selected_vf_id = 0
            # process request that include capability
            LOG.info("capabilitys ====> %s" % capabilitys)
            if capabilitys is not None:
                # get acc_board pf first
                # acc_board device has one primary pf only and ACC_BOARD vf share the capability of all pf
                # db data form: [{"name":"aes", "Mpps":1024,"Mbps":10},{"name":"3des", "Mpps":10240,"Mbps":10}]
                """" request data form:
                    capabilitys":
                    [
                        {"name":"aes","num":800, "unit":"Mbps"},
                        {"name":"aes","num":10240, "unit":"Mpps"},
                        {"name":"3des","num":800, "unit":"Mbps"},
                        {"name":"3des","num":10240, "unit":"Mpps"}
                    ]
                """
                selected_count = 0
                capability_count = 0

                # LOG.info("acc_board_pf_info ====> %s , acc_board_pf_caps:::%s" % (acc_board_pf_info, None))
                if acc_board_pf_info and acc_board_pf_caps:
                    update_capability = acc_board_pf_caps
                    for capability in capabilitys:
                        capability_count += 1
                        for acc_board_pf_cap in acc_board_pf_caps:
                            if acc_board_pf_cap.get('name') == capability.get('name') \
                                    and acc_board_pf_cap.get(capability.get('unit')) >= capability.get('num'):
                                    update_capability[capability.get('unit')] -= capability.get('num')
                                    selected_count += 1
                                    break
                            else:
                                continue
                # LOG.info("selected_count ====> %s , capability_count:::%s" % (selected_count, capability_count))
                # find pf that have most free capability
                if selected_count > 0 and not capability_count > 0 \
                    and selected_count == capability_count:
                    # update pf
                    update_acc_board_pf = session.query(models.Accelerator).\
                        filter_by(compute_node_id = result.compute_node_id).\
                        filter_by(id = board_pf_primary_id).\
                        filter_by(function_type = 'type-PF').\
                        update({'acc_capability': update_capability,
                             'updated_at': timeutils.utcnow()})
                    if not update_acc_board_pf:
                        exception.UpdatePFCapabilityFailed(compute_node_id=result.compute_node_id,
                                                           id=board_pf_primary_id)

                    update_acc_board_pf_list.append(update_acc_board_pf)

                    # update vf
                    selected_pf_id = board_pf_primary_id
                    selected_vf_id = board_vf_id
                else:
                    # get common pf second
                    selected_pf_id = max_common_vf_num_pf_id
                    selected_vf_id = max_common_vf_num_vf_id
            else:
                # LOG.info("max_common_vf_num_pf_id ====> %s , max_common_vf_num_vf_id:::%s" % (max_common_vf_num_pf_id, max_common_vf_num_vf_id))
                # get common pf second
                if not max_common_vf_num_pf_id == 0 and not max_common_vf_num_vf_id == 0:
                    # update common vf
                    selected_pf_id = max_common_vf_num_pf_id
                    selected_vf_id = max_common_vf_num_vf_id
                else:
                    # LOG.info("board_pf_primary_id ====> %s , board_vf_id:::%s" % (board_pf_primary_id, board_vf_id))
                    # update acc_board vf
                    selected_pf_id = board_pf_primary_id
                    selected_vf_id = board_vf_id

            LOG.info("selected_pf_id : %s, selected_vf_id : %s" % (selected_pf_id, selected_vf_id))
            #update
            if not selected_pf_id == 0 and not selected_vf_id == 0:
                # update common vf
                update_vf = session.query(models.Accelerator).\
                    filter_by(compute_node_id=result.id).\
                    filter_by(id = selected_vf_id).\
                    filter_by(belong_pf_id = selected_pf_id).\
                    filter_by(function_type = 'type-VF').\
                    filter_by(status = 'available')
                update_vf.update({'status': 'claimed',
                                  'instance_uuid': instance_uuid,
                                  'updated_at': timeutils.utcnow()})
                # session.add(update_vf)

                if not update_vf:
                    exception.UpdateVFClaimFailed(compute_node_id=result.id,id=selected_vf_id)
                update_vf = session.query(models.Accelerator).filter_by(id = selected_vf_id).first()
                LOG.info("update vf ====> %s" % selected_vf_id)
                update_vf_list.append(selected_vf_id)
            else:
                exception.UpdateVFNoneClaimFailed(compute_node_id=result.compute_node_id)

        if update_vf_list is not None:
            update_info['update_acc_board_pf_list'] = update_acc_board_pf_list
            update_info['update_vf_list'] = update_vf_list
            return update_info
        else:
            return None

def instance_allocate(context, acc_list):
    session = get_session()
    allocate_list = []
    with session.begin():
        for acc in acc_list:
            allocate = session.query(models.Accelerator).filter_by(id=acc.id).filter_by(function_type='type-VF').filter_by(
                status='claimed').update({'status': 'allocated', 'updated_at': timeutils.utcnow()})
            LOG.info("allocate ============> %s, update status::::%s" % (allocate, allocate > 0))
            if allocate > 0:
                acc["status"] = "allocated"
                allocate_list.append(acc)
    return acc_list

def accelerator_filter(cls, context, host_name, req_list):
    """ dev_list form: [{"type":"ipsec", "sub_type":"aes", "unit":"pps","capability":1000},]  """
    session = get_session()
    with session.begin():
        result = model_query(context, models.ComputeNode, session=session).\
            filter_by(hypervisor_hostname=host_name).\
            first()
        if not result:
            raise exception.ComputeNodeNotFoundByNode(host=host_name, node_name=host_name)

        acc_stats = dict_util.unpack_unicode(result.get('accelerator_stats'))
        if acc_stats == None:
            return False

        for k in acc_stats.keys():
            acc_stats[k] = jsonutils.loads(acc_stats[k])
        for req in req_list:
            req_type = req.get('type')
            req_sub_type = req.get('sub_type')
            req_capability = req.get('capability')
            req_unit = req.get('unit')
            match_found = False
            for dev_k in acc_stats.keys():
                dev = acc_stats[dev_k] 
                if not isinstance(dev, dict) or dev.get('free_vf_num', 0) == 0 or not dev.has_key(req_type):
                    continue
                acc = dev.get(req_type)
                if not isinstance(acc, dict) or acc.get('free_vf_num', 0) == 0 or not acc.has_key(req_sub_type):
                    continue
                sub_acc = acc.get(req_sub_type)
                if sub_acc.get(req_unit, 0) <= req_capability:
                    continue
                sub_acc[req_unit] -= req_capability
                acc['free_vf_num'] -= 1
                dev['free_vf_num'] -= 1
                match_found = True
                break
            if not match_found:
                return False
    return True

def release_by_uuid_address(context, uuid, address):
    session = get_session()
    release_list = []
    with session.begin():
        query_release = session.query(models.Accelerator).filter_by(function_type='type-VF')
        if uuid is not None:
            query_release = query_release.filter_by(instance_uuid=uuid)
        if address and 'all' != address:
            query_release = query_release.filter_by(address=address)

        query_list = query_release.all()
        for query in query_list:
            LOG.info("Get release accelerator ====> %s" % query['id'])
            query["status"] = "available"
            release_list.append(query)

        allocate = query_release.update({'status': 'available', 'instance_uuid': '', 'updated_at': timeutils.utcnow()})
        LOG.info("available ============> %s, update status::::%s" % (allocate, allocate > 0))
    return release_list