# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# Copyright 2011 Justin Santa Barbara
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Handles all processes relating to accelerator(rock device including physical
and visual function).

The :py:class:`AgentManager` class is a :py:class:`rock.manager.Manager` that
handles RPC calls relating to creating accelerator.  It is responsible for
configuring a visual function, launching it via the underlying virtualization driver,
responding to calls to check its state, attaching vm instances, and
terminating it.

"""

import os
import socket
import sys

from oslo.config import cfg
from oslo import messaging

from rock import conductor
from rock import exception
from rock import manager

from rock.agent import rpcapi as agent_rpcapi
from rock.agent import acc_tracker
import rock.context
from rock.device import driver
from rock.i18n import _
from rock import objects
from rock.objects import base as rock_base
from rock.openstack.common import log as logging
from rock.openstack.common import periodic_task

from rock.api import rpcapi as rock_rpcapi

rock_opts = [
    cfg.StrOpt('node_name',
               default=socket.gethostname(),
               help='Node name used for rock agent, The name must be really available '
                    'compute-node name.Accelerator in localhost will bind to the compute-node')
]
CONF = cfg.CONF
CONF.register_opts(rock_opts)

LOG = logging.getLogger(__name__)


class AgentManager(manager.Manager):
    """Manages the rock accelerator from creation to destruction."""

    target = messaging.Target(version='1.0')

    def __init__(self, compute_driver=None, *args, **kwargs):
        self.agent_rpcapi = agent_rpcapi.AgentAPI()
        self.conductor_api = conductor.API()
        self._accelerator_tracker_dict = {}
        self.init_agent_node()
        self.test_index = 0
        super(AgentManager, self).__init__(service_name="agent",
                                             *args, **kwargs)

    @periodic_task.periodic_task
    def update_rock_capability(self, context):
        """See driver.get_available_resource()

        Periodic process that keeps that the compute host's understanding of
        resource availability and usage in sync with the underlying hypervisor.

        :param context: security context
        """
        new_accelerator_tracker_dict = {}
        node_list = objects.ComputeNodeList.get_by_hypervisor(context, self.node_name)
        for node in node_list:
            node_name = node['hypervisor_hostname']
            rt = self._get_accelerator_tracker(context, node_name)
            LOG.info('period ==========> %s' % node_name)
            rt.update_rock_capability(context)
            new_accelerator_tracker_dict[node_name] = rt

        self._accelerator_tracker_dict = new_accelerator_tracker_dict

    def _get_accelerator_tracker(self, context= None, node_name = None):
        if not node_name:
            node_name = CONF.node_name
        rt = self._accelerator_tracker_dict.get(node_name)
        if not rt:
            self.driver = driver.load_rock_driver()
            rt = acc_tracker.AcceleratorTracker(context, self.host, self.driver, self.agent_node, node_name)
            self._accelerator_tracker_dict[node_name] = rt
        return rt

    def init_host(self):
        """Initialization for a standalone compute service."""
        self.driver = driver.load_rock_driver()
        for rock_driver in self.driver:
            rock_driver.init_host(host=self.host)

    def pre_start_hook(self):
        """After the service is initialized, but before we fully bring
        the service up by listening on RPC queues, make sure to update
        our available accelerators (and indirectly our available nodes).
        """
        self.update_rock_capability(rock.context.get_admin_context())


    def init_agent_node(self):
        self.host = CONF.host
        if self.host:
            self.node_name = CONF.host
        else:
            default_path = '/proc/sys/kernel/hostname'
            node_name = self._get_file_content(default_path)
            if not node_name:
                node_name = socket.gethostname()
            self.node_name = node_name
        context = rock.context.get_admin_context()
        compute_node_list = list(rock.objects.ComputeNodeList.get_by_hypervisor(context, self.node_name))
        if compute_node_list:
            self.agent_node = compute_node_list[0]
            # LOG.info("agent_node :: id == %s , name == %s " % (self.agent_node['id'], self.agent_node['hypervisor_hostname']))
        else:
            return self.node_name
        return self.agent_node

    def _get_file_content(self, file_path):
        content = ''
        if os.path.isfile(file_path):
            with open(self.file_path) as temp_file:
                content += temp_file.read()
        return content

    def allocate_acc(self, context, req_accelerator):
        if not req_accelerator:
            LOG.warn("Allocate accelerator[%s] is None, EMPTY REQUEST." % req_accelerator)
            raise exception.AcceleratorRequestIsNone()
        acc_tracker = self._get_accelerator_tracker(context, self.node_name)
        if isinstance(req_accelerator, dict) and req_accelerator.has_key("update_vf_list"):
            acc_list = req_accelerator["update_vf_list"]
        elif isinstance(req_accelerator, list):
            acc_list = req_accelerator
        else:
            acc_list = [req_accelerator]
        # LOG.info("====> %s " % acc_list)
        return acc_tracker.allocate_accelerator(acc_list)

    def release_acc(self, context, req_accelerator):
        if not req_accelerator:
            LOG.warn("Release accelerator[%s] is None, EMPTY REQUEST." % req_accelerator)
            return None
            # raise exception.AcceleratorRequestIsNone()
        acc_tracker = self._get_accelerator_tracker(context, self.node_name)
        if isinstance(req_accelerator, dict) and req_accelerator.has_key("update_vf_list"):
            acc_list = req_accelerator["update_vf_list"]
        elif isinstance(req_accelerator, list) or isinstance(req_accelerator, rock_base.ObjectListBase):
            acc_list = req_accelerator
        elif isinstance(req_accelerator, str):
            acc_list = objects.AcceleratorList.get_by_instance_uuid(context, req_accelerator)
        else:
            acc_list = [req_accelerator]
        # LOG.info("====> %s " % acc_list)
        return acc_tracker.delocate_accelerator(acc_list)

    @periodic_task.periodic_task(run_immediately=True)
    def test_rpc_api(self, context):
        instance = {}
        instance["uuid"] = "8e45a2ea-5364-4b0d-a252-bf8becaa606e"
        instance["node"] = "compute1"
        instance["host"] = "compute1"
        # LOG.info("=============> invoke rock rpc api-----------instance::%s" % instance)
        acc_list = list(objects.AcceleratorList.get_by_compute_node(context, '2'))
        acc = acc_list[0]
        LOG.info("==<== accelerator::%s" % acc["acc_capability"])
        # update_dict = {
        #     "acc_capability": {"gb": {"gb": {"pps": 123, "num": 1}}, "num": 1 , "sa": {"sa": {"pps": 321, "num": 1}},
        #     'ipsec': {"3des": {"pps": 2, "num": 1}, "aes": {"pps": 2, "num": 1}, "dh": {"pps": 2, "num": 1}}}}
        # acc.update_device(update_dict)
        # update_dict = {'num': 1, 'ipsec': {'aes': {'pps': 1, 'num': 1}, 'num': 1}}
        # from rock.openstack.common import jsonutils
        # temp_capability = acc["acc_capability"]
        # for key, element in update_dict.items():
        #     if isinstance(element, str):
        #         temp_capability[key] = element
        #     else:
        #         temp_capability[key] = jsonutils.dumps(element)
        # acc["acc_capability"] = temp_capability
        # v = {"gb":{"gb":{"pps":123, "num":1}}, "sa": {"sa": {"pps": 321, "num":1}}}
        # temp_capability = {}
        # for key, element in v.items():
        #     temp_capability[key] = element
        #     if isinstance(element, str):
        #         temp_capability[key] = element
        #     else:
        #         temp_capability[key] = jsonutils.dumps(element)
        # acc["acc_capability"] = temp_capability
        LOG.info("==%s=======================%s=========> accelerator::%s" % (self.test_index, self.test_index, acc["acc_capability"]))
        if self.test_index <=3:
            self.test_index = self.test_index + 1
            rock_rpcapi.RockAPI().allocate_accelerator(context, instance, [acc])
        else:
            self.test_index = 1
            rock_rpcapi.RockAPI().detach_accelerator(context, instance, 'all')
        # rock_rpcapi.RockAPI().detach_accelerator(context, instance, '00:05:00.5')
        # from rock.conductor import rpcapi as conductor_rpc
        # my_list = conductor_rpc.ConductorAPI().service_get_all_by(context, topic='server', host='controller')
        # LOG.info("++++++++++ ======== %s" % my_list)
        # LOG.info(">>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>> %s" % CONF._all_opt_infos())
        # transport = messaging.get_transport(CONF)
        # target = messaging.Target(topic='server')
        # client = messaging.RPCClient(target)
        # cctxt = client.prepare(server='controller', version='1.0')
        # LOG.info('=======host_name======= %s' % target)
        # ctxt_dict = {'project_name': None, 'user_id': None, 'roles': [], 'auth_token': None, 'remote_address': None, 'quota_class': None, 'is_admin': True, 'tenant': None, 'service_catalog': [], 'request_id': 'req-ba2acdf3-613f-429a-a3e2-c5072c24d21e', 'instance_lock_checked': False, 'project_id': None, 'user_name': None, 'read_deleted': 'no', 'user': None}
        # result = cctxt.call(ctxt_dict, 'service_get_all_by', topic='server', host='controller')
        LOG.info("=======> OK ")