#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Handles all requests to the rock service."""

import functools

from oslo.config import cfg
from oslo import messaging

from rock import baserpc
from rock.conductor import manager
from rock.conductor import rpcapi
from rock.i18n import _
from rock.openstack.common import log as logging
from rock import utils
from rock.db import base
from rock.agent import rpcapi as agent_rpcapi
from rock import rpc
from rock import exception
from rock.api import vm_states


rock_opts = [
    cfg.StrOpt('topic',
               default='rock',
               help='The topic on which rock nodes listen'),
    cfg.StrOpt('manager',
               default='rock.api.manager.RockManager',
               help='Full class name for the Manager for rock')
]
CONF = cfg.CONF
CONF.register_opts(rock_opts)

LOG = logging.getLogger(__name__)
get_notifier = functools.partial(rpc.get_notifier, service='rock')

def check_policy(context, action, target, scope='rock'):
    _action = '%s:%s' % (scope, action)
    #rock.policy.enforce(context, _action, target)

def policy_decorator(scope):
    """Check corresponding policy prior of wrapped method to execution."""
    def outer(func):
        @functools.wraps(func)
        def wrapped(self, context, target, *args, **kwargs):
            check_policy(context, func.__name__, target, scope)
            return func(self, context, target, *args, **kwargs)
        return wrapped
    return outer

wrap_check_policy = policy_decorator(scope='rock')

def check_instance_lock(function):
    @functools.wraps(function)
    def inner(self, context, instance, *args, **kwargs):
        if instance['locked'] and not context.is_admin:
            raise exception.InstanceIsLocked(instance_uuid=instance['uuid'])
        return function(self, context, instance, *args, **kwargs)
    return inner

def check_instance_state(vm_state=None, task_state=(None,),
                         must_have_launched=True):
    """Decorator to check VM and/or task state before entry to API functions.

    If the instance is in the wrong state, or has not been successfully
    started at least once the wrapper will raise an exception.
    """

    if vm_state is not None and not isinstance(vm_state, set):
        vm_state = set(vm_state)
    if task_state is not None and not isinstance(task_state, set):
        task_state = set(task_state)

    def outer(f):
        @functools.wraps(f)
        def inner(self, context, instance, *args, **kw):
            if vm_state is not None and instance['vm_state'] not in vm_state:
                raise exception.InstanceInvalidState(
                    attr='vm_state',
                    instance_uuid=instance['uuid'],
                    state=instance['vm_state'],
                    method=f.__name__)
            if (task_state is not None and
                    instance['task_state'] not in task_state):
                raise exception.InstanceInvalidState(
                    attr='task_state',
                    instance_uuid=instance['uuid'],
                    state=instance['task_state'],
                    method=f.__name__)
            if must_have_launched and not instance['launched_at']:
                raise exception.InstanceInvalidState(
                    attr=None,
                    not_launched=True,
                    instance_uuid=instance['uuid'],
                    state=instance['vm_state'],
                    method=f.__name__)

            return f(self, context, instance, *args, **kw)
        return inner
    return outer


class API(base.Base):
    """API for interacting with the rock manager."""

    def __init__(self, agent_rpcapi=None, **kwargs):
        self.agent_rpcapi = agent_rpcapi.AgentAPI()
        self.notifier = rpc.get_notifier('rock', CONF.host)

        super(API, self).__init__(**kwargs)

    @wrap_check_policy
    @check_instance_lock
    @check_instance_state(vm_state=[vm_states.ACTIVE, vm_states.PAUSED,
                                    vm_states.STOPPED],
                          task_state=[None])
    def attach_interface(self, context, instance, accelerator_request):
        """Use hotplug to add an network adapter to an instance."""
        return self.agent_rpcapi.allocate_accelerator(context,
            instance=instance, accelerator_request=accelerator_request)

    def detach_accelerator(self, ctxt, instance, pci_address):
        pass
