# Copyright 2014 eBay Inc.
#
# Author: Ron Rickard <rrickard@ebaysf.com>
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
from oslo_config import cfg
from oslo_log import log as logging
import oslo_messaging as messaging

from designate.i18n import _LI
from designate import rpc


LOG = logging.getLogger(__name__)

MNGR_API = None


class PoolManagerAPI(object):
    """
    Client side of the Pool Manager RPC API.

    API version history:

        API version history:

        1.0 - Initial version
        2.0 - Rename domains to zones
    """
    RPC_API_VERSION = '2.0'

    def __init__(self, topic=None):
        self.topic = topic if topic else cfg.CONF.pool_manager_topic

        target = messaging.Target(topic=self.topic,
                                  version=self.RPC_API_VERSION)
        self.client = rpc.get_client(target, version_cap='2.0')

    @classmethod
    def get_instance(cls):
        """
        The rpc.get_client() which is called upon the API object initialization
        will cause a assertion error if the designate.rpc.TRANSPORT isn't setup
        by rpc.init() before.

        This fixes that by creating the rpcapi when demanded.
        """
        global MNGR_API
        if not MNGR_API:
            MNGR_API = cls()
        return MNGR_API

    def create_zone(self, context, zone):
        LOG.info(_LI("create_zone: Calling pool manager for %(zone)s, "
                     "serial:%(serial)s"),
                 {'zone': zone.name, 'serial': zone.serial})

        # Modifying the topic so it is pool manager instance specific.
        topic = '%s.%s' % (self.topic, zone.pool_id)
        cctxt = self.client.prepare(topic=topic)
        return cctxt.cast(
            context, 'create_zone', zone=zone)

    def delete_zone(self, context, zone):
        LOG.info(_LI("delete_zone: Calling pool manager for %(zone)s, "
                     "serial:%(serial)s"),
                 {'zone': zone.name, 'serial': zone.serial})

        # Modifying the topic so it is pool manager instance specific.
        topic = '%s.%s' % (self.topic, zone.pool_id)
        cctxt = self.client.prepare(topic=topic)
        return cctxt.cast(
            context, 'delete_zone', zone=zone)

    def update_zone(self, context, zone):
        LOG.info(_LI("update_zone: Calling pool manager for %(zone)s, "
                     "serial:%(serial)s"),
                 {'zone': zone.name, 'serial': zone.serial})

        # Modifying the topic so it is pool manager instance specific.
        topic = '%s.%s' % (self.topic, zone.pool_id)
        cctxt = self.client.prepare(topic=topic)
        return cctxt.cast(
            context, 'update_zone', zone=zone)

    def update_status(self, context, zone, nameserver, status,
                      actual_serial):
        LOG.info(_LI("update_status: Calling pool manager for %(zone)s : "
                     "%(action)s : %(status)s : %(serial)s on nameserver "
                     "'%(host)s:%(port)s'"),
                 {'zone': zone.name, 'action': zone.action,
                  'status': status, 'serial': actual_serial,
                  'host': nameserver.host, 'port': nameserver.port})

        # Modifying the topic so it is pool manager instance specific.
        topic = '%s.%s' % (self.topic, zone.pool_id)
        cctxt = self.client.prepare(topic=topic)
        return cctxt.cast(
            context, 'update_status', zone=zone, nameserver=nameserver,
            status=status, actual_serial=actual_serial)
