#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import time

from neutron_lib import constants as n_const

from dragonflow.controller.common import constants as const
from dragonflow.db.models import l2
from dragonflow.tests.common import constants
from dragonflow.tests.common import utils
from dragonflow.tests.fullstack import test_base
from dragonflow.tests.fullstack import test_objects as objects


class TestDbConsistent(test_base.DFTestBase):

    def _check_l2_lookup_rule(self, flows, mac):
        goto_egress = 'goto_table:' + str(const.EGRESS_TABLE)
        for flow in flows:
            if (flow['table'] == str(const.L2_LOOKUP_TABLE)
                    and goto_egress in flow['actions']):
                if 'dl_dst=' + mac in flow['match']:
                    return True
        return False

    def _check_no_lswitch_dhcp_rule(self, flows, network_key):
        if utils.check_dhcp_network_rule(flows, network_key):
            return False
        return True

    def test_db_consistent(self):
        self.db_sync_time = self.conf.db_sync_time
        network = objects.NetworkTestObj(self.neutron, self.nb_api)
        network_id = network.create()
        self.addCleanup(network.close)
        topic = network.get_topic()
        subnet = objects.SubnetTestObj(self.neutron, self.nb_api, network_id)
        subnet_body = {'network_id': network_id,
                       'cidr': '10.50.0.0/24',
                       'gateway_ip': '10.50.0.1',
                       'ip_version': 4,
                       'name': 'private',
                       'enable_dhcp': True}
        subnet.create(subnet=subnet_body)
        self.addCleanup(subnet.close)
        time.sleep(constants.DEFAULT_RESOURCE_READY_TIMEOUT)
        self.assertTrue(network.exists())
        self.assertTrue(subnet.exists())

        vm = objects.VMTestObj(self, self.neutron)
        vm.create(network=network)
        self.addCleanup(vm.close)
        self.assertIsNotNone(vm.server.addresses['mynetwork'])
        mac = vm.server.addresses['mynetwork'][0]['OS-EXT-IPS-MAC:mac_addr']
        self.assertIsNotNone(mac)

        ovs = utils.OvsFlowsParser()
        utils.wait_until_true(
            lambda: self._check_l2_lookup_rule(
                    ovs.dump(self.integration_bridge), mac),
            timeout=10, sleep=1,
            exception=Exception('no rule for vm in l2 lookup table')
        )
        net_id = '11111111-1111-1111-1111-111111111111'
        df_network = l2.LogicalSwitch(
            id=net_id,
            topic=topic,
            name='df_nw1',
            network_type='vxlan',
            segmentation_id=4000,
            is_external=False,
            mtu=1500,
            unique_key=1,
            version=1)
        df_network_json = df_network.to_json()
        self.nb_api.driver.create_key(l2.LogicalSwitch.table_name,
                                      net_id, df_network_json, topic)
        self.addCleanup(self.nb_api.driver.delete_key, 'lswitch',
                        net_id, topic)

        subnet_id = '22222222-2222-2222-2222-222222222222'
        df_subnet = l2.Subnet(
            id=subnet_id,
            topic=topic,
            name='df_sn1',
            enable_dhcp=True,
            cidr='10.60.0.0/24',
            dhcp_ip='10.60.0.2',
            gateway_ip='10.60.0.1',
            version=1,
            lswitch=net_id)
        self.nb_api.driver.create_key(l2.Subnet.table_name,
                                      subnet_id, df_subnet.to_json(), topic)
        self.addCleanup(self.nb_api.driver.delete_key, l2.Subnet.table_name,
                        subnet_id, topic)

        port_id = '33333333-2222-2222-2222-222222222222,'
        dhcp_port = l2.LogicalPort(
            topic=topic,
            name='df_dhcp1',
            macs=['aa:bb:cc:dd:ee:ff'],
            id=port_id,
            ips=['10.60.0.2'],
            subnets=[df_subnet.id],
            device_owner=n_const.DEVICE_OWNER_DHCP,
            lswitch=df_network.id,
            unique_key=1
        ).to_json()

        self.nb_api.driver.create_key(
            'lport', port_id, dhcp_port, topic)

        df_net_unique_key = df_network.unique_key
        time.sleep(self.db_sync_time)
        utils.wait_until_true(
            lambda: utils.check_dhcp_network_rule(
                    ovs.dump(self.integration_bridge), df_net_unique_key),
            timeout=self.db_sync_time + constants.DEFAULT_CMD_TIMEOUT, sleep=1,
            exception=Exception('no goto dhcp rule for lswitch')
        )

        self.nb_api.driver.delete_key('lport', port_id, topic)
        time.sleep(self.db_sync_time)
        utils.wait_until_true(
            lambda: self._check_no_lswitch_dhcp_rule(
                    ovs.dump(self.integration_bridge), df_net_unique_key),
            timeout=self.db_sync_time + constants.DEFAULT_CMD_TIMEOUT, sleep=1,
            exception=Exception('could not delete goto dhcp rule for lswitch')
        )
