#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from heat.common import exception
from heat.common.i18n import _
from heat.engine import attributes
from heat.engine import constraints
from heat.engine import properties
from heat.engine import resource
from heat.engine import support


class ManilaShareNetwork(resource.Resource):
    """
    Stores network information that will be used by share servers,
    where shares are hosted.
    """

    support_status = support.SupportStatus(version='5.0.0')

    PROPERTIES = (
        NAME, NEUTRON_NETWORK, NEUTRON_SUBNET, NOVA_NETWORK,
        DESCRIPTION, SECURITY_SERVICES,
    ) = (
        'name', 'neutron_network', 'neutron_subnet', 'nova_network',
        'description', 'security_services',
    )

    ATTRIBUTES = (
        SEGMENTATION_ID, CIDR, IP_VERSION, NETWORK_TYPE,
    ) = (
        'segmentation_id', 'cidr', 'ip_version', 'network_type',
    )

    properties_schema = {
        NAME: properties.Schema(
            properties.Schema.STRING,
            _('Name of the share network.'),
            update_allowed=True
        ),
        NEUTRON_NETWORK: properties.Schema(
            properties.Schema.STRING,
            _('Neutron network id.'),
            update_allowed=True,
            constraints=[constraints.CustomConstraint('neutron.network')]
        ),
        NEUTRON_SUBNET: properties.Schema(
            properties.Schema.STRING,
            _('Neutron subnet id.'),
            update_allowed=True,
            constraints=[constraints.CustomConstraint('neutron.subnet')]
        ),
        NOVA_NETWORK: properties.Schema(
            properties.Schema.STRING,
            _('Nova network id.'),
            update_allowed=True,
            constraints=[constraints.CustomConstraint('nova.network')]
        ),
        DESCRIPTION: properties.Schema(
            properties.Schema.STRING,
            _('Share network description.'),
            update_allowed=True
        ),
        SECURITY_SERVICES: properties.Schema(
            properties.Schema.LIST,
            _('A list of security services IDs or names.'),
            schema=properties.Schema(
                properties.Schema.STRING
            ),
            update_allowed=True,
            default=[]
        )
    }

    attributes_schema = {
        SEGMENTATION_ID: attributes.Schema(
            _('VLAN ID for VLAN networks or tunnel-id for GRE/VXLAN '
              'networks.'),
            type=attributes.Schema.STRING
        ),
        CIDR: attributes.Schema(
            _('CIDR of subnet.'),
            type=attributes.Schema.STRING
        ),
        IP_VERSION: attributes.Schema(
            _('Version of IP address.'),
            type=attributes.Schema.STRING
        ),
        NETWORK_TYPE: attributes.Schema(
            _('The physical mechanism by which the virtual network is '
              'implemented.'),
            type=attributes.Schema.STRING
        ),
    }

    default_client_name = 'manila'

    entity = 'share_networks'

    def _request_network(self):
        return self.client().share_networks.get(self.resource_id)

    def _resolve_attribute(self, name):
        network = self._request_network()
        return getattr(network, name, None)

    def validate(self):
        super(ManilaShareNetwork, self).validate()
        if (self.properties[self.NEUTRON_NETWORK] and
                self.properties[self.NOVA_NETWORK]):
            raise exception.ResourcePropertyConflict(self.NEUTRON_NETWORK,
                                                     self.NOVA_NETWORK)

        if (self.properties[self.NOVA_NETWORK] and
                self.properties[self.NEUTRON_SUBNET]):
            raise exception.ResourcePropertyConflict(self.NEUTRON_SUBNET,
                                                     self.NOVA_NETWORK)

    def handle_create(self):
        network = self.client().share_networks.create(
            name=self.properties[self.NAME],
            neutron_net_id=self.properties[self.NEUTRON_NETWORK],
            neutron_subnet_id=self.properties[self.NEUTRON_SUBNET],
            nova_net_id=self.properties[self.NOVA_NETWORK],
            description=self.properties[self.DESCRIPTION])
        self.resource_id_set(network.id)

        for service in self.properties.get(self.SECURITY_SERVICES):
            self.client().share_networks.add_security_service(
                self.resource_id,
                self.client_plugin().get_security_service(service).id)

    def handle_update(self, json_snippet, tmpl_diff, prop_diff):
        if self.SECURITY_SERVICES in prop_diff:
            services = prop_diff.pop(self.SECURITY_SERVICES)
            s_curr = set([self.client_plugin().get_security_service(s).id
                          for s in self.properties.get(
                         self.SECURITY_SERVICES)])
            s_new = set([self.client_plugin().get_security_service(s).id
                        for s in services])
            for service in s_curr - s_new:
                self.client().share_networks.remove_security_service(
                    self.resource_id, service)
            for service in s_new - s_curr:
                self.client().share_networks.add_security_service(
                    self.resource_id, service)

        if prop_diff:
            self.client().share_networks.update(
                self.resource_id,
                name=prop_diff.get(self.NAME),
                neutron_net_id=prop_diff.get(self.NEUTRON_NETWORK),
                neutron_subnet_id=prop_diff.get(self.NEUTRON_SUBNET),
                nova_net_id=prop_diff.get(self.NOVA_NETWORK),
                description=prop_diff.get(self.DESCRIPTION))


def resource_mapping():
    return {'OS::Manila::ShareNetwork': ManilaShareNetwork}
