# vim: tabstop=4 shiftwidth=4 softtabstop=4

# Copyright 2012 Nebula, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from django import http
from django.core.urlresolvers import reverse
from mox import IsA

from horizon import api
from horizon import test

from .workflows import CreateProject, UpdateProject
from .views import QUOTA_FIELDS

INDEX_URL = reverse('horizon:admin:projects:index')


class TenantsViewTests(test.BaseAdminViewTests):
    def test_index(self):
        self.mox.StubOutWithMock(api.keystone, 'tenant_list')
        api.keystone.tenant_list(IsA(http.HttpRequest), admin=True) \
                    .AndReturn(self.tenants.list())
        self.mox.ReplayAll()

        res = self.client.get(INDEX_URL)
        self.assertTemplateUsed(res, 'admin/projects/index.html')
        self.assertItemsEqual(res.context['table'].data, self.tenants.list())


class CreateProjectWorkflowTests(test.BaseAdminViewTests):
    def _get_project_info(self, project):
        project_info = {"tenant_name": project.name,
                        "description": project.description,
                        "enabled": project.enabled}
        return project_info

    def _get_workflow_fields(self, project):
        project_info = {"name": project.name,
                        "description": project.description,
                        "enabled": project.enabled}
        return project_info

    def _get_quota_info(self, quota):
        quota_data = {}
        for field in QUOTA_FIELDS:
            quota_data[field] = int(getattr(quota, field, None))
        return quota_data

    def _get_workflow_data(self, project, quota):
        project_info = self._get_workflow_fields(project)
        quota_data = self._get_quota_info(quota)
        project_info.update(quota_data)
        return project_info

    @test.create_stubs({api: ('tenant_quota_defaults',
                              'get_default_role',),
                       api.keystone: ('user_list',
                                      'role_list',)})
    def test_add_project_get(self):
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        # init
        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        self.mox.ReplayAll()

        url = reverse('horizon:admin:projects:create')
        res = self.client.get(url)

        self.assertTemplateUsed(res, 'admin/projects/create.html')

        workflow = res.context['workflow']
        self.assertEqual(res.context['workflow'].name, CreateProject.name)

        step = workflow.get_step("createprojectinfoaction")
        self.assertEqual(step.action.initial['ram'], quota.ram)
        self.assertEqual(step.action.initial['injected_files'],
                         quota.injected_files)
        self.assertQuerysetEqual(workflow.steps,
                            ['<CreateProjectInfo: createprojectinfoaction>',
                             '<UpdateProjectMembers: update_members>',
                             '<UpdateProjectQuota: update_quotas>'])

    @test.create_stubs({api: ('get_default_role',
                              'tenant_quota_defaults',
                              'add_tenant_user_role',),
                        api.keystone: ('tenant_create',
                                       'user_list',
                                       'role_list'),
                        api.nova: ('tenant_quota_update',)})
    def test_add_project_post(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # init
        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        project_details = self._get_project_info(project)
        quota_data = self._get_quota_info(quota)

        api.keystone.tenant_create(IsA(http.HttpRequest), **project_details) \
                    .AndReturn(project)

        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for role in roles:
            if "role_" + role.id in workflow_data:
                ulist = workflow_data["role_" + role.id]
                for user in ulist:
                    api.add_tenant_user_role(IsA(http.HttpRequest),
                                             tenant_id=self.tenant.id,
                                             user_id=user,
                                             role_id=role.id)

        api.nova.tenant_quota_update(IsA(http.HttpRequest),
                                     project.id,
                                     **quota_data)

        self.mox.ReplayAll()

        workflow_data.update(self._get_workflow_data(project, quota))

        url = reverse('horizon:admin:projects:create')
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('tenant_quota_defaults',
                              'get_default_role',),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_add_project_quota_defaults_error(self):
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # init
        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
           .AndRaise(self.exceptions.nova)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        self.mox.ReplayAll()

        url = reverse('horizon:admin:projects:create')
        res = self.client.get(url)

        self.assertTemplateUsed(res, 'admin/projects/create.html')
        self.assertContains(res, "Unable to retrieve default quota values")

    @test.create_stubs({api: ('get_default_role',
                              'tenant_quota_defaults',),
                        api.keystone: ('tenant_create',
                                       'user_list',
                                       'role_list',)})
    def test_add_project_tenant_create_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # init
        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        project_details = self._get_project_info(project)

        api.keystone.tenant_create(IsA(http.HttpRequest), **project_details) \
            .AndRaise(self.exceptions.keystone)

        self.mox.ReplayAll()

        workflow_data = self._get_workflow_data(project, quota)

        url = reverse('horizon:admin:projects:create')
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('get_default_role',
                              'tenant_quota_defaults',
                              'add_tenant_user_role',),
                        api.keystone: ('tenant_create',
                                       'user_list',
                                       'role_list'),
                        api.nova: ('tenant_quota_update',)})
    def test_add_project_quota_update_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # init
        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        project_details = self._get_project_info(project)
        quota_data = self._get_quota_info(quota)

        api.keystone.tenant_create(IsA(http.HttpRequest), **project_details) \
                    .AndReturn(project)

        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for role in roles:
            if "role_" + role.id in workflow_data:
                ulist = workflow_data["role_" + role.id]
                for user in ulist:
                    api.add_tenant_user_role(IsA(http.HttpRequest),
                                             tenant_id=self.tenant.id,
                                             user_id=user,
                                             role_id=role.id)

        api.nova.tenant_quota_update(IsA(http.HttpRequest),
                                     project.id,
                                     **quota_data) \
           .AndRaise(self.exceptions.nova)

        self.mox.ReplayAll()

        workflow_data.update(self._get_workflow_data(project, quota))

        url = reverse('horizon:admin:projects:create')
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('get_default_role',
                              'tenant_quota_defaults',
                              'add_tenant_user_role',),
                        api.keystone: ('tenant_create',
                                       'user_list',
                                       'role_list',),
                        api.nova: ('tenant_quota_update',)})
    def test_add_project_user_update_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # init
        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        project_details = self._get_project_info(project)
        quota_data = self._get_quota_info(quota)

        api.keystone.tenant_create(IsA(http.HttpRequest), **project_details) \
                    .AndReturn(project)

        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for role in roles:
            if "role_" + role.id in workflow_data:
                ulist = workflow_data["role_" + role.id]
                for user in ulist:
                    api.add_tenant_user_role(IsA(http.HttpRequest),
                                             tenant_id=self.tenant.id,
                                             user_id=user,
                                             role_id=role.id) \
                       .AndRaise(self.exceptions.keystone)
                    break
            break

        api.nova.tenant_quota_update(IsA(http.HttpRequest),
                                     project.id,
                                     **quota_data)

        self.mox.ReplayAll()

        workflow_data.update(self._get_workflow_data(project, quota))

        url = reverse('horizon:admin:projects:create')
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('get_default_role',
                              'tenant_quota_defaults',),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_add_project_missing_field_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # init
        api.tenant_quota_defaults(IsA(http.HttpRequest), self.tenant.id) \
           .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        self.mox.ReplayAll()

        workflow_data = self._get_workflow_data(project, quota)
        workflow_data["name"] = ""

        url = reverse('horizon:admin:projects:create')
        res = self.client.post(url, workflow_data)

        self.assertContains(res, "field is required")


class UpdateProjectWorkflowTests(test.BaseAdminViewTests):
    def _get_quota_info(self, quota):
        quota_data = {}
        for field in QUOTA_FIELDS:
            quota_data[field] = int(getattr(quota, field, None))
        return quota_data

    @test.create_stubs({api: ('get_default_role',
                              'roles_for_user',
                              'tenant_get',
                              'tenant_quota_get',),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_update_project_get(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        api.tenant_get(IsA(http.HttpRequest), self.tenant.id, admin=True) \
            .AndReturn(project)
        api.tenant_quota_get(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                               user.id,
                               self.tenant.id).AndReturn(roles)

        self.mox.ReplayAll()

        url = reverse('horizon:admin:projects:update',
                      args=[self.tenant.id])
        res = self.client.get(url)

        self.assertTemplateUsed(res, 'admin/projects/update.html')

        workflow = res.context['workflow']
        self.assertEqual(res.context['workflow'].name, UpdateProject.name)

        step = workflow.get_step("update_info")
        self.assertEqual(step.action.initial['ram'], quota.ram)
        self.assertEqual(step.action.initial['injected_files'],
                         quota.injected_files)
        self.assertEqual(step.action.initial['name'], project.name)
        self.assertEqual(step.action.initial['description'],
                         project.description)
        self.assertQuerysetEqual(workflow.steps,
                            ['<UpdateProjectInfo: update_info>',
                             '<UpdateProjectMembers: update_members>',
                             '<UpdateProjectQuota: update_quotas>'])

    @test.create_stubs({api: ('tenant_get',
                              'tenant_quota_get',
                              'tenant_update',
                              'tenant_quota_update',
                              'get_default_role',
                              'roles_for_user',
                              'remove_tenant_user_role',
                              'add_tenant_user_role'),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_update_project_post(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()
        current_roles = self.roles.list()

        # get/init
        api.tenant_get(IsA(http.HttpRequest), self.tenant.id, admin=True) \
            .AndReturn(project)
        api.tenant_quota_get(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                               user.id,
                               self.tenant.id).AndReturn(roles)
            role_ids = [role.id for role in roles]
            if role_ids:
                workflow_data.setdefault("role_" + role_ids[0], []) \
                             .append(user.id)

        # update some fields
        project._info["name"] = "updated name"
        project._info["description"] = "updated description"
        quota.metadata_items = 444
        quota.volumes = 444

        updated_project = {"tenant_name": project._info["name"],
                           "tenant_id": project.id,
                           "description": project._info["description"],
                           "enabled": project.enabled}
        updated_quota = self._get_quota_info(quota)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        api.tenant_update(IsA(http.HttpRequest), **updated_project) \
            .AndReturn(project)

        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)
        api.keystone.user_list(IsA(http.HttpRequest),
                               tenant_id=self.tenant.id).AndReturn(users)

        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                                        user.id,
                                        self.tenant.id) \
                                .AndReturn(current_roles)
            for role in roles:
                if "role_" + role.id in workflow_data:
                    ulist = workflow_data["role_" + role.id]
                    if role not in current_roles:
                        api.add_tenant_user_role(IsA(http.HttpRequest),
                                             tenant_id=self.tenant.id,
                                             user_id=user,
                                             role_id=role.id)
                    else:
                        current_roles.pop(current_roles.index(role))
            for to_delete in current_roles:
                api.remove_tenant_user_role(IsA(http.HttpRequest),
                                            tenant_id=self.tenant.id,
                                            user_id=user.id,
                                            role_id=to_delete.id)
        for role in roles:
            if "role_" + role.id in workflow_data:
                ulist = workflow_data["role_" + role.id]
                for user in ulist:
                    if not filter(lambda x: user == x.id, users):
                        api.add_tenant_user_role(IsA(http.HttpRequest),
                                                 tenant_id=self.tenant.id,
                                                 user_id=user,
                                                 role_id=role.id)

        api.tenant_quota_update(IsA(http.HttpRequest),
                                project.id,
                                **updated_quota)

        self.mox.ReplayAll()

        # submit form data
        project_data = {"name": project._info["name"],
                         "id": project.id,
                         "description": project._info["description"],
                         "enabled": project.enabled}
        workflow_data.update(project_data)
        workflow_data.update(updated_quota)
        url = reverse('horizon:admin:projects:update',
                      args=[self.tenant.id])
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('tenant_get',)})
    def test_update_project_get_error(self):

        api.tenant_get(IsA(http.HttpRequest), self.tenant.id, admin=True) \
            .AndRaise(self.exceptions.nova)

        self.mox.ReplayAll()

        url = reverse('horizon:admin:projects:update',
                      args=[self.tenant.id])
        res = self.client.get(url)

        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('tenant_get',
                              'tenant_quota_get',
                              'tenant_update',
                              'tenant_quota_update',
                              'get_default_role',
                              'roles_for_user',
                              'remove_tenant_user',
                              'add_tenant_user_role'),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_update_project_tenant_update_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()

        # get/init
        api.tenant_get(IsA(http.HttpRequest), self.tenant.id, admin=True) \
            .AndReturn(project)
        api.tenant_quota_get(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                               user.id,
                               self.tenant.id).AndReturn(roles)
            role_ids = [role.id for role in roles]
            if role_ids:
                workflow_data.setdefault("role_" + role_ids[0], []) \
                             .append(user.id)

        # update some fields
        project._info["name"] = "updated name"
        project._info["description"] = "updated description"
        quota.metadata_items = 444
        quota.volumes = 444

        updated_project = {"tenant_name": project._info["name"],
                           "tenant_id": project.id,
                           "description": project._info["description"],
                           "enabled": project.enabled}
        updated_quota = self._get_quota_info(quota)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        api.tenant_update(IsA(http.HttpRequest), **updated_project) \
            .AndRaise(self.exceptions.keystone)

        self.mox.ReplayAll()

        # submit form data
        project_data = {"name": project._info["name"],
                         "id": project.id,
                         "description": project._info["description"],
                         "enabled": project.enabled}
        workflow_data.update(project_data)
        workflow_data.update(updated_quota)
        url = reverse('horizon:admin:projects:update',
                      args=[self.tenant.id])
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('tenant_get',
                              'tenant_quota_get',
                              'tenant_update',
                              'tenant_quota_update',
                              'get_default_role',
                              'roles_for_user',
                              'remove_tenant_user_role',
                              'add_tenant_user_role'),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_update_project_quota_update_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()
        current_roles = self.roles.list()

        # get/init
        api.tenant_get(IsA(http.HttpRequest), self.tenant.id, admin=True) \
            .AndReturn(project)
        api.tenant_quota_get(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                               user.id,
                               self.tenant.id).AndReturn(roles)
            role_ids = [role.id for role in roles]
            if role_ids:
                workflow_data.setdefault("role_" + role_ids[0], []) \
                             .append(user.id)

        # update some fields
        project._info["name"] = "updated name"
        project._info["description"] = "updated description"
        quota.metadata_items = 444
        quota.volumes = 444

        updated_project = {"tenant_name": project._info["name"],
                           "tenant_id": project.id,
                           "description": project._info["description"],
                           "enabled": project.enabled}
        updated_quota = self._get_quota_info(quota)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        # handle
        api.tenant_update(IsA(http.HttpRequest), **updated_project) \
            .AndReturn(project)

        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)
        api.keystone.user_list(IsA(http.HttpRequest),
                               tenant_id=self.tenant.id).AndReturn(users)

        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                                        user.id,
                                        self.tenant.id) \
                              .AndReturn(current_roles)
            for role in roles:
                if "role_" + role.id in workflow_data:
                    ulist = workflow_data["role_" + role.id]
                    if role not in current_roles:
                        api.add_tenant_user_role(IsA(http.HttpRequest),
                                             tenant_id=self.tenant.id,
                                             user_id=user,
                                             role_id=role.id)
                    else:
                        current_roles.pop(current_roles.index(role))
            for to_delete in current_roles:
                api.remove_tenant_user_role(IsA(http.HttpRequest),
                                            tenant_id=self.tenant.id,
                                            user_id=user.id,
                                            role_id=to_delete.id)
        for role in roles:
            if "role_" + role.id in workflow_data:
                ulist = workflow_data["role_" + role.id]
                for user in ulist:
                    if not filter(lambda x: user == x.id, users):
                        api.add_tenant_user_role(IsA(http.HttpRequest),
                                                 tenant_id=self.tenant.id,
                                                 user_id=user,
                                                 role_id=role.id)

        api.tenant_quota_update(IsA(http.HttpRequest),
                                project.id,
                                **updated_quota).AndRaise(self.exceptions.nova)

        self.mox.ReplayAll()

        # submit form data
        project_data = {"name": project._info["name"],
                         "id": project.id,
                         "description": project._info["description"],
                         "enabled": project.enabled}
        workflow_data.update(project_data)
        workflow_data.update(updated_quota)
        url = reverse('horizon:admin:projects:update',
                      args=[self.tenant.id])
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)

    @test.create_stubs({api: ('tenant_get',
                              'tenant_quota_get',
                              'tenant_update',
                              'get_default_role',
                              'roles_for_user',
                              'remove_tenant_user_role',
                              'add_tenant_user_role'),
                        api.keystone: ('user_list',
                                       'role_list',)})
    def test_update_project_member_update_error(self):
        project = self.tenants.first()
        quota = self.quotas.first()
        default_role = self.roles.first()
        users = self.users.list()
        roles = self.roles.list()
        current_roles = self.roles.list()

        # get/init
        api.tenant_get(IsA(http.HttpRequest), self.tenant.id, admin=True) \
            .AndReturn(project)
        api.tenant_quota_get(IsA(http.HttpRequest), self.tenant.id) \
            .AndReturn(quota)

        api.get_default_role(IsA(http.HttpRequest)).AndReturn(default_role)
        api.keystone.user_list(IsA(http.HttpRequest)).AndReturn(users)
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        workflow_data = {}
        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                               user.id,
                               self.tenant.id).AndReturn(roles)
            role_ids = [role.id for role in roles]
            if role_ids:
                workflow_data.setdefault("role_" + role_ids[0], []) \
                             .append(user.id)

        # update some fields
        project._info["name"] = "updated name"
        project._info["description"] = "updated description"
        quota.metadata_items = 444
        quota.volumes = 444

        updated_project = {"tenant_name": project._info["name"],
                           "tenant_id": project.id,
                           "description": project._info["description"],
                           "enabled": project.enabled}
        updated_quota = self._get_quota_info(quota)

        # contribute
        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)

        # handle
        api.tenant_update(IsA(http.HttpRequest), **updated_project) \
            .AndReturn(project)

        api.keystone.role_list(IsA(http.HttpRequest)).AndReturn(roles)
        api.keystone.user_list(IsA(http.HttpRequest),
                               tenant_id=self.tenant.id).AndReturn(users)

        for user in users:
            api.roles_for_user(IsA(http.HttpRequest),
                                        user.id,
                                        self.tenant.id) \
                              .AndReturn(current_roles)
            for role in roles:
                if "role_" + role.id in workflow_data:
                    if role not in current_roles:
                        api.add_tenant_user_role(IsA(http.HttpRequest),
                                             tenant_id=self.tenant.id,
                                             user_id=user,
                                             role_id=role.id)
                    else:
                        current_roles.pop(current_roles.index(role))
            for to_delete in current_roles:
                api.remove_tenant_user_role(IsA(http.HttpRequest),
                                            tenant_id=self.tenant.id,
                                            user_id=user.id,
                                            role_id=to_delete.id) \
                                            .AndRaise(self.exceptions.nova)
                break
            break

        self.mox.ReplayAll()

        # submit form data
        project_data = {"name": project._info["name"],
                         "id": project.id,
                         "description": project._info["description"],
                         "enabled": project.enabled}
        workflow_data.update(project_data)
        workflow_data.update(updated_quota)
        url = reverse('horizon:admin:projects:update',
                      args=[self.tenant.id])
        res = self.client.post(url, workflow_data)

        self.assertNoFormErrors(res)
        self.assertRedirectsNoFollow(res, INDEX_URL)
