# Copyright 2013 Rackspace Hosting.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import logging

from django.conf import settings
from troveclient.v1 import client

from openstack_dashboard.api import base

from horizon.utils import functions as utils

LOG = logging.getLogger(__name__)


def troveclient(request):
    insecure = getattr(settings, 'OPENSTACK_SSL_NO_VERIFY', False)
    cacert = getattr(settings, 'OPENSTACK_SSL_CACERT', None)
    trove_url = base.url_for(request, 'database')
    LOG.debug('troveclient connection created using token "%s" and url "%s"' %
              (request.user.token.id, trove_url))
    c = client.Client(request.user.username,
                      request.user.token.id,
                      project_id=request.user.project_id,
                      auth_url=trove_url,
                      insecure=insecure,
                      cacert=cacert,
                      http_log_debug=settings.DEBUG)
    c.client.auth_token = request.user.token.id
    c.client.management_url = trove_url
    return c


def instance_list(request, marker=None):
    page_size = utils.get_page_size(request)
    return troveclient(request).instances.list(limit=page_size, marker=marker)


def instance_get(request, instance_id):
    return troveclient(request).instances.get(instance_id)


def instance_delete(request, instance_id):
    return troveclient(request).instances.delete(instance_id)


def instance_create(request, name, volume, flavor, databases=None,
                    users=None, restore_point=None, nics=None,
                    datastore=None, datastore_version=None):
    # TODO(dklyle): adding conditional to support trove without volume
    # support for now until API supports checking for volume support
    if volume > 0:
        volume_params = {'size': volume}
    else:
        volume_params = None
    return troveclient(request).instances.create(
        name,
        flavor,
        volume=volume_params,
        databases=databases,
        users=users,
        restorePoint=restore_point,
        nics=nics,
        datastore=datastore,
        datastore_version=datastore_version)


def instance_resize_volume(request, instance_id, size):
    return troveclient(request).instances.resize_volume(instance_id, size)


def instance_backups(request, instance_id):
    return troveclient(request).instances.backups(instance_id)


def instance_restart(request, instance_id):
    return troveclient(request).instances.restart(instance_id)


def database_list(request, instance_id):
    return troveclient(request).databases.list(instance_id)


def database_delete(request, instance_id, db_name):
    return troveclient(request).databases.delete(instance_id, db_name)


def backup_list(request):
    return troveclient(request).backups.list()


def backup_get(request, backup_id):
    return troveclient(request).backups.get(backup_id)


def backup_delete(request, backup_id):
    return troveclient(request).backups.delete(backup_id)


def backup_create(request, name, instance_id, description=None, parent_id=None):
    return troveclient(request).backups.create(name, instance_id,
                                               description, parent_id)


def flavor_list(request):
    return troveclient(request).flavors.list()


def flavor_get(request, flavor_id):
    return troveclient(request).flavors.get(flavor_id)


def users_list(request, instance_id):
    return troveclient(request).users.list(instance_id)


def user_delete(request, instance_id, user):
    return troveclient(request).users.delete(instance_id, user)


def user_list_access(request, instance_id, user):
    return troveclient(request).users.list_access(instance_id, user)


def datastore_list(request):
    return troveclient(request).datastores.list()


def datastore_version_list(request, datastore):
    return troveclient(request).datastore_versions.list(datastore)
