/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The Ironic Root Application.
 *
 * This module defines dependencies and root routes, but no actual
 * functionality.
 */
angular.module('ironic', ['ui.router', 'ui.bootstrap', 'ironic.util', 'ironic.api'])
  .config(function($urlRouterProvider, $httpProvider, $stateProvider, $$configurationProvider) {
    'use strict';

    // Default UI route
    $urlRouterProvider.otherwise('/ironic');

    // Enable all of our configuration detection methods
    $$configurationProvider.$enableDefault(true);
    $$configurationProvider.$enableConfigLoad(true);
    $$configurationProvider.$enableLocalStorage(true);

    // Ironic's root state, used to resolve global resources before
    // the application fully initializes.
    $stateProvider
      .state('root', {
        abstract: true,
        url: '',
        templateUrl: 'view/ironic/index.html'
      })
      .state('root.ironic', {
        url: '/ironic',
        views: {
          header: {
            templateUrl: 'view/ironic/header.html',
            controller: 'HeaderController as headerCtrl'
          },
          main: {
            templateUrl: 'view/ironic/node_list.html',
            controller: 'NodeListController as nodeListCtrl'
          }
        },
        resolve: {
          configurations: function($$configuration) {
            return $$configuration.query({}).$promise;
          },
          currentConfiguration: function($$selectedConfiguration, $q) {
            var deferred = $q.defer();

            var resource = $$selectedConfiguration.get();
            resource.$promise.then(
              function() {
                deferred.resolve(resource);
              },
              function() {
                deferred.reject('no_config');
              });
            return deferred.promise;
          }
        }
      })
      .state('root.ironic.nodes', {
        abstract: true,
        url: '/nodes'
      })
      .state('root.ironic.nodes.detail', {
        abstract: true,
        url: '/:uuid',
        resolve: {
          nodeUuid: function($stateParams) {
            return $stateParams.uuid;
          }
        },
        views: {
          'main@root': {
            templateUrl: 'view/ironic/detail.html',
            controller: 'NodeDetailController as nodeCtrl'
          }
        }
      })
      .state('root.ironic.nodes.detail.node', {
        url: '/node',
        templateUrl: 'view/ironic/detail_node.html'
      })
      .state('root.ironic.nodes.detail.ports', {
        url: '/ports',
        templateUrl: 'view/ironic/detail_ports.html',
        controller: 'NodeDetailPortsController as portCtrl'
      })
      .state('root.ironic.nodes.detail.driver', {
        url: '/driver',
        templateUrl: 'view/ironic/detail_driver.html',
        controller: 'NodeDetailDriverController as driverCtrl'
      })
      .state('root.config', {
        url: '/config',
        views: {
          main: {
            templateUrl: 'view/ironic/config.html',
            controller: 'ConfigurationController as ctrl'
          }
        }
      });
  })
  .run(function($rootScope, $state) {
    'use strict';

    var listener = $rootScope.$on('$stateChangeError',
      function(evt, toState, toParams, fromState, fromParams, reason) {
        if (reason === 'no_config') {
          $state.go('root.config');
        } else {
          $state.go('root.ironic');
        }
      });
    $rootScope.$on('$destroy', listener);
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller allows the creation of a new configuration.
 */
angular.module('ironic').controller('ConfigurationAddController',
  function($scope, $state, $location, $$configuration, $modalInstance, configuration) {
    'use strict';
    var vm = this;

    vm.configuration = configuration;
    vm.newConfiguration = {
      name: '',
      ironic: {
        apiRoot: ''
      }
    };

    vm.location = {
      host: $location.host(),
      protocol: $location.protocol(),
      port: $location.port()
    };

    vm.save = function() {
      vm.newConfiguration.id = vm.newConfiguration.name;
      $modalInstance.close(vm.newConfiguration);
    };

    vm.close = function() {
      $modalInstance.dismiss();
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller allows the management of all the cloud configuration entries.
 */
angular.module('ironic').controller('ConfigurationController',
  function($state, $location, $$configuration, $$selectedConfiguration, $modal) {
    'use strict';
    var vm = this;

    vm.configurations = [];

    vm.location = {
      host: $location.host(),
      protocol: $location.protocol(),
      port: $location.port()
    };

    /**
     * Reload all configurations in this controller.
     *
     * @returns {void}
     */
    function reloadConfigurations () {
      vm.configurations = $$configuration.query({});
    }

    /**
     * Select a single configuration for the current application runtime.
     *
     * @param {{}} configuration The configuration to select.
     * @returns {void}
     */
    vm.select = function(configuration) {
      $$selectedConfiguration.set(configuration).$promise.then(
        function() {
          $state.go('root.ironic');
        }
      );
    };

    /**
     * Displays the local configuration add modal.
     *
     * @returns {void}
     */
    vm.add = function() {
      //  var deferred = $q.defer();
      $modal.open({
        templateUrl: 'view/ironic/config_add.html',
        controller: 'ConfigurationAddController as ctrl',
        backdrop: 'static',
        resolve: {
          configuration: function() {
            return $$configuration.query({}).$promise;
          }
        }
      }).result.then(
        function(newConfig) {
          $$configuration.create(newConfig).$promise.then(reloadConfigurations);
        });
    };

    /**
     * Remove a configuration.
     * @param {{}} config The configuration to remove.
     * @returns {void}
     */
    vm.remove = function(config) {
      config.$remove().$promise.then(reloadConfigurations);
    };

    reloadConfigurations();
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller handles the enrollment of a new, single, node
 */
angular.module('ironic').controller('EnrollModalController',
  function($scope, $modalInstance, IronicDriver, IronicDriverProperties,
           IronicNode) {
    'use strict';
    var vm = this;

    vm.drivers = null;
    vm.errorMessage = null;
    vm.driverProperties = null;
    vm.node = null;

    vm.loadDrivers = function() {
      vm.errorMessage = null;
      vm.drivers = IronicDriver.query({}, function() {
        // Do nothing on success.
      }, function(error) {
        vm.errorMessage = error.data.error_message;
        vm.drivers = null;
      });
    };

    // Given a driver name, retrieve all the driver's options.
    vm.loadDriverProperties = function(driverName) {
      vm.driverProperties = null;
      vm.errorMessage = null;
      vm.node = {
        name: vm.node ? vm.node.name : '', // Preserve previously entered names.
        driver: driverName,
        driver_info: {}
      };

      var params = {
        driver_name: driverName
      };

      IronicDriverProperties.read(params).$promise.then(
        function(properties) {
          vm.driverProperties = properties;
        },
        function(error) {
          vm.errorMessage = error.data.error_message;
        });
    };

    vm.enroll = function() {
      vm.errorMessage = null;
      IronicNode.create(vm.node,
        function(node) {
          $modalInstance.close(node);
        },
        function(error) {
          vm.errorMessage = error.data.error_message;
        }
      );
    };

    vm.close = function() {
      $modalInstance.dismiss();
    };

    // Initialize the drivers.
    vm.loadDrivers();
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller for our application header.
 */
angular.module('ironic').controller('HeaderController',
  function ($scope, $state, configurations, currentConfiguration, $$selectedConfiguration) {
    'use strict';
    var vm = this;

    vm.configurations = configurations;
    vm.currentConfiguration = currentConfiguration;

    vm.switchCloud = function (cloudConfig) {
      $$selectedConfiguration.set(cloudConfig).$promise.then(
        function() {
          $state.reload();
        });
    };
  });

/*
 * Copyright (c) 2016 Hewlett Packard Enterprise Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Controller which handles manipulation of individual nodes.
 */
angular.module('ironic').controller('NodeActionController',
  function($modal, $q) {
    'use strict';

    var vm = this;

    // Set up controller parameters
    vm.errorMessage = null;
    vm.node = null;

    /**
     * Initialize this controller with a specific node.
     *
     * @param {IronicNode} node The node to initialize this controller with.
     * @return {void}
     */
    vm.init = function(node) {
      vm.node = node;
    };

    /**
     * Delete the node in this controller.
     *
     * @return {Promise} A promise that will resolve true if the modal closed with some deletions.
     */
    vm.remove = function() {
      if (vm.node === null) {
        // init() not called, or called with invalid value.
        return $q.reject();
      }

      // Return the result of the modal.
      return $modal.open({
        templateUrl: 'view/ironic/action/remove_node.html',
        controller: 'RemoveNodeModalController as ctrl',
        backdrop: 'static',
        resolve: {
          nodes: function() {
            return [vm.node];
          }
        }
      }).result;
    };
  });

/*
 * Copyright (c) 2015 Hewlett Packard Enterprise Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Node handling for the Ironic UI.
 */
angular.module('ironic').controller('NodeDetailController',
  function(nodeUuid, IronicNode) {
    'use strict';
    var vm = this;

    // Set up controller parameters
    vm.errorMessage = null;
    vm.node = null;

    // Load the node.
    vm.node = IronicNode.read({
      uuid: nodeUuid
    }, angular.noop, function(error) {
      // Set the error message and clear the node promise.
      vm.errorMessage = error.data.error_message;
      vm.node = null;
    });
  });

/*
 * Copyright (c) 2015 Hewlett Packard Enterprise Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Node driver handling for the Ironic UI.
 */
angular.module('ironic').controller('NodeDetailDriverController',
  function(nodeUuid, IronicNode, IronicDriver) {
    'use strict';
    var vm = this;

    // Set up controller parameters
    vm.errorMessage = null;
    vm.driver = null;

    // Generic, shared error handler.
    function errorHandler (error) {
      // Set the error message and clear the port promise.
      vm.errorMessage = error.data.error_message;
      vm.driver = null;
    }

    // Load the node.
    IronicNode.read({
      uuid: nodeUuid
    }, function(node) {
      vm.driver = IronicDriver.read({
        uuid: node.driver
      }, angular.noop, errorHandler);
    }, errorHandler);

  });

/*
 * Copyright (c) 2015 Hewlett Packard Enterprise Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Node port handling for the Ironic UI.
 */
angular.module('ironic').controller('NodeDetailPortsController',
  function(nodeUuid, IronicPort) {
    'use strict';
    var vm = this;

    // Set up controller parameters
    vm.errorMessage = null;
    vm.ports = null;

    // Load the node.
    vm.ports = IronicPort.query({
      node: nodeUuid
    }, angular.noop, function(error) {
      // Set the error message and clear the node promise.
      vm.errorMessage = error.data.error_message;
      vm.ports = null;
    });
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Node handling for the Ironic UI.
 */
angular.module('ironic')
  .controller('NodeListController', function($scope, IronicNode, $modal, $log) {
    'use strict';
    var vm = this;

    // Set up controller parameters
    vm.errorMessage = null;
    vm.nodes = null;

    // Load the node list.
    vm.loadNodes = function() {
      vm.errorMessage = null;
      vm.nodes = IronicNode.query({}, function() {
        // Do nothing on success.
      }, function(error) {
        vm.errorMessage = error.data.error_message;
        vm.nodes = null;
      });
    };

    /**
     * Set the power state for the provided node node.
     *
     * @param {IronicNode} node The ironic node to modify.
     * @param {String} stateName The name of the power state.
     * @return {void}
     */
    vm.setPowerState = function(node, stateName) {
      // Do nothing, yet.
      $log.info('Set power state on ' + node.uuid + ' to ' + stateName);
    };

    vm.enroll = function() {
      $modal.open({
        templateUrl: 'view/ironic/enroll/index.html',
        controller: 'EnrollModalController as ctrl',
        backdrop: 'static'
      }).result.then(vm.loadNodes);
    };

    vm.loadNodes();
  });

/*
 * Copyright (c) 2016 Hewlett Packard Enterprise Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This controller backs the "Remove Node" modal.
 */
angular.module('ironic').controller('RemoveNodeModalController',
  function($modalInstance, nodes, $q) {
    'use strict';
    var vm = this;

    // Wrap all the nodes in their own context objects.
    vm.someDeleted = false;
    vm.deleting = false;
    vm.nodes = [];
    angular.forEach(nodes, function(node) {
      vm.nodes.push({
        node: node,
        error: null,
        state: 'present'
      });
    });

    /**
     * Remove all the nodes on this controller.
     *
     * @returns {void}
     */
    vm.remove = function() {
      vm.deleting = true;
      var promises = [];

      // For each context object in our controller...
      angular.forEach(vm.nodes, function(ctx) {
        // Try to delete it.
        ctx.state = 'removing';
        var promise = ctx.node.$remove(
          function() {
            vm.someDeleted = true;
            ctx.state = 'removed';
          },
          function(error) {
            ctx.state = 'error';
            ctx.error = error.data.error_message;
          }
        );
        promises.push(promise);
      });

      // Wait for all the promises...
      $q.all(promises).then(
        function() {
          // If all are successful, just close.
          vm.deleting = false;
          if (vm.someDeleted) {
            $modalInstance.close();
          } else {
            $modalInstance.dismiss();
          }
        },
        function() {
          // Flip the deleting bit.
          vm.deleting = false;
        });
    };

    /**
     * Close this modal.
     *
     * @returns {void}
     */
    vm.close = function() {
      if (vm.someDeleted) {
        // Something was changed.
        $modalInstance.close();
      } else {
        // Nothing was changed.
        $modalInstance.dismiss();
      }
    };
  });
