/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A module of resources that talk with the ironic API.
 */
angular.module('ironic.api', ['openstack'])
  .config(function ($$configurationProvider, $windowProvider) {
    'use strict';

    // Build a default ironic location from the $window provider
    var location = $windowProvider.$get().location;
    var apiRoot = location.protocol + '//' + location.hostname + ':6385';
    $$configurationProvider.$registerDefault('ironic', apiRoot);
  })
  .config(function($$resourceFactoryProvider) {

    $$resourceFactoryProvider
      .$addServiceFactory('ironic', function ($resource) {
        return function (baseUri, resourceName) {
          var resourceUrl = new URL(baseUri + '/' + resourceName +
            '/:uuid');

          // normalize the path
          resourceUrl.pathname =
            resourceUrl.pathname.replace('//', '/');

          return $resource(resourceUrl.toString(),
            {'uuid': '@uuid'}, {
              'query': {
                'method': 'GET',
                'isArray': true,
                'transformResponse': function (data) {
                  var parsed = angular.fromJson(data);
                  return parsed[resourceName];
                }
              },
              'create': {
                'method': 'POST'
              },
              'read': {
                'method': 'GET'
              },
              'update': {
                'method': 'PUT'
              },
              'delete': {
                'method': 'DELETE'
              }
            });
        };
      });
  })
  .config(function ($provide) {
    'use strict';

    function buildResource (resourceName) {

      return function ($$configuration, $$resourceFactory) {

        function getResource () {
          var serviceName = 'ironic';
          return $$resourceFactory.build(serviceName, resourceName);
        }

        return {
          'query': function () {
            var r = getResource(resourceName);
            return r.query.apply(r, arguments);
          },
          'create': function () {
            var r = getResource(resourceName);
            return r.create.apply(r, arguments);
          },
          'read': function () {
            var r = getResource(resourceName);
            return r.read.apply(r, arguments);
          },
          'update': function () {
            var r = getResource(resourceName);
            return r.update.apply(r, arguments);
          },
          'delete': function () {
            var r = getResource(resourceName);
            return r.delete.apply(r, arguments);
          }
        };
      };
    }

    $provide.service('IronicChassis', buildResource('chassis'));
    $provide.service('IronicDriver', buildResource('drivers'));
    $provide.service('IronicNode', buildResource('nodes'));
    $provide.service('IronicPort', buildResource('ports'));
  }
);

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * Form validator that will perform an asynchronous check to see if Ironic is
 * available at the provided url. It will expose a variable on the control to
 * which it is applied, with an array of detected API versions.
 */
/*eslint-disable angular/ng_no_services */
angular.module('ironic.api').directive('ironicApiUrl',
  function ($q, $http) {
/*eslint-enable angular/ng_no_services */
    'use strict';

    return {
      'require': 'ngModel',
      'link': function (scope, elm, attrs, ctrl) {

        ctrl.$ironicVersions = [];

        ctrl.$asyncValidators.ironicApiUrl =
          function (modelValue) {
            var def = $q.defer();

            $http.get(modelValue).then(function (result) {
              var name = result.data.name;

              if (name !== 'OpenStack Ironic API') {
                def.reject();
              }

              var versions = result.data.versions || [];
              for (var i = 0; i < versions.length; i++) {
                versions[i] = versions[i].id;
              }
              ctrl.$ironicVersions = versions;
              def.resolve();
            }, function () {
              def.reject();
            });

            return def.promise;
          };
      }
    };
  });
