/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * OpenStack specific configuration discovery.
 */
angular.module('openstack', ['ngCookies', 'ngResource']);

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This resource attempts to automatically detect your cloud's configuration
 * by searching for it in common locations. First we check config.json, living
 * on the server adjacent to index.html. Secondly, we construct a default
 * configuration object that assumes that all services are running on
 * the current domain. Note that the format is an array: it is feasible to
 * configure multiple access points within config.json.
 *
 * [{
 *     "name": "My Little Cloud",
 *     "ironic": {
 *          "api": "https://ironic.example.com:6385/"
 *     },
 *     "glance": {
 *          "api": "https://glance.example.com:9292/"
 *     }
 * }]
 */
angular.module('openstack').service('$$configuration',
  function ($q, $http, $log, $location, $$persistentStorage) {
    'use strict';

    /**
     * The storage key used in $$persistentStorage to store our local
     * configuration.
     *
     * @type {string}
     */
    var configStorageKey = 'local-configuration';

    /**
     * The resolved configuration stored in our local storage.
     */
    var localConfig = angular.fromJson(
      $$persistentStorage.get(configStorageKey) || '[]'
    );

    /**
     * In scope storage for the currently selected configuration.
     */
    var selectedConfig;

    /**
     * The key we use in our persistent storage to keep track of the
     * selected id.
     */
    var selectedStorageKey = 'selected-cloud';

    /**
     * Promise for the deferred file configuration.
     */
    var deferFile = null;

    /**
     * Promise for the autodetection configuration.
     */
    var deferAuto = null;

    /**
     * Saves a config value to the local storage.
     *
     * @return {void}
     */
    function saveLocal () {
      $$persistentStorage.set(configStorageKey, angular.toJson(localConfig));
    }

    /**
     * Removes a passed configuration from the local list.
     *
     * @param {{}} config The configuration object to remove.
     * @return {void}
     */
    function removeLocal (config) {
      for (var i = 0; i < localConfig.length; i++) {
        var c = localConfig[i];
        if (c.id === config.id) {
          localConfig.splice(i, 1);
          saveLocal();
          break;
        }
      }
    }

    /**
     * Attempt to load an included configuration file.
     *
     * @return {promise} A resolution promise.
     */
    function resolveConfigurationFile () {
      if (!deferFile) {
        $log.info('Attempting to load parameters from ./config.json');
        deferFile = $q.defer();
        $http.get('./config.json').then(
          function (response) {
            deferFile.resolve(response);
          },
          function () {
            $log.warn('Cannot load ./config.json, using defaults.');
            deferFile.resolve([]);
          }
        );
      }
      return deferFile.promise;
    }

    /**
     * Attempt to resolve configurations from localStorage.
     *
     * @return {promise} A resolution promise.
     */
    function resolveLocalStorage () {
      $log.info('Attempting to load parameters from localStorage');
      var deferred = $q.defer();
      deferred.resolve(localConfig);
      return deferred.promise;
    }

    /**
     * Build default configuration for services on the local server.
     *
     * @return {promise} A resolution promise.
     */
    function resolveAutodetect () {
      if (!deferAuto) {
        $log.info('Configuring local API endpoint.');
        deferAuto = $q.defer();
        var ironicApi =
          $location.protocol() + '://' + $location.host() + ':6385/';

        $http.get(ironicApi, {'timeout': 1000}).then(function (response) {
          var name = response.data.name || 'Local';
          var config = [
            {
              'id': 'localhost',
              'name': name,
              'ironic': {
                'api': ironicApi
              }
            }
          ];
          deferAuto.resolve(config);
        }, function () {
          deferAuto.resolve([]);
        });
      }
      return deferAuto.promise;
    }

    /**
     * Resolve all the configurations.
     *
     * @return {promise} A resolution promise.
     */
    function resolveAllConfigurations () {
      var deferAll = $q.defer();

      // Resolve the configuration.
      $q.all({
        'config': resolveConfigurationFile(),
        'default': resolveAutodetect(),
        'local': resolveLocalStorage()
      }).then(function (results) {
        var fileConfigs = results.config;
        var defaultConfigs = results.default;
        var localConfigs = results.local;

        var config = [];

        function addConfig (c) {
          if (c.hasOwnProperty('id')) {
            config.push(c);
          } else {
            $log.warn('Config block missing "id" ' +
              'field, ignoring.', c);
          }
        }

        fileConfigs.forEach(addConfig);
        defaultConfigs.forEach(addConfig);
        localConfigs.forEach(addConfig);

        deferAll.resolve(config);
      }, function () {
        deferAll.resolve([]);
      });

      return deferAll.promise;
    }

    /**
     * Resolve the current selected configuration.
     *
     * @return {promise} A resolution promise.
     */
    function resolveSelectedConfiguration () {
      var deferSelected = $q.defer();

      var selectedId = $$persistentStorage.get(selectedStorageKey);

      resolveAllConfigurations().then(function(configs) {
        // Pick the configuration from the loaded configs. Note
        // that if the selectedId is null, this will never
        // match.
        var selectedConfig;
        configs.forEach(function(config) {
          if (config.id === selectedId) {
            $log.debug('Selecting config: ' + selectedId);
            selectedConfig = config;
          }
        });

        // If the selectedConfig is null, chances are it was removed at runtime. Clear the
        // selectedId, if it exists.
        if (!selectedConfig) {
          $$persistentStorage.remove(selectedStorageKey);
        }
        deferSelected.resolve(selectedConfig);
      });

      return deferSelected.promise;
    }

    /**
     * Resolve any configuration parameters.
     */
    return {
      /**
       * Retrieve the currently selected API base for the provided
       * service.
       *
       * @param {String} service The name of the service.
       * @returns {String} The Base API for the configured service.
       */
      'getApiRoot': function (service) {
        if (!selectedConfig || !selectedConfig.hasOwnProperty(service)) {
          return '/';
        }
        return selectedConfig[service].apiRoot;
      },

      /**
       * Asynchronously resolve the Cloud Configuration. This will always
       * resolve, however it is likely that the resulting configuration
       * array is empty.
       *
       * @returns {promise} A promise that resolves all available configuration objects as an array.
       */
      'resolveAll': function () {
        return resolveAllConfigurations();
      },

      /**
       * Asynchronously resolve the local configuration.
       *
       * @return {promise} The content of any configuration file loaded.
       */
      'resolveConfigured': function () {
        return resolveConfigurationFile();
      },

      /**
       * Asynchronously resolve autodetected api's on the same domain.
       *
       * @return {promise} A configuration constructed from autodetected API's.
       */
      'resolveAutodetection': function () {
        return resolveAutodetect();
      },

      /**
       * Asynchronously resolve configurations in localStorage.
       *
       * @return {promise} A configuration configured by the user.
       */
      'resolveLocal': function () {
        return resolveLocalStorage();
      },

      /**
       * Asynchronously resolve the selected configuration. This promise
       * will be rejected if the detected cloud configuration has no
       * valid configuration blocks.
       *
       * @returns {promise} The user-selected configuration.
       */
      'resolveSelected': function () {
        return resolveSelectedConfiguration();
      },

      /**
       * Set the selected configuration. Note that you're going to have
       * to reload the entire application to make this work.
       *
       * @param {String} selectedId Set the current selected configuration name.
       * @return {void}
       */
      'setSelected': function (selectedId) {
        $$persistentStorage.set(selectedStorageKey, selectedId);
      },

      /**
       * This method adds a new configuration to the application.
       *
       * @returns {Object} The added configuration.
       */
      'add': function (newConfig) {
        localConfig.push(newConfig);
        saveLocal();
        return newConfig;
      },

      /**
       * This method removes a configuration from the application.
       *
       * @param {Object} config The configuration to remove.
       * @returns {Object} The removed configuration.
       */
      'remove': function (config) {
        return removeLocal(config);
      }
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * An application-wide static constant that contains error code constants.
 * Different modules and components may dynamically add error codes that serve
 * their own needs.
 */
angular.module('openstack').constant('$$errorCode', {});

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The resource cache acts as a centralized location where different services'
 * service abstractions may live. During the configuration phase, each
 * loaded API is expected to register a factory with the cache, which will
 * be later queried to build instances of resources for different cloud
 * configuration parameters.
 *
 * @deprecated
 */
angular.module('openstack').provider('$$resourceFactory',
  function () {
    'use strict';

    /**
     * Cache of all the factories.
     *
     * @type {{}}
     */
    var serviceFactoryCache = {};

    /**
     * Cache of all the constructed services by configName.
     *
     * @type {{}}
     */
    var apiCache = {};

    /**
     * Add a service factory for a given openstack service to the central
     * service factory. For each service, it will have its $build method
     * invoked with a specific global configuration object, and the name of
     * the resource that was requested.
     *
     * @param {String} serviceName The name of the service.
     * @param {Function} factory The factory implementation. Must be callable.
     * @returns {void}
     */
    this.$addServiceFactory = function (serviceName, factory) {
      serviceFactoryCache[serviceName] = factory;
    };

    /**
     * The factory's provider. Creates a singleton instance of the
     * $$resourceFactory service that permits the creation of different
     * API's.
     *
     * @param {*} $$configuration Injected configuration service.
     * @param {*} $log Injected log abstraction.
     * @param {*} $injector Injected injector.
     * @returns {{build: Function}} The constructed service.
     */
    this.$get = function resourceFactoryProvider ($$configuration, $log,
                                                  $injector) {

      /**
       * Get the cache of services available for a specific API root.
       *
       * @param {String} baseUri The root URI for the service cache.
       * @returns {*} The in-memory service cache.
       */
      function getServiceCache (baseUri) {
        if (!apiCache.hasOwnProperty(baseUri)) {
          apiCache[baseUri] = {};
        }
        return apiCache[baseUri];
      }

      /**
       * An internal resource factory, that builds ngResource instances in
       * accordance to the provided configuration.
       *
       * @param {String} serviceName A unique key under which this resource should be cached.
       * @param {String} baseUri The API's Base URI.
       * @param {String} resourceName The name of the resource to construct.
       * @returns {*} A constructed service.
       */
      function resourceFactory (serviceName, baseUri, resourceName) {

        // Get the appropriate resource factory.
        if (!serviceFactoryCache.hasOwnProperty(serviceName)) {
          $log.error('No resource factory for service [' +
            serviceName + '] was registered. Did you import the ' +
            'library?');
        }

        /**
         * Build the service.
         */
        var factoryBuilder = serviceFactoryCache[serviceName];
        var factory = $injector.invoke(factoryBuilder);
        return factory(baseUri, resourceName);
      }

      /**
       * Retrieve a specific resource for a given service in a given
       * configuration scope.
       *
       * @param {String} serviceName The name of the service. e.g. 'ironic'.
       * @param {String} resourceName The name of the root resource. e.g. 'node'
       * @returns {*} The constructed service.
       */
      function getService (serviceName, resourceName) {
        var baseUri = $$configuration.getApiRoot(serviceName);
        var cache = getServiceCache(baseUri);

        if (!cache.hasOwnProperty(resourceName)) {
          cache[resourceName] =
            resourceFactory(serviceName, baseUri, resourceName);
        }
        return cache[resourceName];
      }

      return {
        /**
         * Return an instance of a service for the given configuration
         * name, service name, and resource name.
         *
         * @param {String} serviceName The name of the service in the configuration.
         * @param {String} resourceName The name of the resource to construct.
         * @returns {*} A singleton instance of the service.
         */
        'build': function (serviceName, resourceName) {
          return getService(serviceName, resourceName);
        }
      };
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * The resource cache acts as a centralized location where different services' resource
 * instances may be stored. It plays a central role in multi-cloud environments, as a resource
 * abstraction - say IronicNode - can create and cache the actual ngResource instances for N>1
 * configured ironic endpoints.
 *
 * In order to maintain flexibility, it does not build resources, it only accepts preconstructed
 * instances. It is strongly recommended that you store your resource instances using the root
 * URI of the API to which your instance is talking, so that 'https://somecloud.com:6385/v1/nodes'
 * will be stored separately from 'https://someothercloud.com:6385/v1/nodes'.
 */
angular.module('openstack').factory('$$resourceCache',
  function() {
    'use strict';

    /**
     * Cache of all the resource instances.
     *
     * @type {{}}
     */
    var resourceCache = {};

    return {

      /**
       * Store a resource instance in the cache.
       *
       * @param {String} uri The root uri of the resource.
       * @param {*} resource The resource to store.
       * @return {*} The stored resource.
       */
      'set': function(uri, resource) {
        resourceCache[uri] = resource;
        return resource;
      },

      /**
       * Retrieve a resource instance from the cache.
       *
       * @param {String} uri The uri of the resource to retrieve.
       * @return {*|undefined} The resource, or undefined.
       */
      'get': function(uri) {
        if (resourceCache[uri]) {
          return resourceCache[uri];
        }
      },

      /**
       * Check whether or not a resource is in the cache.
       *
       * @param {String} uri The uri of the resource to check.
       * @return {true|false} Whether this resource has already been cached.
       */
      'has': function(uri) {
        return resourceCache.hasOwnProperty(uri);
      },

      /**
       * Remove a specific resource from the cache.
       *
       * @param {String} uri The uri of the resource to remove.
       * @returns {void}
       */
      'remove': function(uri) {
        if (resourceCache.hasOwnProperty(uri)) {
          delete resourceCache[uri];
        }
      },

      /**
       * Return all the uri keys currently registered.
       *
       * @returns {Array} An array of all registered uri keys.
       */
      'keys': function() {
        var keys = [];
        /*eslint-disable guard-for-in*/
        for (var key in resourceCache) {
          keys.push(key);
        }
        /*eslint-enable guard-for-in*/
        return keys;
      },

      /**
       * Remove everything from the cache.
       *
       * @returns {void}
       */
      'clearAll': function() {
        var keys = this.keys();
        for (var i = 0; i < keys.length; i++) {
          delete resourceCache[keys[i]];
        }
      },

      /**
       * Return the number of resources in the cache.
       *
       * @returns {number} The number of resources cached.
       */
      'length': function() {
        return this.keys().length;
      }
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This file provides an implementation of the storage API, backed by cookies.
 * This particular implementation is not intelligent: It will access the
 * cookie for this domain, as configured by the $cookieProvider, and will
 * grant access to all values stored this way.
 */
angular.module('openstack').factory('$$cookieStorage',
  function($cookies) {
    'use strict';

    return {

      /**
       * Is this storage type supported?
       *
       * @returns {boolean} True if it is supported, otherwise false.
       */
      'isSupported': function() {
        return true;
      },

      /**
       * Set a value to the provided key in memory storage. If the
       * value already exists it will be overwritten.
       *
       * @param {String} key The key to store the value at.
       * @param {*} value The value to store.
       * @return {*} The stored value.
       */
      'set': function(key, value) {
        $cookies.put(key, angular.toJson(value));
        return value;
      },

      /**
       * Retrieve a value from this storage provider.
       *
       * @param {String} key The key to retrieve.
       * @return {*|undefined} The value, or undefined if it is not set.
       */
      'get': function(key) {
        var result = angular.fromJson($cookies.get(key));
        if (result) {
          return result;
        }
      },

      /**
       * Remove a specific value from the storage provider.
       *
       * @param {String} key The key to remove.
       * @returns {void}
       */
      'remove': function(key) {
        $cookies.remove(key);
      },

      /**
       * Return all the keys currently registered.
       *
       * @returns {Array} An array of all registered keys.
       */
      'keys': function() {
        var all = $cookies.getAll();
        var keys = [];
        /*eslint-disable guard-for-in*/
        for (var key in all) {
          keys.push(key);
        }
        /*eslint-enable guard-for-in*/
        return keys;
      },

      /**
       * Remove everything from the memory storage mechanism.
       *
       * @returns {void}
       */
      'clearAll': function() {
        var all = $cookies.getAll();
        /*eslint-disable guard-for-in*/
        for (var key in all) {
          $cookies.remove(key);
        }
        /*eslint-enable guard-for-in*/
      },

      /**
       * Return the size of the current memory storage.
       *
       * @returns {number} The number of keys in storage.
       */
      'length': function() {
        return this.keys().length;
      }
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A $$localStorage service behind a common API. If localStorage is not
 * supported, this will log a warning to the console. If you want a provider
 * that gracefully degrades, use $$persistentStorage.
 */
angular.module('openstack').factory('$$localStorage',
  function($window, $log) {
    'use strict';

    /**
     * Detect whether localStorage is supported, and make sure we can write
     * to it.
     */
    var isSupported = (function() {

      // Does it exist?
      if (!$window.localStorage) {
        return false;
      }

      // Can we write to it?
      var testKey = '__' + Math.round(Math.random() * 1e7);
      try {
        $window.localStorage.setItem(testKey, '');
        $window.localStorage.removeItem(testKey);
        return true;
      } catch (e) {
        return false;
      }
    })();

    return {

      /**
       * Is this storage type supported?
       *
       * @returns {boolean} True if it is supported, otherwise false.
       */
      'isSupported': function() {
        return isSupported;
      },

      /**
       * Set a value of the provided key. If the
       * value already exists it will be overwritten.
       *
       * @param {String} key The key to store the value at.
       * @param {*} value The value to store.
       * @return {*} The stored value.
       */
      'set': function(key, value) {
        if (isSupported) {
          $window.localStorage.setItem(key, angular.toJson(value));
          return value;
        }
        $log.warn('$$localStorage not supported');
      },

      /**
       * Retrieve a value from this storage provider.
       *
       * @param {String} key The key to retrieve.
       * @return {*|undefined} The value, or undefined if it is not set.
       */
      'get': function(key) {
        if (isSupported) {
          var result = angular.fromJson($window.localStorage.getItem(key));
          if (result) {
            return result;
          }
          return; // undefined
        }
        $log.warn('$$localStorage not supported');
      },

      /**
       * Remove a specific value from the storage provider.
       *
       * @param {String} key The key to remove.
       * @returns {void}
       */
      'remove': function(key) {
        if (isSupported) {
          return $window.localStorage.removeItem(key);
        }
        $log.warn('$$localStorage not supported');
      },

      /**
       * Return all the keys currently registered.
       *
       * @returns {Array} An array of all registered keys.
       */
      'keys': function() {
        if (isSupported) {
          var keys = [];
          for (var i = 0; i < $window.localStorage.length; i++) {
            keys.push($window.localStorage.key(i));
          }
          return keys;
        }
        $log.warn('$$localStorage not supported');
        return [];
      },

      /**
       * Remove everything from the memory storage mechanism.
       *
       * @returns {void}
       */
      'clearAll': function() {
        if (isSupported) {
          var keys = this.keys();
          for (var i = 0; i < keys.length; i++) {
            this.remove(keys[i]);
          }
        }
        $log.warn('$$localStorage not supported');
      },

      /**
       * Return the size of the current memory storage.
       *
       * @returns {number} The number of keys in storage.
       */
      'length': function() {
        if (isSupported) {
          return $window.localStorage.length;
        }
        $log.warn('$$localStorage not supported');
        return 0;
      }
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * This provides a memory-based key/value storage mechanism. It's provided as
 * a fallback option for all other storage mechanisms, to prevent unexpected
 * runtime failures.
 */
angular.module('openstack').factory('$$memoryStorage',
  function() {
    'use strict';

    var memoryStorage = {};

    return {

      /**
       * Is this storage type supported?
       *
       * @returns {boolean} True if it is supported, otherwise false.
       */
      'isSupported': function() {
        return true;
      },

      /**
       * Set a value to the provided key in memory storage. If the
       * value already exists it will be overwritten.
       *
       * @param {String} key The key to store the value at.
       * @param {*} value The value to store.
       * @return {*} The stored value.
       */
      'set': function(key, value) {
        memoryStorage[key] = value;

        return value;
      },

      /**
       * Retrieve a value from this storage provider.
       *
       * @param {String} key The key to retrieve.
       * @return {*|undefined} The value, or undefined if it is not set.
       */
      'get': function(key) {
        if (memoryStorage.hasOwnProperty(key)) {
          return memoryStorage[key];
        }
      },

      /**
       * Remove a specific value from the storage provider.
       *
       * @param {String} key The key to remove.
       * @returns {void}
       */
      'remove': function(key) {
        delete memoryStorage[key];
      },

      /**
       * Return all the keys currently registered.
       *
       * @returns {Array} An array of all registered keys.
       */
      'keys': function() {
        var keys = [];
        /*eslint-disable guard-for-in*/
        for (var key in memoryStorage) {
          keys.push(key);
        }
        /*eslint-enable guard-for-in*/
        return keys;
      },

      /**
       * Remove everything from the memory storage mechanism.
       *
       * @returns {void}
       */
      'clearAll': function() {
        var keys = [];
        /*eslint-disable guard-for-in*/
        for (var key in memoryStorage) {
          keys.push(key);
        }
        /*eslint-enable guard-for-in*/

        for (var i = 0; i < keys.length; i++) {
          delete memoryStorage[keys[i]];
        }
      },

      /**
       * Return the size of the current memory storage.
       *
       * @returns {number} The number of keys in storage.
       */
      'length': function() {
        return this.keys().length;
      }
    };
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A convenience component that automatically selects the most secure, and most
 * persistent, storage mechanism available in the current runtime. This does
 * not include sessionStorage, which must be used independently.
 */
angular.module('openstack').factory('$$persistentStorage',
  function($log, $$cookieStorage, $$memoryStorage, $$localStorage) {
    'use strict';

    // Check for local storage.
    if ($$localStorage.isSupported()) {
      return $$localStorage;
    }

    // Check for cookie storage.
    if ($$cookieStorage.isSupported()) {
      return $$cookieStorage;
    }

    $log.warn('Warning: No persistent storage mechanism supported, all storage will be transient.');
    return $$memoryStorage;
  });

/*
 * Copyright (c) 2015 Hewlett-Packard Development Company, L.P.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License'); you may
 * not use this file except in compliance with the License. You may obtain
 * a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an 'AS IS' BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/**
 * A $$sessionStorage service behind a common API. If sessionStorage is not
 * supported, this will log a warning to the console. If you want a provider
 * that gracefully degrades, use $$persistentStorage.
 */
angular.module('openstack').factory('$$sessionStorage',
  function($window, $log) {
    'use strict';

    /**
     * Detect whether sessionStorage is supported, and make sure we can
     * write to it.
     */
    var isSupported = (function() {

      // Does it exist?
      if (!$window.sessionStorage) {
        return false;
      }

      // Can we write to it?
      var testKey = '__' + Math.round(Math.random() * 1e7);
      try {
        $window.sessionStorage.setItem(testKey, '');
        $window.sessionStorage.removeItem(testKey);
        return true;
      } catch (e) {
        return false;
      }
    })();

    return {

      /**
       * Is this storage type supported?
       *
       * @returns {boolean} True if it is supported, otherwise false.
       */
      'isSupported': function() {
        return isSupported;
      },

      /**
       * Set a value of the provided key. If the
       * value already exists it will be overwritten.
       *
       * @param {String} key The key to store the value at.
       * @param {*} value The value to store.
       * @return {*} The stored value.
       */
      'set': function(key, value) {
        if (isSupported) {
          $window.sessionStorage.setItem(key, angular.toJson(value));
          return value;
        }
        $log.warn('$$sessionStorage not supported');
      },

      /**
       * Retrieve a value from this storage provider.
       *
       * @param {String} key The key to retrieve.
       * @return {*|undefined} The value, or undefined if it is not set.
       */
      'get': function(key) {
        if (isSupported) {
          var result = angular.fromJson($window.sessionStorage.getItem(key));
          if (result) {
            return result;
          }
          return; // undefined
        }
        $log.warn('$$sessionStorage not supported');
      },

      /**
       * Remove a specific value from the storage provider.
       *
       * @param {String} key The key to remove.
       * @returns {void}
       */
      'remove': function(key) {
        if (isSupported) {
          return $window.sessionStorage.removeItem(key);
        }
        $log.warn('$$sessionStorage not supported');
      },

      /**
       * Return all the keys currently registered.
       *
       * @returns {Array} An array of all registered keys.
       */
      'keys': function() {
        if (isSupported) {
          var keys = [];
          for (var i = 0; i < $window.sessionStorage.length; i++) {
            keys.push($window.sessionStorage.key(i));
          }
          return keys;
        }
        $log.warn('$$sessionStorage not supported');
        return [];
      },

      /**
       * Remove everything from the memory storage mechanism.
       *
       * @returns {void}
       */
      'clearAll': function() {
        if (isSupported) {
          var keys = this.keys();
          for (var i = 0; i < keys.length; i++) {
            this.remove(keys[i]);
          }
          return;
        }
        $log.warn('$$sessionStorage not supported');
      },

      /**
       * Return the size of the current memory storage.
       *
       * @returns {number} The number of keys in storage.
       */
      'length': function() {
        if (isSupported) {
          return $window.sessionStorage.length;
        }
        $log.warn('$$sessionStorage not supported');
        return 0;
      }
    };
  });
