# Copyright 2012 Hewlett-Packard Development Company, L.P.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations


# Mapping of --driver options to a class
known_drivers = {
    'haproxy': 'libra.worker.drivers.haproxy.driver.HAProxyDriver'
}


class LoadBalancerDriver(object):
    """
    Load balancer device driver base class.

    This defines the API for interacting with various load balancing
    appliances. Drivers for these appliances should inherit from this
    class and implement the relevant API methods that it can support.

    Generally, an appliance driver should queue up any changes made
    via these API calls until the create() method is called.

    This design allows for a single load balancer to support multiple
    protocols simultaneously. Each protocol added via the add_protocol()
    method is assumed to be unique, and one protocol per port. This same
    protocol is then supplied to other methods (e.g., add_server() and
    set_algorithm()) to make changes for that specific protocol.
    """

    # Load balancer algorithms
    ROUNDROBIN = 1
    LEASTCONN = 2

    def init(self):
        """ Allows the driver to do any initialization for a new config. """
        raise NotImplementedError()

    def add_protocol(self, protocol, port):
        """ Add a supported protocol and listening port for the instance. """
        raise NotImplementedError()

    def add_server(self, protocol, host, port, weight):
        """ Add a server for the protocol for which we will proxy. """
        raise NotImplementedError()

    def set_algorithm(self, protocol, algo):
        """ Set the algorithm used by the load balancer for this protocol. """
        raise NotImplementedError()

    def create(self):
        """ Create the load balancer. """
        raise NotImplementedError()

    def suspend(self):
        """ Suspend the load balancer. """
        raise NotImplementedError()

    def enable(self):
        """ Enable a suspended load balancer. """
        raise NotImplementedError()

    def delete(self):
        """ Delete a load balancer. """
        raise NotImplementedError()

    def get_stats(self, protocol):
        """ Get load balancer statistics for specified protocol. """
        raise NotImplementedError()

    def archive(self, method, params):
        """ Archive the load balancer logs using the specified method. """
        raise NotImplementedError()
