# Copyright 2013 Hewlett-Packard Development Company, L.P.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import requests
import random
import sys
import json


class APIError(Exception):
    pass


class AdminAPI(object):
    def __init__(self, addresses, logger):
        self.logger = logger
        self.headers = {'Content-type': 'application/json'}
        random.shuffle(addresses)
        for address in addresses:
            self.url = 'https://{0}/v1'.format(address)
            self.logger.info('Trying {url}'.format(url=self.url))
            status, data = self._get('{url}/devices/usage'
                                     .format(url=self.url))
            if status:
                self.logger.info('API Server is online')
                self.online = True
                return

        # if we get this far all API servers are down
        self.online = False

    def get_ping_list(self):
        marker = 0
        limit = 20
        lb_list = []
        while True:
            success, nodes = self._get_node_list(limit, marker)
            if not success:
                raise APIError
            # if we hit an empty device list we have hit end of list
            if not len(nodes['devices']):
                break

            for device in nodes['devices']:
                if device['status'] == 'ONLINE':
                    lb_list.append(device)
            marker = marker + limit
        return lb_list

    def get_repair_list(self):
        marker = 0
        limit = 20
        lb_list = []
        while True:
            success, nodes = self._get_node_list(limit, marker)
            if not success:
                raise APIError
            # if we hit an empty device list we have hit end of list
            if not len(nodes['devices']):
                break

            for device in nodes['devices']:
                if device['status'] == 'ERROR' and \
                   len(device['loadBalancers']) > 0:
                    lb_list.append(device)
            marker = marker + limit
        return lb_list

    def fail_device(self, device_id):
        body = {
            "status": "ERROR",
            "statusDescription": "Load balancer failed ping test"
        }
        self._put(
            '{url}/devices/{device_id}'.format(
                url=self.url, device_id=device_id
            ), body
        )

    def repair_device(self, device_id):
        body = {
            "status": "ONLINE",
            "statusDescription": "Load balancer repaired"
        }
        self._put(
            '{url}/devices/{device_id}'.format(
                url=self.url, device_id=device_id
            ), body
        )

    def get_device(self, device_id):
        return self._get(
            '{url}/devices/{device_id}'.format(
                url=self.url, device_id=device_id
            )
        )

    def _get_node_list(self, limit, marker):
        return self._get(
            '{url}/devices?marker={marker}&limit={limit}'
            .format(url=self.url, marker=marker, limit=limit)
        )

    def _put(self, url, data):
        try:
            r = requests.put(
                url, data=json.dumps(data), verify=False, headers=self.headers
            )
        except requests.exceptions.RequestException:
            self.logger.exception('Exception communicating to server')
            return False, None

        if r.status_code != 200:
            self.logger.error('Server returned error {code}'
                              .format(code=r.status_code))
            return False, r.json()
        return True, r.json()

    def _get(self, url):
        try:
            r = requests.get(url, verify=False)
        except requests.exceptions.RequestException:
            self.logger.error('Exception communicating to server: {exc}'
                              .format(exc=sys.exc_info()[0]))
            return False, None

        if r.status_code != 200:
            self.logger.error('Server returned error {code}'
                              .format(code=r.status_code))
            return False, r.json()
        return True, r.json()

    def is_online(self):
        return self.online
