# Copyright 2010 United States Government as represented by the
# Administrator of the National Aeronautics and Space Administration.
# All Rights Reserved.
# Copyright 2012 Red Hat, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""Command-line flag library.

Emulates gflags by wrapping cfg.ConfigOpts.

The idea is to move fully to cfg eventually, and this wrapper is a
stepping stone.

"""

import socket

from oslo_config import cfg

from manila.i18n import _

CONF = cfg.CONF


def _get_my_ip():
    """Returns the actual ip of the local machine.

    This code figures out what source address would be used if some traffic
    were to be sent out to some well known address on the Internet. In this
    case, a Google DNS server is used, but the specific address does not
    matter much.  No traffic is actually sent.
    """
    try:
        csock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        csock.connect(('8.8.8.8', 80))
        (addr, port) = csock.getsockname()
        csock.close()
        return addr
    except socket.error:
        return "127.0.0.1"


core_opts = [
    cfg.StrOpt('api_paste_config',
               default="api-paste.ini",
               help='File name for the paste.deploy config for manila-api.'),
    cfg.StrOpt('state_path',
               default='/var/lib/manila',
               help="Top-level directory for maintaining manila's state."), ]

debug_opts = [
]

CONF.register_cli_opts(core_opts)
CONF.register_cli_opts(debug_opts)

global_opts = [
    cfg.StrOpt('my_ip',
               default=_get_my_ip(),
               help='IP address of this host.'),
    cfg.StrOpt('scheduler_topic',
               default='manila-scheduler',
               help='The topic scheduler nodes listen on.'),
    cfg.StrOpt('share_topic',
               default='manila-share',
               help='The topic share nodes listen on.'),
    cfg.BoolOpt('enable_v1_api',
                default=True,
                help=_("Deploy v1 of the Manila API.")),
    cfg.BoolOpt('enable_v2_api',
                default=True,
                help=_("Deploy v2 of the Manila API.")),
    cfg.BoolOpt('api_rate_limit',
                default=True,
                help='Whether to rate limit the API.'),
    cfg.ListOpt('osapi_share_ext_list',
                default=[],
                help='Specify list of extensions to load when using osapi_'
                     'share_extension option with manila.api.contrib.'
                     'select_extensions.'),
    cfg.MultiStrOpt('osapi_share_extension',
                    default=['manila.api.contrib.standard_extensions'],
                    help='The osapi share extension to load.'),
    cfg.StrOpt('osapi_share_base_URL',
               default=None,
               help='Base URL that will be presented to users in links '
                    'to the OpenStack Share API.',
               deprecated_name='osapi_compute_link_prefix'),
    cfg.IntOpt('osapi_max_limit',
               default=1000,
               help='The maximum number of items returned in a single '
                    'response from a collection resource.'),
    cfg.StrOpt('sqlite_db',
               default='manila.sqlite',
               help='The filename to use with sqlite.'),
    cfg.BoolOpt('sqlite_synchronous',
                default=True,
                help='If passed, use synchronous mode for sqlite.'),
    cfg.IntOpt('sql_idle_timeout',
               default=3600,
               help='Timeout before idle SQL connections are reaped.'),
    cfg.IntOpt('sql_max_retries',
               default=10,
               help='Maximum database connection retries during startup. '
                    '(setting -1 implies an infinite retry count).'),
    cfg.IntOpt('sql_retry_interval',
               default=10,
               help='Interval between retries of opening a SQL connection.'),
    cfg.StrOpt('scheduler_manager',
               default='manila.scheduler.manager.SchedulerManager',
               help='Full class name for the scheduler manager.'),
    cfg.StrOpt('share_manager',
               default='manila.share.manager.ShareManager',
               help='Full class name for the share manager.'),
    cfg.StrOpt('host',
               default=socket.gethostname(),
               help='Name of this node.  This can be an opaque identifier.  '
                    'It is not necessarily a hostname, FQDN, or IP address.'),
    # NOTE(vish): default to nova for compatibility with nova installs
    cfg.StrOpt('storage_availability_zone',
               default='nova',
               help='Availability zone of this node.'),
    cfg.ListOpt('memcached_servers',
                default=None,
                help='Memcached servers or None for in process cache.'),
    cfg.StrOpt('share_usage_audit_period',
               default='month',
               help='Time period to generate share usages for.  '
                    'Time period must be hour, day, month or year.'),
    cfg.StrOpt('root_helper',
               default='sudo',
               help='Deprecated: command to use for running commands as '
                    'root.'),
    cfg.StrOpt('rootwrap_config',
               default=None,
               help='Path to the rootwrap configuration file to use for '
                    'running commands as root.'),
    cfg.BoolOpt('monkey_patch',
                default=False,
                help='Whether to log monkey patching.'),
    cfg.ListOpt('monkey_patch_modules',
                default=[],
                help='List of modules or decorators to monkey patch.'),
    cfg.IntOpt('service_down_time',
               default=60,
               help='Maximum time since last check-in for up service.'),
    cfg.StrOpt('share_api_class',
               default='manila.share.api.API',
               help='The full class name of the share API class to use.'),
    cfg.StrOpt('auth_strategy',
               default='keystone',
               help='The strategy to use for auth. Supports noauth, keystone, '
                    'and deprecated.'),
    cfg.ListOpt('enabled_backends',
                default=None,
                help='A list of backend names to use. These backend names '
                     'should be backed by a unique [CONFIG] group '
                     'with its options.'),
    cfg.ListOpt('enabled_share_backends',
                default=None,
                help='A list of share backend names to use. These backend '
                     'names should be backed by a unique [CONFIG] group '
                     'with its options.'),
    cfg.BoolOpt('no_snapshot_gb_quota',
                default=False,
                help='Whether snapshots count against Gigabyte quota.'), ]

CONF.register_opts(global_opts)
