# Copyright 2015 Mirantis Inc.
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from tempest_lib import exceptions as lib_exc  # noqa
import testtools  # noqa

from tempest.api.share import base
from tempest.common.utils import data_utils
from tempest import config_share as config
from tempest import test

CONF = config.CONF


class ManageNFSShareTest(base.BaseSharesAdminTest):
    protocol = 'nfs'

    # NOTE(vponomaryov): be careful running these tests using generic driver
    # because cinder volumes will stay attached to service Nova VM and
    # won't be deleted.

    @classmethod
    @testtools.skipIf(
        CONF.share.multitenancy_enabled,
        "Only for driver_handles_share_servers = False driver mode.")
    @testtools.skipUnless(
        CONF.share.run_manage_unmanage_tests,
        "Manage/unmanage tests are disabled.")
    def resource_setup(cls):
        super(ManageNFSShareTest, cls).resource_setup()
        if cls.protocol not in CONF.share.enable_protocols:
            message = "%s tests are disabled" % cls.protocol
            raise cls.skipException(message)

        # Create share types
        cls.st_name = data_utils.rand_name("manage-st-name")
        cls.st_name_invalid = data_utils.rand_name("manage-st-name-invalid")
        cls.extra_specs = {
            'storage_protocol': CONF.share.storage_protocol,
            'driver_handles_share_servers': False
        }
        cls.extra_specs_invalid = {
            'storage_protocol': CONF.share.storage_protocol,
            'driver_handles_share_servers': True
        }

        __, cls.st = cls.create_share_type(
            name=cls.st_name,
            cleanup_in_class=True,
            extra_specs=cls.extra_specs)

        __, cls.st_invalid = cls.create_share_type(
            name=cls.st_name_invalid,
            cleanup_in_class=True,
            extra_specs=cls.extra_specs_invalid)

        creation_data = {'kwargs': {
            'share_type_id': cls.st['share_type']['id'],
            'share_protocol': cls.protocol,
        }}
        # Create two shares in parallel
        cls.shares = cls.create_shares([creation_data, creation_data])

        # Load all share data (host, etc.)
        __, cls.share1 = cls.shares_client.get_share(cls.shares[0]['id'])
        __, cls.share2 = cls.shares_client.get_share(cls.shares[1]['id'])

        # Unmanage shares from manila
        cls.shares_client.unmanage_share(cls.share1['id'])
        cls.shares_client.wait_for_resource_deletion(share_id=cls.share1['id'])
        cls.shares_client.unmanage_share(cls.share2['id'])
        cls.shares_client.wait_for_resource_deletion(share_id=cls.share2['id'])

    @test.attr(type=["gate", "smoke"])
    def test_manage(self):
        # manage share
        resp, share = self.shares_client.manage_share(
            service_host=self.share1['host'],
            export_path=self.share1['export_locations'][0],
            protocol=self.share1['share_proto'],
            share_type_id=self.st['share_type']['id'])
        self.assertIn(int(resp["status"]), self.HTTP_SUCCESS)

        # Add managed share to cleanup queue
        self.method_resources.insert(
            0, {'type': 'share_type', 'id': share['id'],
                'client': self.shares_client})

        # Wait for success
        self.shares_client.wait_for_share_status(share['id'], 'available')

        # delete share
        resp, __ = self.shares_client.delete_share(share['id'])
        self.assertIn(int(resp["status"]), self.HTTP_SUCCESS)
        self.shares_client.wait_for_resource_deletion(share_id=share['id'])
        self.assertRaises(lib_exc.NotFound,
                          self.shares_client.get_share,
                          share['id'])

    @test.attr(type=["gate", "smoke"])
    def test_manage_retry(self):
        # manage share with invalid parameters
        share = None
        parameters = [(self.st_invalid['share_type']['id'], 'MANAGE_ERROR'),
                      (self.st['share_type']['id'], 'available')]

        for share_type_id, status in parameters:
            resp, share = self.shares_client.manage_share(
                service_host=self.share2['host'],
                export_path=self.share2['export_locations'][0],
                protocol=self.share2['share_proto'],
                share_type_id=share_type_id)
            self.assertIn(int(resp["status"]), self.HTTP_SUCCESS)

            # Add managed share to cleanup queue
            self.method_resources.insert(
                0, {'type': 'share_type', 'id': share['id'],
                    'client': self.shares_client})

            # Wait for success
            self.shares_client.wait_for_share_status(share['id'], status)

        # delete share
        resp, __ = self.shares_client.delete_share(share['id'])
        self.assertIn(int(resp["status"]), self.HTTP_SUCCESS)
        self.shares_client.wait_for_resource_deletion(share_id=share['id'])
        self.assertRaises(lib_exc.NotFound,
                          self.shares_client.get_share,
                          share['id'])


class ManageCIFSShareTest(ManageNFSShareTest):
    protocol = 'cifs'
