#
# Copyright (C) 2015 Ericsson India Global Services Pvt Ltd.
#
#  Licensed under the Apache License, Version 2.0 (the "License"); you may
#  not use this file except in compliance with the License. You may obtain
#  a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#  License for the specific language governing permissions and limitations
#  under the License.
#

from oslo_config import cfg
from oslo_log import log as logging
from oslo_utils import excutils
import requests

from networking_bgpvpn.neutron.extensions import bgpvpn as bgpvpn_ext
from networking_bgpvpn.neutron.services.service_drivers import driver_api
from networking_odl.common import client as odl_client

cfg.CONF.import_group('ml2_odl', 'networking_odl.common.config')

LOG = logging.getLogger(__name__)
BGPVPNS = 'bgpvpns'
OPENDAYLIGHT_BGPVPN_DRIVER_NAME = 'OpenDaylight'


class OpenDaylightBgpvpnDriver(driver_api.BGPVPNDriver):

    """OpenDaylight BGPVPN Driver

    This code is the backend implementation for the OpenDaylight BGPVPN
    driver for Openstack Neutron.
    """

    def __init__(self, service_plugin):
        LOG.debug("Initializing OpenDaylight BGPVPN driver")
        super(OpenDaylightBgpvpnDriver, self).__init__(service_plugin)
        self.service_plugin = service_plugin

        self.client = odl_client.OpenDaylightRestClient(
            cfg.CONF.ml2_odl.url, cfg.CONF.ml2_odl.username,
            cfg.CONF.ml2_odl.password, cfg.CONF.ml2_odl.timeout)

    def create_bgpvpn_postcommit(self, context, bgpvpn):
        url = BGPVPNS
        try:
            self.client.sendjson('post', url, {BGPVPNS[:-1]: bgpvpn})
        except requests.exceptions.RequestException:
            with excutils.save_and_reraise_exception():
                # delete from db
                d_bgpvpn = self.bgpvpn_db.delete_bgpvpn(context, bgpvpn['id'])
                LOG.debug("Deleted bgpvpn %s from db", d_bgpvpn)

    def delete_bgpvpn_postcommit(self, context, bgpvpn):
        url = BGPVPNS + '/' + bgpvpn['id']
        self.client.sendjson('delete', url, None)

    def update_bgpvpn_postcommit(self, context, old_bgpvpn, bgpvpn):
        url = BGPVPNS + '/' + bgpvpn['id']
        self.client.sendjson('put', url, {BGPVPNS[:-1]: bgpvpn})

    def create_net_assoc_postcommit(self, context, net_assoc):
        bgpvpn = self.get_bgpvpn(context, net_assoc['bgpvpn_id'])
        url = BGPVPNS + '/' + bgpvpn['id']
        try:
            self.client.sendjson('put', url, {BGPVPNS[:-1]: bgpvpn})
        except requests.exceptions.RequestException:
            with excutils.save_and_reraise_exception():
                # delete from db
                d_netassoc = self.bgpvpn_db.delete_net_assoc(
                    context, net_assoc['id'], net_assoc['bgpvpn_id'])
                LOG.debug("Deleted net_assoc %s from db", d_netassoc)

    def delete_net_assoc_postcommit(self, context, net_assoc):
        bgpvpn = self.get_bgpvpn(context, net_assoc['bgpvpn_id'])
        url = BGPVPNS + '/' + bgpvpn['id']
        self.client.sendjson('put', url, {BGPVPNS[:-1]: bgpvpn})

    def create_router_assoc(self, context, bgpvpn_id, router_association):
        raise bgpvpn_ext.BGPVPNRouterAssociationNotSupported(
            driver=OPENDAYLIGHT_BGPVPN_DRIVER_NAME)

    def get_router_assoc(self, context, assoc_id, bgpvpn_id, fields=None):
        raise bgpvpn_ext.BGPVPNRouterAssociationNotSupported(
            driver=OPENDAYLIGHT_BGPVPN_DRIVER_NAME)

    def get_router_assocs(self, context, bgpvpn_id, filters=None, fields=None):
        raise bgpvpn_ext.BGPVPNRouterAssociationNotSupported(
            driver=OPENDAYLIGHT_BGPVPN_DRIVER_NAME)

    def delete_router_assoc(self, context, assoc_id, bgpvpn_id):
        raise bgpvpn_ext.BGPVPNRouterAssociationNotSupported(
            driver=OPENDAYLIGHT_BGPVPN_DRIVER_NAME)
