# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import mock

from oslo_config import cfg
from oslo_db import options
from oslotest import base

from neutron_lib.api import attributes
from neutron_lib.callbacks import registry
from neutron_lib.db import model_base
from neutron_lib import fixture
from neutron_lib.plugins import directory
from neutron_lib.tests.unit.api import test_attributes


class PluginDirectoryFixtureTestCase(base.BaseTestCase):

    def setUp(self):
        super(PluginDirectoryFixtureTestCase, self).setUp()
        self.directory = mock.Mock()
        self.useFixture(fixture.PluginDirectoryFixture(
            plugin_directory=self.directory))

    def test_fixture(self):
        directory.add_plugin('foo', 'foo')
        self.assertTrue(self.directory.add_plugin.called)


class CallbackRegistryFixtureTestCase(base.BaseTestCase):

    def setUp(self):
        super(CallbackRegistryFixtureTestCase, self).setUp()
        self.manager = mock.Mock()
        self.useFixture(fixture.CallbackRegistryFixture(
            callback_manager=self.manager))

    def test_fixture(self):
        registry.notify('a', 'b', self)
        self.assertTrue(self.manager.notify.called)


class SqlFixtureTestCase(base.BaseTestCase):

    def setUp(self):
        super(SqlFixtureTestCase, self).setUp()
        options.set_defaults(
            cfg.CONF,
            connection='sqlite://')
        self.useFixture(fixture.SqlFixture())

    def test_fixture(self):
        self.assertIsNotNone(model_base.BASEV2.metadata.sorted_tables)


class APIDefinitionFixtureTestCase(base.BaseTestCase):

    def _test_all_api_definitions_fixture(self, global_cleanup=True):
        apis = fixture.APIDefinitionFixture.all_api_definitions_fixture()
        apis.backup_global_resources = global_cleanup
        apis.setUp()

        asserteq = self.assertNotEqual if global_cleanup else self.assertEqual
        asserteq({}, apis._orig_resources)

        for r in test_attributes.TestCoreResources.CORE_DEFS:
            attributes.RESOURCES[r.COLLECTION_NAME]['_test_'] = {}
            r.RESOURCE_ATTRIBUTE_MAP['_test_'] = {}

        apis.cleanUp()
        for r in test_attributes.TestCoreResources.CORE_DEFS:
            self.assertNotIn('_test_', r.RESOURCE_ATTRIBUTE_MAP)
            global_assert = (self.assertNotIn
                             if global_cleanup else self.assertIn)
            global_assert('_test_', attributes.RESOURCES[r.COLLECTION_NAME])
            # cleanup
            if not global_cleanup:
                del attributes.RESOURCES[r.COLLECTION_NAME]['_test_']

    def test_all_api_definitions_fixture_no_global_backup(self):
        self._test_all_api_definitions_fixture(global_cleanup=False)

    def test_all_api_definitions_fixture_with_global_backup(self):
        self._test_all_api_definitions_fixture(global_cleanup=True)
