# bash completion for kubectl                              -*- shell-script -*-

__debug()
{
    if [[ -n ${BASH_COMP_DEBUG_FILE} ]]; then
        echo "$*" >> "${BASH_COMP_DEBUG_FILE}"
    fi
}

# Homebrew on Macs have version 1.3 of bash-completion which doesn't include
# _init_completion. This is a very minimal version of that function.
__my_init_completion()
{
    COMPREPLY=()
    _get_comp_words_by_ref "$@" cur prev words cword
}

__index_of_word()
{
    local w word=$1
    shift
    index=0
    for w in "$@"; do
        [[ $w = "$word" ]] && return
        index=$((index+1))
    done
    index=-1
}

__contains_word()
{
    local w word=$1; shift
    for w in "$@"; do
        [[ $w = "$word" ]] && return
    done
    return 1
}

__handle_reply()
{
    __debug "${FUNCNAME[0]}"
    case $cur in
        -*)
            if [[ $(type -t compopt) = "builtin" ]]; then
                compopt -o nospace
            fi
            local allflags
            if [ ${#must_have_one_flag[@]} -ne 0 ]; then
                allflags=("${must_have_one_flag[@]}")
            else
                allflags=("${flags[*]} ${two_word_flags[*]}")
            fi
            COMPREPLY=( $(compgen -W "${allflags[*]}" -- "$cur") )
            if [[ $(type -t compopt) = "builtin" ]]; then
                [[ "${COMPREPLY[0]}" == *= ]] || compopt +o nospace
            fi

            # complete after --flag=abc
            if [[ $cur == *=* ]]; then
                if [[ $(type -t compopt) = "builtin" ]]; then
                    compopt +o nospace
                fi

                local index flag
                flag="${cur%%=*}"
                __index_of_word "${flag}" "${flags_with_completion[@]}"
                if [[ ${index} -ge 0 ]]; then
                    COMPREPLY=()
                    PREFIX=""
                    cur="${cur#*=}"
                    ${flags_completion[${index}]}
                    if [ -n "${ZSH_VERSION}" ]; then
                        # zfs completion needs --flag= prefix
                        eval "COMPREPLY=( \"\${COMPREPLY[@]/#/${flag}=}\" )"
                    fi
                fi
            fi
            return 0;
            ;;
    esac

    # check if we are handling a flag with special work handling
    local index
    __index_of_word "${prev}" "${flags_with_completion[@]}"
    if [[ ${index} -ge 0 ]]; then
        ${flags_completion[${index}]}
        return
    fi

    # we are parsing a flag and don't have a special handler, no completion
    if [[ ${cur} != "${words[cword]}" ]]; then
        return
    fi

    local completions
    completions=("${commands[@]}")
    if [[ ${#must_have_one_noun[@]} -ne 0 ]]; then
        completions=("${must_have_one_noun[@]}")
    fi
    if [[ ${#must_have_one_flag[@]} -ne 0 ]]; then
        completions+=("${must_have_one_flag[@]}")
    fi
    COMPREPLY=( $(compgen -W "${completions[*]}" -- "$cur") )

    if [[ ${#COMPREPLY[@]} -eq 0 && ${#noun_aliases[@]} -gt 0 && ${#must_have_one_noun[@]} -ne 0 ]]; then
        COMPREPLY=( $(compgen -W "${noun_aliases[*]}" -- "$cur") )
    fi

    if [[ ${#COMPREPLY[@]} -eq 0 ]]; then
        declare -F __custom_func >/dev/null && __custom_func
    fi

    __ltrim_colon_completions "$cur"
}

# The arguments should be in the form "ext1|ext2|extn"
__handle_filename_extension_flag()
{
    local ext="$1"
    _filedir "@(${ext})"
}

__handle_subdirs_in_dir_flag()
{
    local dir="$1"
    pushd "${dir}" >/dev/null 2>&1 && _filedir -d && popd >/dev/null 2>&1
}

__handle_flag()
{
    __debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    # if a command required a flag, and we found it, unset must_have_one_flag()
    local flagname=${words[c]}
    local flagvalue
    # if the word contained an =
    if [[ ${words[c]} == *"="* ]]; then
        flagvalue=${flagname#*=} # take in as flagvalue after the =
        flagname=${flagname%%=*} # strip everything after the =
        flagname="${flagname}=" # but put the = back
    fi
    __debug "${FUNCNAME[0]}: looking for ${flagname}"
    if __contains_word "${flagname}" "${must_have_one_flag[@]}"; then
        must_have_one_flag=()
    fi

    # if you set a flag which only applies to this command, don't show subcommands
    if __contains_word "${flagname}" "${local_nonpersistent_flags[@]}"; then
      commands=()
    fi

    # keep flag value with flagname as flaghash
    if [ -n "${flagvalue}" ] ; then
        flaghash[${flagname}]=${flagvalue}
    elif [ -n "${words[ $((c+1)) ]}" ] ; then
        flaghash[${flagname}]=${words[ $((c+1)) ]}
    else
        flaghash[${flagname}]="true" # pad "true" for bool flag
    fi

    # skip the argument to a two word flag
    if __contains_word "${words[c]}" "${two_word_flags[@]}"; then
        c=$((c+1))
        # if we are looking for a flags value, don't show commands
        if [[ $c -eq $cword ]]; then
            commands=()
        fi
    fi

    c=$((c+1))

}

__handle_noun()
{
    __debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    if __contains_word "${words[c]}" "${must_have_one_noun[@]}"; then
        must_have_one_noun=()
    elif __contains_word "${words[c]}" "${noun_aliases[@]}"; then
        must_have_one_noun=()
    fi

    nouns+=("${words[c]}")
    c=$((c+1))
}

__handle_command()
{
    __debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"

    local next_command
    if [[ -n ${last_command} ]]; then
        next_command="_${last_command}_${words[c]//:/__}"
    else
        if [[ $c -eq 0 ]]; then
            next_command="_$(basename "${words[c]//:/__}")"
        else
            next_command="_${words[c]//:/__}"
        fi
    fi
    c=$((c+1))
    __debug "${FUNCNAME[0]}: looking for ${next_command}"
    declare -F $next_command >/dev/null && $next_command
}

__handle_word()
{
    if [[ $c -ge $cword ]]; then
        __handle_reply
        return
    fi
    __debug "${FUNCNAME[0]}: c is $c words[c] is ${words[c]}"
    if [[ "${words[c]}" == -* ]]; then
        __handle_flag
    elif __contains_word "${words[c]}" "${commands[@]}"; then
        __handle_command
    elif [[ $c -eq 0 ]] && __contains_word "$(basename "${words[c]}")" "${commands[@]}"; then
        __handle_command
    else
        __handle_noun
    fi
    __handle_word
}

# call kubectl get $1,
__kubectl_override_flag_list=(kubeconfig cluster user context namespace server)
__kubectl_override_flags()
{
    local ${__kubectl_override_flag_list[*]} two_word_of of
    for w in "${words[@]}"; do
        if [ -n "${two_word_of}" ]; then
            eval "${two_word_of}=\"--${two_word_of}=\${w}\""
            two_word_of=
            continue
        fi
        for of in "${__kubectl_override_flag_list[@]}"; do
            case "${w}" in
                --${of}=*)
                    eval "${of}=\"${w}\""
                    ;;
                --${of})
                    two_word_of="${of}"
                    ;;
            esac
        done
        if [ "${w}" == "--all-namespaces" ]; then
            namespace="--all-namespaces"
        fi
    done
    for of in "${__kubectl_override_flag_list[@]}"; do
        if eval "test -n \"\$${of}\""; then
            eval "echo \${${of}}"
        fi
    done
}

__kubectl_get_namespaces()
{
    local template kubectl_out
    template="{# range .items  #}{# .metadata.name #} {# end #}"
    if kubectl_out=$(kubectl get -o template --template="${template}" namespace 2>/dev/null); then
        COMPREPLY=( $( compgen -W "${kubectl_out[*]}" -- "$cur" ) )
    fi
}

__kubectl_parse_get()
{
    local template
    template="{# range .items  #}{# .metadata.name #} {# end #}"
    local kubectl_out
    if kubectl_out=$(kubectl get $(__kubectl_override_flags) -o template --template="${template}" "$1" 2>/dev/null); then
        COMPREPLY=( $( compgen -W "${kubectl_out[*]}" -- "$cur" ) )
    fi
}

__kubectl_get_resource()
{
    if [[ ${#nouns[@]} -eq 0 ]]; then
        return 1
    fi
    __kubectl_parse_get "${nouns[${#nouns[@]} -1]}"
}

__kubectl_get_resource_pod()
{
    __kubectl_parse_get "pod"
}

__kubectl_get_resource_rc()
{
    __kubectl_parse_get "rc"
}

__kubectl_get_resource_node()
{
    __kubectl_parse_get "node"
}

# $1 is the name of the pod we want to get the list of containers inside
__kubectl_get_containers()
{
    local template
    template="{# range .spec.containers  #}{# .name #} {# end #}"
    __debug "${FUNCNAME} nouns are ${nouns[*]}"

    local len="${#nouns[@]}"
    if [[ ${len} -ne 1 ]]; then
        return
    fi
    local last=${nouns[${len} -1]}
    local kubectl_out
    if kubectl_out=$(kubectl get $(__kubectl_override_flags) -o template --template="${template}" pods "${last}" 2>/dev/null); then
        COMPREPLY=( $( compgen -W "${kubectl_out[*]}" -- "$cur" ) )
    fi
}

# Require both a pod and a container to be specified
__kubectl_require_pod_and_container()
{
    if [[ ${#nouns[@]} -eq 0 ]]; then
        __kubectl_parse_get pods
        return 0
    fi;
    __kubectl_get_containers
    return 0
}

__custom_func() {
    case ${last_command} in
        kubectl_get | kubectl_describe | kubectl_delete | kubectl_label | kubectl_stop | kubectl_edit | kubectl_patch |\
        kubectl_annotate | kubectl_expose | kubectl_scale | kubectl_autoscale | kubectl_taint | kubectl_rollout_*)
            __kubectl_get_resource
            return
            ;;
        kubectl_logs | kubectl_attach)
            __kubectl_require_pod_and_container
            return
            ;;
        kubectl_exec | kubectl_port-forward | kubectl_top_pod)
            __kubectl_get_resource_pod
            return
            ;;
        kubectl_rolling-update)
            __kubectl_get_resource_rc
            return
            ;;
        kubectl_cordon | kubectl_uncordon | kubectl_drain | kubectl_top_node)
            __kubectl_get_resource_node
            return
            ;;
        *)
            ;;
    esac
}

_kubectl_annotate()
{
    last_command="kubectl_annotate"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("cluster")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("storageclass")
    must_have_one_noun+=("thirdpartyresource")
    must_have_one_noun+=("thirdpartyresourcedata")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("clusters")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
    noun_aliases+=("thirdpartyresourcedatas")
    noun_aliases+=("thirdpartyresources")
}

_kubectl_api-versions()
{
    last_command="kubectl_api-versions"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_apply()
{
    last_command="kubectl_apply"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_attach()
{
    last_command="kubectl_attach"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_autoscale()
{
    last_command="kubectl_autoscale"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cpu-percent=")
    local_nonpersistent_flags+=("--cpu-percent=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--max=")
    local_nonpersistent_flags+=("--max=")
    flags+=("--min=")
    local_nonpersistent_flags+=("--min=")
    flags+=("--name=")
    local_nonpersistent_flags+=("--name=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--max=")
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rs")
}

_kubectl_cluster-info_dump()
{
    last_command="kubectl_cluster-info_dump"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--namespaces=")
    local_nonpersistent_flags+=("--namespaces=")
    flags+=("--output-directory=")
    local_nonpersistent_flags+=("--output-directory=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cluster-info()
{
    last_command="kubectl_cluster-info"
    commands=()
    commands+=("dump")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_completion()
{
    last_command="kubectl_completion"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("bash")
    must_have_one_noun+=("zsh")
    noun_aliases=()
}

_kubectl_config_current-context()
{
    last_command="kubectl_config_current-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_delete-cluster()
{
    last_command="kubectl_config_delete-cluster"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_delete-context()
{
    last_command="kubectl_config_delete-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-clusters()
{
    last_command="kubectl_config_get-clusters"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_get-contexts()
{
    last_command="kubectl_config_get-contexts"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set()
{
    last_command="kubectl_config_set"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--set-raw-bytes")
    local_nonpersistent_flags+=("--set-raw-bytes")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-cluster()
{
    last_command="kubectl_config_set-cluster"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--api-version=")
    local_nonpersistent_flags+=("--api-version=")
    flags+=("--certificate-authority=")
    flags_with_completion+=("--certificate-authority")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--certificate-authority=")
    flags+=("--embed-certs")
    local_nonpersistent_flags+=("--embed-certs")
    flags+=("--insecure-skip-tls-verify")
    local_nonpersistent_flags+=("--insecure-skip-tls-verify")
    flags+=("--server=")
    local_nonpersistent_flags+=("--server=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-context()
{
    last_command="kubectl_config_set-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cluster=")
    local_nonpersistent_flags+=("--cluster=")
    flags+=("--namespace=")
    local_nonpersistent_flags+=("--namespace=")
    flags+=("--user=")
    local_nonpersistent_flags+=("--user=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_set-credentials()
{
    last_command="kubectl_config_set-credentials"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--auth-provider=")
    local_nonpersistent_flags+=("--auth-provider=")
    flags+=("--auth-provider-arg=")
    local_nonpersistent_flags+=("--auth-provider-arg=")
    flags+=("--client-certificate=")
    flags_with_completion+=("--client-certificate")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags_with_completion+=("--client-key")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--client-key=")
    flags+=("--embed-certs")
    local_nonpersistent_flags+=("--embed-certs")
    flags+=("--password=")
    local_nonpersistent_flags+=("--password=")
    flags+=("--token=")
    local_nonpersistent_flags+=("--token=")
    flags+=("--username=")
    local_nonpersistent_flags+=("--username=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--user=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_unset()
{
    last_command="kubectl_config_unset"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_use-context()
{
    last_command="kubectl_config_use-context"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config_view()
{
    last_command="kubectl_config_view"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--flatten")
    local_nonpersistent_flags+=("--flatten")
    flags+=("--merge")
    local_nonpersistent_flags+=("--merge")
    flags+=("--minify")
    local_nonpersistent_flags+=("--minify")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--raw")
    local_nonpersistent_flags+=("--raw")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_config()
{
    last_command="kubectl_config"
    commands=()
    commands+=("current-context")
    commands+=("delete-cluster")
    commands+=("delete-context")
    commands+=("get-clusters")
    commands+=("get-contexts")
    commands+=("set")
    commands+=("set-cluster")
    commands+=("set-context")
    commands+=("set-credentials")
    commands+=("unset")
    commands+=("use-context")
    commands+=("view")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--kubeconfig=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_convert()
{
    last_command="kubectl_convert"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_cordon()
{
    last_command="kubectl_cordon"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_configmap()
{
    last_command="kubectl_create_configmap"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--from-file=")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--from-literal=")
    local_nonpersistent_flags+=("--from-literal=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_deployment()
{
    last_command="kubectl_create_deployment"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--image=")
    local_nonpersistent_flags+=("--image=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_namespace()
{
    last_command="kubectl_create_namespace"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_quota()
{
    last_command="kubectl_create_quota"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--hard=")
    local_nonpersistent_flags+=("--hard=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--scopes=")
    local_nonpersistent_flags+=("--scopes=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_docker-registry()
{
    last_command="kubectl_create_secret_docker-registry"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--docker-email=")
    local_nonpersistent_flags+=("--docker-email=")
    flags+=("--docker-password=")
    local_nonpersistent_flags+=("--docker-password=")
    flags+=("--docker-server=")
    local_nonpersistent_flags+=("--docker-server=")
    flags+=("--docker-username=")
    local_nonpersistent_flags+=("--docker-username=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--docker-email=")
    must_have_one_flag+=("--docker-password=")
    must_have_one_flag+=("--docker-username=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_generic()
{
    last_command="kubectl_create_secret_generic"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--from-file=")
    local_nonpersistent_flags+=("--from-file=")
    flags+=("--from-literal=")
    local_nonpersistent_flags+=("--from-literal=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    local_nonpersistent_flags+=("--type=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret_tls()
{
    last_command="kubectl_create_secret_tls"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cert=")
    local_nonpersistent_flags+=("--cert=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--key=")
    local_nonpersistent_flags+=("--key=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_secret()
{
    last_command="kubectl_create_secret"
    commands=()
    commands+=("docker-registry")
    commands+=("generic")
    commands+=("tls")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_clusterip()
{
    last_command="kubectl_create_service_clusterip"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--clusterip=")
    local_nonpersistent_flags+=("--clusterip=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_loadbalancer()
{
    last_command="kubectl_create_service_loadbalancer"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service_nodeport()
{
    last_command="kubectl_create_service_nodeport"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--tcp=")
    local_nonpersistent_flags+=("--tcp=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_service()
{
    last_command="kubectl_create_service"
    commands=()
    commands+=("clusterip")
    commands+=("loadbalancer")
    commands+=("nodeport")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create_serviceaccount()
{
    last_command="kubectl_create_serviceaccount"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_create()
{
    last_command="kubectl_create"
    commands=()
    commands+=("configmap")
    commands+=("deployment")
    commands+=("namespace")
    commands+=("quota")
    commands+=("secret")
    commands+=("service")
    commands+=("serviceaccount")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_delete()
{
    last_command="kubectl_delete"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--ignore-not-found")
    local_nonpersistent_flags+=("--ignore-not-found")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--now")
    local_nonpersistent_flags+=("--now")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("cluster")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("storageclass")
    must_have_one_noun+=("thirdpartyresource")
    must_have_one_noun+=("thirdpartyresourcedata")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("clusters")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
    noun_aliases+=("thirdpartyresourcedatas")
    noun_aliases+=("thirdpartyresources")
}

_kubectl_describe()
{
    last_command="kubectl_describe"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-events")
    local_nonpersistent_flags+=("--show-events")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("cm")
    noun_aliases+=("configmaps")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("svc")
}

_kubectl_drain()
{
    last_command="kubectl_drain"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--delete-local-data")
    local_nonpersistent_flags+=("--delete-local-data")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--ignore-daemonsets")
    local_nonpersistent_flags+=("--ignore-daemonsets")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_edit()
{
    last_command="kubectl_edit"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--windows-line-endings")
    local_nonpersistent_flags+=("--windows-line-endings")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("cluster")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("storageclass")
    must_have_one_noun+=("thirdpartyresource")
    must_have_one_noun+=("thirdpartyresourcedata")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("clusters")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
    noun_aliases+=("thirdpartyresourcedatas")
    noun_aliases+=("thirdpartyresources")
}

_kubectl_exec()
{
    last_command="kubectl_exec"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--pod=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--pod=")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_explain()
{
    last_command="kubectl_explain"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--recursive")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_expose()
{
    last_command="kubectl_expose"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cluster-ip=")
    local_nonpersistent_flags+=("--cluster-ip=")
    flags+=("--container-port=")
    local_nonpersistent_flags+=("--container-port=")
    flags+=("--create-external-load-balancer")
    local_nonpersistent_flags+=("--create-external-load-balancer")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--external-ip=")
    local_nonpersistent_flags+=("--external-ip=")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--labels=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--labels=")
    flags+=("--load-balancer-ip=")
    local_nonpersistent_flags+=("--load-balancer-ip=")
    flags+=("--name=")
    local_nonpersistent_flags+=("--name=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--overrides=")
    local_nonpersistent_flags+=("--overrides=")
    flags+=("--port=")
    local_nonpersistent_flags+=("--port=")
    flags+=("--protocol=")
    local_nonpersistent_flags+=("--protocol=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--selector=")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--session-affinity=")
    local_nonpersistent_flags+=("--session-affinity=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--target-port=")
    local_nonpersistent_flags+=("--target-port=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    local_nonpersistent_flags+=("--type=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("service")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rs")
    noun_aliases+=("services")
    noun_aliases+=("svc")
}

_kubectl_get()
{
    last_command="kubectl_get"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--export")
    local_nonpersistent_flags+=("--export")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--label-columns=")
    two_word_flags+=("-L")
    local_nonpersistent_flags+=("--label-columns=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--raw=")
    local_nonpersistent_flags+=("--raw=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-kind")
    local_nonpersistent_flags+=("--show-kind")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--watch")
    flags+=("-w")
    local_nonpersistent_flags+=("--watch")
    flags+=("--watch-only")
    local_nonpersistent_flags+=("--watch-only")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("cluster")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("storageclass")
    must_have_one_noun+=("thirdpartyresource")
    must_have_one_noun+=("thirdpartyresourcedata")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("clusters")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
    noun_aliases+=("thirdpartyresourcedatas")
    noun_aliases+=("thirdpartyresources")
}

_kubectl_label()
{
    last_command="kubectl_label"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("cluster")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("storageclass")
    must_have_one_noun+=("thirdpartyresource")
    must_have_one_noun+=("thirdpartyresourcedata")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("clusters")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
    noun_aliases+=("thirdpartyresourcedatas")
    noun_aliases+=("thirdpartyresources")
}

_kubectl_logs()
{
    last_command="kubectl_logs"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    two_word_flags+=("-c")
    local_nonpersistent_flags+=("--container=")
    flags+=("--follow")
    flags+=("-f")
    local_nonpersistent_flags+=("--follow")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--interactive")
    local_nonpersistent_flags+=("--interactive")
    flags+=("--limit-bytes=")
    local_nonpersistent_flags+=("--limit-bytes=")
    flags+=("--previous")
    flags+=("-p")
    local_nonpersistent_flags+=("--previous")
    flags+=("--since=")
    local_nonpersistent_flags+=("--since=")
    flags+=("--since-time=")
    local_nonpersistent_flags+=("--since-time=")
    flags+=("--tail=")
    local_nonpersistent_flags+=("--tail=")
    flags+=("--timestamps")
    local_nonpersistent_flags+=("--timestamps")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_namespace()
{
    last_command="kubectl_namespace"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_options()
{
    last_command="kubectl_options"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_patch()
{
    last_command="kubectl_patch"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--patch=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--patch=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--type=")
    local_nonpersistent_flags+=("--type=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--patch=")
    must_have_one_flag+=("-p")
    must_have_one_noun=()
    must_have_one_noun+=("certificatesigningrequest")
    must_have_one_noun+=("cluster")
    must_have_one_noun+=("clusterrole")
    must_have_one_noun+=("clusterrolebinding")
    must_have_one_noun+=("componentstatus")
    must_have_one_noun+=("configmap")
    must_have_one_noun+=("daemonset")
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("endpoints")
    must_have_one_noun+=("event")
    must_have_one_noun+=("horizontalpodautoscaler")
    must_have_one_noun+=("ingress")
    must_have_one_noun+=("job")
    must_have_one_noun+=("limitrange")
    must_have_one_noun+=("namespace")
    must_have_one_noun+=("networkpolicy")
    must_have_one_noun+=("node")
    must_have_one_noun+=("persistentvolume")
    must_have_one_noun+=("persistentvolumeclaim")
    must_have_one_noun+=("petset")
    must_have_one_noun+=("pod")
    must_have_one_noun+=("podsecuritypolicy")
    must_have_one_noun+=("podtemplate")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    must_have_one_noun+=("resourcequota")
    must_have_one_noun+=("role")
    must_have_one_noun+=("rolebinding")
    must_have_one_noun+=("scheduledjob")
    must_have_one_noun+=("secret")
    must_have_one_noun+=("service")
    must_have_one_noun+=("serviceaccount")
    must_have_one_noun+=("storageclass")
    must_have_one_noun+=("thirdpartyresource")
    must_have_one_noun+=("thirdpartyresourcedata")
    noun_aliases=()
    noun_aliases+=("certificatesigningrequests")
    noun_aliases+=("clusterrolebindings")
    noun_aliases+=("clusterroles")
    noun_aliases+=("clusters")
    noun_aliases+=("cm")
    noun_aliases+=("componentstatuses")
    noun_aliases+=("configmaps")
    noun_aliases+=("cs")
    noun_aliases+=("csr")
    noun_aliases+=("daemonsets")
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("ds")
    noun_aliases+=("endpoints")
    noun_aliases+=("ep")
    noun_aliases+=("ev")
    noun_aliases+=("events")
    noun_aliases+=("horizontalpodautoscalers")
    noun_aliases+=("hpa")
    noun_aliases+=("ing")
    noun_aliases+=("ingresses")
    noun_aliases+=("jobs")
    noun_aliases+=("limitranges")
    noun_aliases+=("limits")
    noun_aliases+=("namespaces")
    noun_aliases+=("networkpolicies")
    noun_aliases+=("no")
    noun_aliases+=("nodes")
    noun_aliases+=("ns")
    noun_aliases+=("persistentvolumeclaims")
    noun_aliases+=("persistentvolumes")
    noun_aliases+=("petsets")
    noun_aliases+=("po")
    noun_aliases+=("pods")
    noun_aliases+=("podsecuritypolicies")
    noun_aliases+=("podtemplates")
    noun_aliases+=("pv")
    noun_aliases+=("pvc")
    noun_aliases+=("quota")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("resourcequotas")
    noun_aliases+=("rolebindings")
    noun_aliases+=("roles")
    noun_aliases+=("rs")
    noun_aliases+=("sa")
    noun_aliases+=("scheduledjobs")
    noun_aliases+=("secrets")
    noun_aliases+=("serviceaccounts")
    noun_aliases+=("services")
    noun_aliases+=("storageclasses")
    noun_aliases+=("svc")
    noun_aliases+=("thirdpartyresourcedatas")
    noun_aliases+=("thirdpartyresources")
}

_kubectl_port-forward()
{
    last_command="kubectl_port-forward"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--pod=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--pod=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_proxy()
{
    last_command="kubectl_proxy"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--accept-hosts=")
    local_nonpersistent_flags+=("--accept-hosts=")
    flags+=("--accept-paths=")
    local_nonpersistent_flags+=("--accept-paths=")
    flags+=("--address=")
    local_nonpersistent_flags+=("--address=")
    flags+=("--api-prefix=")
    local_nonpersistent_flags+=("--api-prefix=")
    flags+=("--disable-filter")
    local_nonpersistent_flags+=("--disable-filter")
    flags+=("--port=")
    two_word_flags+=("-p")
    local_nonpersistent_flags+=("--port=")
    flags+=("--reject-methods=")
    local_nonpersistent_flags+=("--reject-methods=")
    flags+=("--reject-paths=")
    local_nonpersistent_flags+=("--reject-paths=")
    flags+=("--unix-socket=")
    two_word_flags+=("-u")
    local_nonpersistent_flags+=("--unix-socket=")
    flags+=("--www=")
    two_word_flags+=("-w")
    local_nonpersistent_flags+=("--www=")
    flags+=("--www-prefix=")
    two_word_flags+=("-P")
    local_nonpersistent_flags+=("--www-prefix=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_replace()
{
    last_command="kubectl_replace"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--cascade")
    local_nonpersistent_flags+=("--cascade")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--force")
    local_nonpersistent_flags+=("--force")
    flags+=("--grace-period=")
    local_nonpersistent_flags+=("--grace-period=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_rolling-update()
{
    last_command="kubectl_rolling-update"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--container=")
    local_nonpersistent_flags+=("--container=")
    flags+=("--deployment-label-key=")
    local_nonpersistent_flags+=("--deployment-label-key=")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--image=")
    local_nonpersistent_flags+=("--image=")
    flags+=("--image-pull-policy=")
    local_nonpersistent_flags+=("--image-pull-policy=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--poll-interval=")
    local_nonpersistent_flags+=("--poll-interval=")
    flags+=("--rollback")
    local_nonpersistent_flags+=("--rollback")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--update-period=")
    local_nonpersistent_flags+=("--update-period=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--filename=")
    must_have_one_flag+=("-f")
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_rollout_history()
{
    last_command="kubectl_rollout_history"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--revision=")
    local_nonpersistent_flags+=("--revision=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout_pause()
{
    last_command="kubectl_rollout_pause"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout_resume()
{
    last_command="kubectl_rollout_resume"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout_status()
{
    last_command="kubectl_rollout_status"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout_undo()
{
    last_command="kubectl_rollout_undo"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--to-revision=")
    local_nonpersistent_flags+=("--to-revision=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
}

_kubectl_rollout()
{
    last_command="kubectl_rollout"
    commands=()
    commands+=("history")
    commands+=("pause")
    commands+=("resume")
    commands+=("status")
    commands+=("undo")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_run()
{
    last_command="kubectl_run"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--attach")
    local_nonpersistent_flags+=("--attach")
    flags+=("--command")
    local_nonpersistent_flags+=("--command")
    flags+=("--dry-run")
    local_nonpersistent_flags+=("--dry-run")
    flags+=("--env=")
    local_nonpersistent_flags+=("--env=")
    flags+=("--expose")
    local_nonpersistent_flags+=("--expose")
    flags+=("--generator=")
    local_nonpersistent_flags+=("--generator=")
    flags+=("--hostport=")
    local_nonpersistent_flags+=("--hostport=")
    flags+=("--image=")
    local_nonpersistent_flags+=("--image=")
    flags+=("--image-pull-policy=")
    local_nonpersistent_flags+=("--image-pull-policy=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--labels=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--labels=")
    flags+=("--leave-stdin-open")
    local_nonpersistent_flags+=("--leave-stdin-open")
    flags+=("--limits=")
    local_nonpersistent_flags+=("--limits=")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--overrides=")
    local_nonpersistent_flags+=("--overrides=")
    flags+=("--port=")
    local_nonpersistent_flags+=("--port=")
    flags+=("--quiet")
    local_nonpersistent_flags+=("--quiet")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--replicas=")
    two_word_flags+=("-r")
    local_nonpersistent_flags+=("--replicas=")
    flags+=("--requests=")
    local_nonpersistent_flags+=("--requests=")
    flags+=("--restart=")
    local_nonpersistent_flags+=("--restart=")
    flags+=("--rm")
    local_nonpersistent_flags+=("--rm")
    flags+=("--save-config")
    local_nonpersistent_flags+=("--save-config")
    flags+=("--schedule=")
    local_nonpersistent_flags+=("--schedule=")
    flags+=("--service-generator=")
    local_nonpersistent_flags+=("--service-generator=")
    flags+=("--service-overrides=")
    local_nonpersistent_flags+=("--service-overrides=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--stdin")
    flags+=("-i")
    local_nonpersistent_flags+=("--stdin")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--tty")
    flags+=("-t")
    local_nonpersistent_flags+=("--tty")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--image=")
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_scale()
{
    last_command="kubectl_scale"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--current-replicas=")
    local_nonpersistent_flags+=("--current-replicas=")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--replicas=")
    local_nonpersistent_flags+=("--replicas=")
    flags+=("--resource-version=")
    local_nonpersistent_flags+=("--resource-version=")
    flags+=("--timeout=")
    local_nonpersistent_flags+=("--timeout=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_flag+=("--replicas=")
    must_have_one_noun=()
    must_have_one_noun+=("deployment")
    must_have_one_noun+=("job")
    must_have_one_noun+=("replicaset")
    must_have_one_noun+=("replicationcontroller")
    noun_aliases=()
    noun_aliases+=("deploy")
    noun_aliases+=("deployments")
    noun_aliases+=("jobs")
    noun_aliases+=("rc")
    noun_aliases+=("replicasets")
    noun_aliases+=("replicationcontrollers")
    noun_aliases+=("rs")
}

_kubectl_set_image()
{
    last_command="kubectl_set_image"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--filename=")
    flags_with_completion+=("--filename")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    two_word_flags+=("-f")
    flags_with_completion+=("-f")
    flags_completion+=("__handle_filename_extension_flag json|yaml|yml")
    local_nonpersistent_flags+=("--filename=")
    flags+=("--local")
    local_nonpersistent_flags+=("--local")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--record")
    local_nonpersistent_flags+=("--record")
    flags+=("--recursive")
    flags+=("-R")
    local_nonpersistent_flags+=("--recursive")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_set()
{
    last_command="kubectl_set"
    commands=()
    commands+=("image")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_taint()
{
    last_command="kubectl_taint"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all")
    local_nonpersistent_flags+=("--all")
    flags+=("--include-extended-apis")
    local_nonpersistent_flags+=("--include-extended-apis")
    flags+=("--no-headers")
    local_nonpersistent_flags+=("--no-headers")
    flags+=("--output=")
    two_word_flags+=("-o")
    local_nonpersistent_flags+=("--output=")
    flags+=("--output-version=")
    local_nonpersistent_flags+=("--output-version=")
    flags+=("--overwrite")
    local_nonpersistent_flags+=("--overwrite")
    flags+=("--schema-cache-dir=")
    flags_with_completion+=("--schema-cache-dir")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--schema-cache-dir=")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--show-all")
    flags+=("-a")
    local_nonpersistent_flags+=("--show-all")
    flags+=("--show-labels")
    local_nonpersistent_flags+=("--show-labels")
    flags+=("--sort-by=")
    local_nonpersistent_flags+=("--sort-by=")
    flags+=("--template=")
    flags_with_completion+=("--template")
    flags_completion+=("_filedir")
    local_nonpersistent_flags+=("--template=")
    flags+=("--validate")
    local_nonpersistent_flags+=("--validate")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    must_have_one_noun+=("node")
    noun_aliases=()
    noun_aliases+=("no")
    noun_aliases+=("nodes")
}

_kubectl_top_node()
{
    last_command="kubectl_top_node"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_top_pod()
{
    last_command="kubectl_top_pod"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--all-namespaces")
    local_nonpersistent_flags+=("--all-namespaces")
    flags+=("--containers")
    local_nonpersistent_flags+=("--containers")
    flags+=("--selector=")
    two_word_flags+=("-l")
    local_nonpersistent_flags+=("--selector=")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_top()
{
    last_command="kubectl_top"
    commands=()
    commands+=("node")
    commands+=("pod")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_uncordon()
{
    last_command="kubectl_uncordon"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl_version()
{
    last_command="kubectl_version"
    commands=()

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--client")
    flags+=("-c")
    local_nonpersistent_flags+=("--client")
    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

_kubectl()
{
    last_command="kubectl"
    commands=()
    commands+=("annotate")
    commands+=("api-versions")
    commands+=("apply")
    commands+=("attach")
    commands+=("autoscale")
    commands+=("cluster-info")
    commands+=("completion")
    commands+=("config")
    commands+=("convert")
    commands+=("cordon")
    commands+=("create")
    commands+=("delete")
    commands+=("describe")
    commands+=("drain")
    commands+=("edit")
    commands+=("exec")
    commands+=("explain")
    commands+=("expose")
    commands+=("get")
    commands+=("label")
    commands+=("logs")
    commands+=("namespace")
    commands+=("options")
    commands+=("patch")
    commands+=("port-forward")
    commands+=("proxy")
    commands+=("replace")
    commands+=("rolling-update")
    commands+=("rollout")
    commands+=("run")
    commands+=("scale")
    commands+=("set")
    commands+=("taint")
    commands+=("top")
    commands+=("uncordon")
    commands+=("version")

    flags=()
    two_word_flags=()
    local_nonpersistent_flags=()
    flags_with_completion=()
    flags_completion=()

    flags+=("--allow-verification-with-non-compliant-keys")
    flags+=("--alsologtostderr")
    flags+=("--api-version=")
    flags+=("--application-metrics-count-limit=")
    flags+=("--as=")
    flags+=("--boot-id-file=")
    flags+=("--certificate-authority=")
    flags+=("--client-certificate=")
    flags+=("--client-key=")
    flags+=("--cluster=")
    flags+=("--container-hints=")
    flags+=("--context=")
    flags+=("--docker=")
    flags+=("--docker-env-metadata-whitelist=")
    flags+=("--docker-only")
    flags+=("--docker-root=")
    flags+=("--enable-load-reader")
    flags+=("--event-storage-age-limit=")
    flags+=("--event-storage-event-limit=")
    flags+=("--global-housekeeping-interval=")
    flags+=("--google-json-key=")
    flags+=("--housekeeping-interval=")
    flags+=("--insecure-skip-tls-verify")
    flags+=("--ir-data-source=")
    flags+=("--ir-dbname=")
    flags+=("--ir-hawkular=")
    flags+=("--ir-influxdb-host=")
    flags+=("--ir-namespace-only")
    flags+=("--ir-password=")
    flags+=("--ir-percentile=")
    flags+=("--ir-user=")
    flags+=("--kubeconfig=")
    flags+=("--log-backtrace-at=")
    flags+=("--log-cadvisor-usage")
    flags+=("--log-dir=")
    flags+=("--log-flush-frequency=")
    flags+=("--loglevel=")
    flags+=("--logtostderr")
    flags+=("--machine-id-file=")
    flags+=("--match-server-version")
    flags+=("--namespace=")
    flags_with_completion+=("--namespace")
    flags_completion+=("__kubectl_get_namespaces")
    two_word_flags+=("-n")
    flags_with_completion+=("-n")
    flags_completion+=("__kubectl_get_namespaces")
    flags+=("--password=")
    flags+=("--server=")
    two_word_flags+=("-s")
    flags+=("--stderrthreshold=")
    flags+=("--storage-driver-buffer-duration=")
    flags+=("--storage-driver-db=")
    flags+=("--storage-driver-host=")
    flags+=("--storage-driver-password=")
    flags+=("--storage-driver-secure")
    flags+=("--storage-driver-table=")
    flags+=("--storage-driver-user=")
    flags+=("--token=")
    flags+=("--user=")
    flags+=("--username=")
    flags+=("--v=")
    two_word_flags+=("-v")
    flags+=("--version")
    flags+=("--vmodule=")

    must_have_one_flag=()
    must_have_one_noun=()
    noun_aliases=()
}

__start_kubectl()
{
    local cur prev words cword
    declare -A flaghash 2>/dev/null || :
    if declare -F _init_completion >/dev/null 2>&1; then
        _init_completion -s || return
    else
        __my_init_completion -n "=" || return
    fi

    local c=0
    local flags=()
    local two_word_flags=()
    local local_nonpersistent_flags=()
    local flags_with_completion=()
    local flags_completion=()
    local commands=("kubectl")
    local must_have_one_flag=()
    local must_have_one_noun=()
    local last_command
    local nouns=()

    __handle_word
}

if [[ $(type -t compopt) = "builtin" ]]; then
    complete -o default -F __start_kubectl kubectl
else
    complete -o default -o nospace -F __start_kubectl kubectl
fi

# ex: ts=4 sw=4 et filetype=sh