#
# Copyright (C) 2015 eNovance SAS <licensing@enovance.com>
#
# Author: Emilien Macchi <emilien.macchi@enovance.com>
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# Class to serve aodh API with apache mod_wsgi in place of aodh-api service.
#
# Serving aodh API from apache is the recommended way to go for production
# because of limited performance for concurrent accesses when running eventlet.
#
# When using this class you should disable your aodh-api service.
#
# == Parameters
#
#   [*servername*]
#     The servername for the virtualhost.
#     Optional. Defaults to $::fqdn
#
#   [*port*]
#     The port.
#     Optional. Defaults to 8042
#
#   [*bind_host*]
#     The host/ip address Apache will listen on.
#     Optional. Defaults to undef (listen on all ip addresses).
#
#   [*path*]
#     The prefix for the endpoint.
#     Optional. Defaults to '/'
#
#   [*ssl*]
#     Use ssl ? (boolean)
#     Optional. Defaults to true
#
#   [*workers*]
#     Number of WSGI workers to spawn.
#     Optional. Defaults to $::os_workers
#
#   [*priority*]
#     (optional) The priority for the vhost.
#     Defaults to '10'
#
#   [*threads*]
#     (optional) The number of threads for the vhost.
#     Defaults to 1
#
#   [*wsgi_process_display_name*]
#     (optional) Name of the WSGI process display-name.
#     Defaults to undef
#
#   [*ssl_cert*]
#   [*ssl_key*]
#   [*ssl_chain*]
#   [*ssl_ca*]
#   [*ssl_crl_path*]
#   [*ssl_crl*]
#   [*ssl_certs_dir*]
#     apache::vhost ssl parameters.
#     Optional. Default to apache::vhost 'ssl_*' defaults.
#
#   [*access_log_file*]
#     The log file name for the virtualhost.
#     Optional. Defaults to false.
#
#   [*access_log_format*]
#     The log format for the virtualhost.
#     Optional. Defaults to false.
#
#   [*error_log_file*]
#     The error log file name for the virtualhost.
#     Optional. Defaults to undef.
#
#  [*custom_wsgi_process_options*]
#    (optional) gives you the oportunity to add custom process options or to
#    overwrite the default options for the WSGI main process.
#    eg. to use a virtual python environment for the WSGI process
#    you could set it to:
#    { python-path => '/my/python/virtualenv' }
#    Defaults to {}
#
#   [*wsgi_script_dir*]
#     (optional) The directory to install the WSGI script for apache to read
#     Optional. Defaults to $::aodh::params::aodh_wsgi_script_path
#
#   [*wsgi_script_source*]
#     (optional) The location of the aodh WSGI script
#     Optional. Defaults to $::aodh::params::aodh_wsgi_script_source
#
#   [*vhost_custom_fragment*]
#     (optional) Passes a string of custom configuration
#     directives to be placed at the end of the vhost configuration.
#     Defaults to undef.
#
#
# == Dependencies
#
#   requires Class['apache'] & Class['aodh']
#
# == Examples
#
#   include apache
#
#   class { 'aodh::wsgi::apache': }
#
class aodh::wsgi::apache (
  $servername                  = $::fqdn,
  $port                        = 8042,
  $bind_host                   = undef,
  $path                        = '/',
  $ssl                         = undef,
  $workers                     = $::os_workers,
  $ssl_cert                    = undef,
  $ssl_key                     = undef,
  $ssl_chain                   = undef,
  $ssl_ca                      = undef,
  $ssl_crl_path                = undef,
  $ssl_crl                     = undef,
  $ssl_certs_dir               = undef,
  $wsgi_process_display_name   = undef,
  $threads                     = 1,
  $priority                    = '10',
  $access_log_file             = false,
  $access_log_format           = false,
  $error_log_file              = undef,
  $custom_wsgi_process_options = {},
  $wsgi_script_dir             = $::aodh::params::aodh_wsgi_script_dir,
  $wsgi_script_source          = $::aodh::params::aodh_wsgi_script_source,
  $vhost_custom_fragment       = undef,
) inherits aodh::params {

  if $ssl == undef {
    warning('Default of the ssl parameter will be changed in a future release')
  }
  $ssl_real = pick($ssl, true)

  include aodh::deps

  # NOTE(aschultz): needed because the packaging may introduce some apache
  # configuration files that apache may remove. See LP#1657847
  Anchor['aodh::install::end'] -> Class['apache']

  openstacklib::wsgi::apache { 'aodh_wsgi':
    bind_host                   => $bind_host,
    bind_port                   => $port,
    group                       => $::aodh::params::group,
    path                        => $path,
    priority                    => $priority,
    servername                  => $servername,
    ssl                         => $ssl_real,
    ssl_ca                      => $ssl_ca,
    ssl_cert                    => $ssl_cert,
    ssl_certs_dir               => $ssl_certs_dir,
    ssl_chain                   => $ssl_chain,
    ssl_crl                     => $ssl_crl,
    ssl_crl_path                => $ssl_crl_path,
    ssl_key                     => $ssl_key,
    threads                     => $threads,
    user                        => $::aodh::params::user,
    vhost_custom_fragment       => $vhost_custom_fragment,
    workers                     => $workers,
    wsgi_daemon_process         => 'aodh',
    wsgi_process_display_name   => $wsgi_process_display_name,
    wsgi_process_group          => 'aodh',
    wsgi_script_dir             => $wsgi_script_dir,
    wsgi_script_file            => 'app',
    wsgi_script_source          => $wsgi_script_source,
    custom_wsgi_process_options => $custom_wsgi_process_options,
    access_log_file             => $access_log_file,
    access_log_format           => $access_log_format,
    error_log_file              => $error_log_file,
  }
}
