# Copyright 2016 Red Hat, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# == Class: tripleo::profile::base::octavia::api
#
# Octavia API server profile for tripleo
#
# === Parameters
#
# [*bootstrap_node*]
#   (Optional) The hostname of the node responsible for bootstrapping tasks
#   Defaults to lookup('octavia_api_short_bootstrap_node_name', undef, undef, undef)
#
# [*certificates_specs*]
#   (Optional) The specifications to give to certmonger for the certificate(s)
#   it will create.
#   Example with hiera:
#     apache_certificates_specs:
#       httpd-internal_api:
#         hostname: <overcloud controller fqdn>
#         service_certificate: <service certificate path>
#         service_key: <service key path>
#         principal: "haproxy/<overcloud controller fqdn>"
#   Defaults to lookup('apache_certificates_specs', undef, undef, {}).
#
# [*enable_internal_tls*]
#   (Optional) Whether TLS in the internal network is enabled or not.
#   Defaults to lookup('enable_internal_tls', undef, undef, false)
#
# [*octavia_network*]
#   (Optional) The network name where the barbican endpoint is listening on.
#   This is set by t-h-t.
#   Defaults to lookup('octavia_api_network', undef, undef, undef)
#
# [*step*]
#   (Optional) The current step in deployment. See tripleo-heat-templates
#   for more details.
#   Defaults to Integer(lookup('step'))
#
# DEPRECATED PARAMETERS
#
# [*ovn_db_host*]
#   (Optional) The IP-Address where OVN DBs are listening.
#   Defaults to undef
#
# [*ovn_nb_port*]
#   (Optional) Port number on which northbound database is listening
#   Defaults to undef
#
# [*neutron_driver*]
#   (Optional) The neutron driver for ml2 currently default tripleo value is ovn.
#   Defaults to undef
#   Not used any more.
#
# [*configure_apache*]
#   (Optional) Whether apache is configured via puppet or not.
#   Defaults to lookup('configure_apache', undef, undef, true)
#
class tripleo::profile::base::octavia::api (
  $bootstrap_node      = lookup('octavia_api_short_bootstrap_node_name', undef, undef, undef),
  $certificates_specs  = lookup('apache_certificates_specs', undef, undef, {}),
  $enable_internal_tls = lookup('enable_internal_tls', undef, undef, false),
  $octavia_network     = lookup('octavia_api_network', undef, undef, undef),
  $step                = Integer(lookup('step')),
  $ovn_db_host         = undef,
  $ovn_nb_port         = undef,
  $neutron_driver      = undef,
  $configure_apache    = lookup('configure_apache', undef, undef, true),
) {
  if $bootstrap_node and $::hostname == downcase($bootstrap_node) {
    $sync_db = true
  } else {
    $sync_db = false
  }

  include tripleo::profile::base::octavia
  include tripleo::profile::base::octavia::authtoken

  if $step >= 4 or ($step >= 3 and $sync_db) {
    if $enable_internal_tls {
      if !$octavia_network {
        fail('octavia_api_network is not set in the hieradata.')
      }
      $tls_certfile = $certificates_specs["httpd-${octavia_network}"]['service_certificate']
      $tls_keyfile = $certificates_specs["httpd-${octavia_network}"]['service_key']
    } else {
      $tls_certfile = undef
      $tls_keyfile = undef
    }
  }
  # We start the Octavia API server on the bootstrap node first, because
  # it will try to populate tables and we need to make sure this happens
  # before it starts on other nodes
  if ($step >= 4 and $sync_db) or ($step >= 5 and !$sync_db) {
    include octavia::controller
    if $ovn_db_host or $ovn_nb_port {
      warning('The ovn_db_host and ovn_nb_port parameters are deprecated from tripleo::profile::base::octavia::api. \
Use tripleo::profile::base::octavia::provider::ovn.')
    }
    class { 'octavia::api':
      sync_db           => $sync_db,
    }
    include octavia::healthcheck
    if $configure_apache {
      include tripleo::profile::base::apache
      class { 'octavia::wsgi::apache':
        ssl_cert => $tls_certfile,
        ssl_key  => $tls_keyfile
      }
    }
  }
}
