
#
# Copyright (C) 2017 Red Hat, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#

require 'spec_helper'

sensubility_conf = "
[default]

[sensu]
connection=ampq://sensu:sensu@localhost:5672//sensu
subscriptions=default,test
tmp_base_dir=/var/lib/collectd-sensubility/checks
checks={\"standalone_check\":{\"command\":\"echo 'foobar'\",\"interval\":5}}

[amqp1]
"

libpodstats_typesdb = '/usr/share/collectd/types.db.libpodstats'

libpodstats_conf = '
LoadPlugin libpodstats
<Plugin "libpodstats">
</Plugin>
'

exec_cmd = <<-EOS
  Exec \"collectd:collectd\" \"collectd-sensubility\"
EOS


describe 'tripleo::profile::base::metrics::collectd' do
  shared_examples_for 'tripleo::profile::base::metrics::collectd' do
    before :each do
      facts.merge!({ :step => params[:step] })
    end

    context 'with step less than 3' do
      let(:params) { { :step => 2 } }
      it 'should do nothing' do
        is_expected.to_not contain_class('collectd')
      end
    end

    context 'with enabled file_logging and step greater than 3' do
      let(:params) do
        { :step => 3,
          :enable_file_logging => true }
      end
      it 'Contains both' do
        is_expected.to compile.with_all_deps
        is_expected.to contain_class('collectd')
        is_expected.to contain_class('collectd::plugin::logfile')
      end
    end

    context 'with defaults and step greater than 3, amqp deploy' do
      let(:params) do
        { :step => 3,
          :amqp_host => 'localhost' }
      end
      it 'has amqp class' do
        is_expected.to compile.with_all_deps
        is_expected.to contain_class('collectd')
        is_expected.to contain_class('collectd::plugin::amqp1').with(
          :manage_package => true,
        )
        is_expected.to contain_service('collectd').with(
          :ensure => 'running',
          :enable => true,
        )
      end
    end

    context 'with defaults and enabled sensubility' do
      let(:params) do
        { :step => 3,
          :amqp_host => 'localhost',
          :enable_sensubility => true }
      end
      it 'has collectd class with exec plugin and enabled sensubility' do
        is_expected.to compile.with_all_deps
        is_expected.to contain_package('collectd-sensubility').with(:ensure => 'present')
        is_expected.to contain_class('collectd')
        is_expected.to contain_class('collectd::plugin::exec')
        is_expected.to contain_concat__fragment('collectd_plugin_exec_conf_sensubility').with({
          :order => 50,
          :target => '/etc/collectd.d/exec-config.conf',
          :content => exec_cmd,
        })
        is_expected.to contain_file('/etc/collectd-sensubility.conf').with_content(sensubility_conf)
      end
    end

    context 'with defaults and enabled libpodstats' do
      let(:params) do
        { :step => 3,
          :enable_libpodstats => true }
      end
      it 'has libpodstats' do
        is_expected.to compile.with_all_deps
        is_expected.to contain_package('collectd-libpod-stats').with(:ensure => 'present')
        is_expected.to contain_class('collectd').with({
          :typesdb => [
            '/usr/share/collectd/types.db',
            libpodstats_typesdb,
          ],
        })
        is_expected.to contain_collectd__type('pod_memory').with({
          :target => libpodstats_typesdb,
          :ds_type => 'GAUGE',
          :min     => 0,
          :max     => 281474976710656,
          :ds_name => 'value',
        })
        is_expected.to contain_collectd__type('pod_cpu').with({
          :target => libpodstats_typesdb,
          :types  => [{
                'ds_type' => 'GAUGE',
                'min'     => 0,
                'max'     => 100.1,
                'ds_name' => 'percent',
            },
            {
                'ds_type' => 'DERIVE',
                'min'     => 0,
                'max'     => 'U',
                'ds_name' => 'time',
            },
          ],
        })
      end
    end
  end

  on_supported_os.each do |os, facts|
    context "on #{os}" do
      let (:facts) {
        facts
      }
      it_behaves_like 'tripleo::profile::base::metrics::collectd'
    end
  end
end
