#
# Copyright (C) 2015 eNovance SAS <licensing@enovance.com>
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
#
# Unit tests for tripleo
#

require 'spec_helper'

describe 'tripleo::firewall' do

  let :params do
    { }
  end

  shared_examples_for 'tripleo::firewall' do

    context 'with firewall enabled' do
      before :each do
        params.merge!(
          :manage_firewall => true,
        )
      end

      it 'configure basic pre firewall rules' do
        is_expected.to contain_firewall('000 accept related established rules').with(
          :proto  => 'all',
          :state  => ['RELATED', 'ESTABLISHED'],
          :action => 'accept',
        )
        is_expected.to contain_firewall('001 accept all icmp').with(
          :proto  => 'icmp',
          :action => 'accept',
          :state  => ['NEW'],
        )
        is_expected.to contain_firewall('002 accept all to lo interface').with(
          :proto   => 'all',
          :iniface => 'lo',
          :action  => 'accept',
          :state   => ['NEW'],
        )
        is_expected.to contain_firewall('003 accept ssh').with(
          :dport   => '22',
          :proto  => 'tcp',
          :action => 'accept',
          :state  => ['NEW'],
        )
      end

      it 'configure basic post firewall rules' do
        is_expected.to contain_firewall('999 drop all').with(
          :proto  => 'all',
          :action => 'drop',
          :source => '0.0.0.0/0',
        )
      end
    end

    context 'with custom firewall rules' do
      before :each do
        params.merge!(
          :manage_firewall     => true,
          :firewall_rules => {
            '300 add custom application 1' => {'port' => '999', 'proto' => 'udp', 'action' => 'accept'},
            '301 add custom application 2' => {'port' => '8081', 'proto' => 'tcp', 'action' => 'accept'},
            '302 fwd custom cidr 1'        => {'port' => 'all', 'chain' => 'FORWARD', 'destination' => '192.0.2.0/24'},
            '303 add custom application 3' => {'dport' => '8081', 'proto' => 'tcp', 'action' => 'accept'},
            '304 add custom application 4' => {'sport' => '1000', 'proto' => 'tcp', 'action' => 'accept'},
            '305 add gre rule'             => {'proto' => 'gre'}
          }
        )
      end
      it 'configure custom firewall rules' do
        is_expected.to contain_firewall('300 add custom application 1').with(
          :port   => '999',
          :proto  => 'udp',
          :action => 'accept',
          :state  => ['NEW'],
        )
        is_expected.to contain_firewall('301 add custom application 2').with(
          :port   => '8081',
          :proto  => 'tcp',
          :action => 'accept',
          :state  => ['NEW'],
        )
        is_expected.to contain_firewall('302 fwd custom cidr 1').with(
          :chain   => 'FORWARD',
          :proto   => 'tcp',
          :destination => '192.0.2.0/24',
        )
        is_expected.to contain_firewall('303 add custom application 3').with(
          :dport   => '8081',
          :proto  => 'tcp',
          :action => 'accept',
          :state  => ['NEW'],
        )
        is_expected.to contain_firewall('304 add custom application 4').with(
          :sport   => '1000',
          :proto  => 'tcp',
          :action => 'accept',
          :state  => ['NEW'],
        )
        is_expected.to contain_firewall('305 add gre rule').without(:state)
      end
    end

    context 'with TCP rule without port or dport or sport specified' do
      before :each do
        params.merge!(
          :manage_firewall => true,
          :firewall_rules  => {
            '500 wrong tcp rule' => {'proto' => 'tcp', 'action' => 'accept'},
          }
        )
      end
      it_raises 'a Puppet::Error', /500 wrong tcp rule firewall rule cannot be created. TCP or UDP rules for INPUT or OUTPUT need port or sport or dport./
    end

  end

  on_supported_os.each do |os, facts|
    context "on #{os}" do
      let(:facts) do
        facts.merge({})
      end

      it_behaves_like 'tripleo::firewall'
    end
  end
end
