# Copyright (c) 2012 OpenStack, LLC.
#
# Licensed under the Apache License, Version 2.0 (the 'License');
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an 'AS IS' BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

ATTR_NOT_SPECIFIED = object()

# Note: a default of ATTR_NOT_SPECIFIED indicates that an
# attribute is not required, but will be generated by the plugin
# if it is not specified.  Particularly, a value of ATTR_NOT_SPECIFIED
# is different from an attribute that has been specified with a value of
# None.  For example, if 'gateway_ip' is ommitted in a request to
# create a subnet, the plugin will receive ATTR_NOT_SPECIFIED
# and the default gateway_ip will be generated.
# However, if gateway_ip is specified as None, this means that
# the subnet does not have a gateway IP.

import logging
import netaddr
import re

from quantum.common import exceptions as q_exc


LOG = logging.getLogger(__name__)


def _validate_boolean(data, valid_values=None):
    if data in [True, False]:
        return
    else:
        msg = _("%s is not boolean") % data
        LOG.debug("validate_boolean: %s", msg)
        return msg


def _validate_values(data, valid_values=None):
    if data in valid_values:
        return
    else:
        msg_dict = dict(data=data, values=valid_values)
        msg = _("%(data)s is not in %(values)s") % msg_dict
        LOG.debug("validate_values: %s", msg)
        return msg


def _validate_mac_address(data, valid_values=None):
    try:
        netaddr.EUI(data)
        return
    except Exception:
        msg = _("%s is not a valid MAC address") % data
        LOG.debug("validate_mac_address: %s", msg)
        return msg


def _validate_ip_address(data, valid_values=None):
    try:
        netaddr.IPAddress(data)
        return
    except Exception:
        msg = _("%s is not a valid IP address") % data
        LOG.debug("validate_ip_address: %s", msg)
        return msg


def _validate_subnet(data, valid_values=None):
    try:
        netaddr.IPNetwork(data)
        return
    except Exception:
        msg = _("%s is not a valid IP subnet") % data
        LOG.debug("validate_subnet: %s", msg)
        return msg


def _validate_regex(data, valid_values=None):
    match = re.match(valid_values, data)
    if match:
        return
    else:
        msg = _("%s is not valid") % data
        LOG.debug("validate_regex: %s", msg)
        return msg


def convert_to_boolean(data):
    try:
        i = int(data)
        if i in [True, False]:
            # Ensure that the value is True or False
            if i:
                return True
            else:
                return False
    except ValueError, TypeError:
        if (data == "True" or data == "true"):
            return True
        if (data == "False" or data == "false"):
            return False
    msg = _("%s is not boolean") % data
    raise q_exc.InvalidInput(error_message=msg)


HEX_ELEM = '[0-9A-Fa-f]'
UUID_PATTERN = '-'.join([HEX_ELEM + '{8}', HEX_ELEM + '{4}',
                         HEX_ELEM + '{4}', HEX_ELEM + '{4}',
                         HEX_ELEM + '{12}'])

# Dictionary that maintains a list of validation functions
validators = {'type:boolean': _validate_boolean,
              'type:values': _validate_values,
              'type:mac_address': _validate_mac_address,
              'type:ip_address': _validate_ip_address,
              'type:subnet': _validate_subnet,
              'type:regex': _validate_regex}

# Note: a default of ATTR_NOT_SPECIFIED indicates that an
# attribute is not required, but will be generated by the plugin
# if it is not specified.  Particularly, a value of ATTR_NOT_SPECIFIED
# is different from an attribute that has been specified with a value of
# None.  For example, if 'gateway_ip' is ommitted in a request to
# create a subnet, the plugin will receive ATTR_NOT_SPECIFIED
# and the default gateway_ip will be generated.
# However, if gateway_ip is specified as None, this means that
# the subnet does not have a gateway IP.
# Some of the following attributes are used by the policy engine.
# They are explicitly marked with the required_by_policy flag to ensure
# they are always returned by a plugin for policy processing, even if
# they are not specified in the 'fields' query param
RESOURCE_ATTRIBUTE_MAP = {
    'networks': {
        'id': {'allow_post': False, 'allow_put': False,
               'validate': {'type:regex': UUID_PATTERN}},
        'name': {'allow_post': True, 'allow_put': True},
        'subnets': {'allow_post': True, 'allow_put': True, 'default': []},
        'admin_state_up': {'allow_post': True, 'allow_put': True,
                           'default': True, 'convert_to': convert_to_boolean,
                           'validate': {'type:boolean': None}},
        'status': {'allow_post': False, 'allow_put': False},
        'tenant_id': {'allow_post': True, 'allow_put': False,
                      'required_by_policy': True},
    },
    'ports': {
        'id': {'allow_post': False, 'allow_put': False,
               'validate': {'type:regex': UUID_PATTERN}},
        'network_id': {'allow_post': True, 'allow_put': False,
                       'validate': {'type:regex': UUID_PATTERN}},
        'admin_state_up': {'allow_post': True, 'allow_put': True,
                           'default': True, 'convert_to': convert_to_boolean,
                           'validate': {'type:boolean': None}},
        'mac_address': {'allow_post': True, 'allow_put': False,
                        'default': ATTR_NOT_SPECIFIED,
                        'validate': {'type:mac_address': None}},
        'fixed_ips': {'allow_post': True, 'allow_put': True,
                      'default': ATTR_NOT_SPECIFIED},
        'host_routes': {'allow_post': True, 'allow_put': True,
                        'default': ATTR_NOT_SPECIFIED},
        'device_id': {'allow_post': True, 'allow_put': True, 'default': ''},
        'tenant_id': {'allow_post': True, 'allow_put': False,
                      'required_by_policy': True},
    },
    'subnets': {
        'id': {'allow_post': False, 'allow_put': False,
               'validate': {'type:regex': UUID_PATTERN}},
        'ip_version': {'allow_post': True, 'allow_put': False,
                       'convert_to': int,
                       'validate': {'type:values': [4, 6]}},
        'network_id': {'allow_post': True, 'allow_put': False,
                       'validate': {'type:regex': UUID_PATTERN}},
        'cidr': {'allow_post': True, 'allow_put': False,
                 'validate': {'type:subnet': None}},
        'gateway_ip': {'allow_post': True, 'allow_put': True,
                       'default': ATTR_NOT_SPECIFIED,
                       'validate': {'type:ip_address': None}},
        #TODO(salvatore-orlando): Enable PUT on allocation_pools
        'allocation_pools': {'allow_post': True, 'allow_put': False,
                             'default': ATTR_NOT_SPECIFIED},
        'dns_namesevers': {'allow_post': True, 'allow_put': True,
                           'default': ATTR_NOT_SPECIFIED},
        'additional_host_routes': {'allow_post': True, 'allow_put': True,
                                   'default': ATTR_NOT_SPECIFIED},
        'tenant_id': {'allow_post': True, 'allow_put': False,
                      'required_by_policy': True},
    }
}
