/*
 * Decompiled with CFR 0.152.
 */
package org.apache.oozie.util.db;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import java.util.concurrent.Callable;
import org.apache.oozie.util.XLog;
import org.apache.oozie.util.db.RetryAttemptState;

public class OperationRetryHandler {
    private static XLog LOG = XLog.getLog(OperationRetryHandler.class);
    @VisibleForTesting
    static final RetryAttemptState RETRY_ATTEMPT_STATE = new RetryAttemptState();
    private final int maxRetryCount;
    private final long initialWaitTime;
    private final long maxWaitTime;
    private final Predicate<Throwable> retryPredicate;
    private final boolean shouldRetry;

    public OperationRetryHandler(int maxRetryCount, long initialWaitTime, long maxWaitTime, Predicate<Throwable> retryPredicate) {
        Preconditions.checkArgument((maxRetryCount >= 0 ? 1 : 0) != 0, (Object)"Retry count must not be less than zero");
        Preconditions.checkArgument((initialWaitTime > 0L ? 1 : 0) != 0, (Object)"Initial wait time must be greater than zero");
        Preconditions.checkArgument((maxWaitTime >= 0L ? 1 : 0) != 0, (Object)"Maximum wait time must not be less than zero");
        this.maxRetryCount = maxRetryCount;
        this.initialWaitTime = initialWaitTime;
        this.maxWaitTime = maxWaitTime;
        this.retryPredicate = (Predicate)Preconditions.checkNotNull(retryPredicate, (Object)"Retry predicate must not be null");
        this.shouldRetry = maxRetryCount != 0 && maxWaitTime != 0L;
        LOG.trace("Retry handler parameters are set.[maxRetryCount={0};initialWaitTime={1};maxWaitTime={2};retryPredicate.class={3};shouldRetry={4}]", this.maxRetryCount, this.initialWaitTime, this.maxWaitTime, this.retryPredicate.getClass().getName(), this.shouldRetry);
    }

    public <V> V executeWithRetry(Callable<V> operation) throws Exception {
        int retries = 0;
        long waitTime = this.initialWaitTime;
        Exception lastException = null;
        if (!this.shouldRetry) {
            try {
                LOG.trace("Configured not to retry, calling operation once.");
                V result = operation.call();
                LOG.trace("Operation called once successfully.");
                return result;
            }
            catch (Exception e) {
                LOG.error("An error occurred while calling the operation once. [e.message={0}]", e.getMessage());
                throw e;
            }
        }
        RETRY_ATTEMPT_STATE.signalStart();
        while (retries < this.maxRetryCount) {
            V v;
            try {
                LOG.trace("Calling operation. [retries={0}]", retries);
                ++retries;
                V result = operation.call();
                LOG.trace("Operation called successfully.");
                v = result;
            }
            catch (Exception e) {
                LOG.warn((Object)"Database error", e);
                if (RETRY_ATTEMPT_STATE.isExhausted()) {
                    LOG.error("Retry attempts have been exhausted. [e.message={0}]", e.getMessage());
                    throw e;
                }
                if (this.retryPredicate.apply((Object)e)) {
                    LOG.trace("Exception is not on blacklist, handling retry. [retries={0};e.class={1}]", retries, e.getClass().getName());
                    waitTime = this.handleRetry(waitTime, retries);
                    lastException = e;
                    continue;
                }
                LOG.warn("Exception is on blacklist, not handling retry. [retries={0};e.class={1}]", retries, e.getClass().getName());
                throw e;
            }
            return v;
        }
        LOG.error("Number of maximum retry attempts exhausted");
        RETRY_ATTEMPT_STATE.signalExhausted();
        throw lastException;
        finally {
            RETRY_ATTEMPT_STATE.signalEnd();
        }
    }

    private long handleRetry(long sleepBeforeRetryMs, int retries) throws InterruptedException {
        LOG.warn("Operation failed, sleeping {0} milliseconds before retry #{1}", sleepBeforeRetryMs, retries);
        Thread.sleep(sleepBeforeRetryMs);
        return (sleepBeforeRetryMs *= 2L) > this.maxWaitTime ? this.maxWaitTime : sleepBeforeRetryMs;
    }
}

