/*
 * Decompiled with CFR 0.152.
 */
package org.apache.oozie.tools;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.FlushModeType;
import javax.persistence.Table;
import javax.persistence.metamodel.EntityType;
import javax.persistence.metamodel.Metamodel;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.oozie.BundleActionBean;
import org.apache.oozie.BundleJobBean;
import org.apache.oozie.CoordinatorActionBean;
import org.apache.oozie.CoordinatorJobBean;
import org.apache.oozie.WorkflowActionBean;
import org.apache.oozie.WorkflowJobBean;
import org.apache.oozie.cli.CLIParser;
import org.apache.oozie.executor.jpa.JPAExecutorException;
import org.apache.oozie.service.ConfigurationService;
import org.apache.oozie.service.JPAService;
import org.apache.oozie.service.Services;
import org.apache.oozie.sla.SLARegistrationBean;
import org.apache.oozie.sla.SLASummaryBean;
import org.apache.oozie.store.StoreException;
import org.apache.openjpa.persistence.OpenJPAEntityManagerSPI;
import org.apache.openjpa.persistence.RollbackException;

public class OozieDBImportCLI {
    private static final String[] HELP_INFO = new String[]{"", "OozieDBImportCLI reads Oozie database from a zip file."};
    private static final String IMPORT_CMD = "import";
    private static final String HELP_CMD = "help";
    public static final String OOZIE_DB_IMPORT_BATCH_SIZE_KEY = "oozie.db.import.batch.size";
    static final int DEFAULT_BATCH_SIZE = 1000;
    private static int IMPORT_BATCH_SIZE;
    private static final String OPTION_VERBOSE_SHORT = "v";
    private static final String OPTION_VERBOSE_LONG = "verbose";
    private final EntityManager entityManager;
    private final ZipFile mainZipFile;
    private final boolean verbose;
    private boolean cleanupNecessary = false;
    private final Set<Class<?>> entityClasses = Sets.newLinkedHashSet();

    private OozieDBImportCLI(EntityManager entityManager, ZipFile mainZipFile, boolean verbose) {
        this.entityManager = entityManager;
        this.mainZipFile = mainZipFile;
        this.verbose = verbose;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) throws ParseException {
        CLIParser parser = new CLIParser("oozie-setup.sh", HELP_INFO);
        parser.addCommand(HELP_CMD, "", "display usage for all commands or specified command", new Options(), false);
        parser.addCommand(IMPORT_CMD, "", "imports the contents of the Oozie database from the specified file", new Options().addOption(OPTION_VERBOSE_SHORT, OPTION_VERBOSE_LONG, false, "Enables verbose logging."), true);
        boolean verbose = false;
        try {
            CLIParser.Command command = parser.parse(args);
            if (command.getName().equals(IMPORT_CMD)) {
                Services services = new Services();
                services.getConf().set("oozie.services", JPAService.class.getName());
                services.getConf().set("oozie.services.ext", "");
                services.init();
                OozieDBImportCLI.setImportBatchSize();
                System.out.println("==========================================================");
                System.out.println(Arrays.toString(command.getCommandLine().getArgs()));
                System.out.println(String.format("Import batch length is %d", IMPORT_BATCH_SIZE));
                verbose = command.getCommandLine().hasOption(OPTION_VERBOSE_SHORT) || command.getCommandLine().hasOption(OPTION_VERBOSE_LONG);
                OozieDBImportCLI.importAllDBTables(command.getCommandLine().getArgs()[0], verbose);
            } else if (command.getName().equals(HELP_CMD)) {
                parser.showHelp(command.getCommandLine());
            }
        }
        catch (ParseException pex) {
            System.err.println("Invalid sub-command: " + pex.getMessage());
            System.err.println();
            System.err.println(parser.shortHelp());
            System.exit(1);
        }
        catch (Throwable e) {
            System.err.println();
            System.err.println("Error: " + e.getMessage());
            System.err.println();
            if (verbose) {
                System.err.println("Stack trace for the error was (for debug purposes):");
                System.err.println("--------------------------------------");
                e.printStackTrace(System.err);
                System.err.println("--------------------------------------");
                System.err.println();
            }
            System.exit(1);
        }
        finally {
            if (Services.get() != null) {
                Services.get().destroy();
            }
        }
    }

    private static void setImportBatchSize() {
        if (!Strings.isNullOrEmpty((String)System.getProperty(OOZIE_DB_IMPORT_BATCH_SIZE_KEY))) {
            try {
                IMPORT_BATCH_SIZE = Integer.parseInt(System.getProperty(OOZIE_DB_IMPORT_BATCH_SIZE_KEY));
            }
            catch (NumberFormatException e) {
                IMPORT_BATCH_SIZE = ConfigurationService.getInt((String)OOZIE_DB_IMPORT_BATCH_SIZE_KEY, (int)1000);
            }
        } else {
            IMPORT_BATCH_SIZE = ConfigurationService.getInt((String)OOZIE_DB_IMPORT_BATCH_SIZE_KEY, (int)1000);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void importAllDBTables(String zipFileName, boolean verbose) throws StoreException, IOException, JPAExecutorException, SQLException {
        try (EntityManager entityManager = null;
             ZipFile mainZipFile = new ZipFile(zipFileName);){
            entityManager = ((JPAService)Services.get().get(JPAService.class)).getEntityManager();
            entityManager.setFlushMode(FlushModeType.COMMIT);
            OozieDBImportCLI importer = new OozieDBImportCLI(entityManager, mainZipFile, verbose);
            importer.checkDBVersion();
            importer.checkTablesArePresentAndEmpty();
            importer.importOneInputFileToOneEntityTable(WorkflowJobBean.class, "ooziedb_wf.json");
            importer.importOneInputFileToOneEntityTable(WorkflowActionBean.class, "ooziedb_ac.json");
            importer.importOneInputFileToOneEntityTable(CoordinatorJobBean.class, "ooziedb_cj.json");
            importer.importOneInputFileToOneEntityTable(CoordinatorActionBean.class, "ooziedb_ca.json");
            importer.importOneInputFileToOneEntityTable(BundleJobBean.class, "ooziedb_bnj.json");
            importer.importOneInputFileToOneEntityTable(BundleActionBean.class, "ooziedb_bna.json");
            importer.importOneInputFileToOneEntityTable(SLARegistrationBean.class, "ooziedb_slareg.json");
            importer.importOneInputFileToOneEntityTable(SLASummaryBean.class, "ooziedb_slasum.json");
            boolean cleanupPerformed = importer.cleanupIfNecessary();
            Preconditions.checkState((!cleanupPerformed ? 1 : 0) != 0, (Object)"DB cleanup happened due to skipped rows. See previous log entries about what rows were skipped and why.");
        }
    }

    private void checkDBVersion() throws IOException {
        block18: {
            try {
                String currentDBVersion = (String)this.entityManager.createNativeQuery("select data from OOZIE_SYS where name='db.version'").getSingleResult();
                String dumpDBVersion = null;
                ZipEntry sysInfoEntry = this.mainZipFile.getEntry("ooziedb_sysinfo.json");
                try (BufferedReader reader = new BufferedReader(new InputStreamReader(this.mainZipFile.getInputStream(sysInfoEntry), Charsets.UTF_8));){
                    String line;
                    Gson gson = new Gson();
                    while ((line = reader.readLine()) != null) {
                        List info = (List)gson.fromJson(line, List.class);
                        if (info.size() <= 1 || !"db.version".equals(info.get(0))) continue;
                        dumpDBVersion = (String)info.get(1);
                    }
                }
                if (currentDBVersion.equals(dumpDBVersion)) {
                    System.out.println("Loading to Oozie database version " + currentDBVersion);
                } else {
                    System.err.println("ERROR Oozie database version mismatch.");
                    System.err.println("Oozie DB version:\t" + currentDBVersion);
                    System.err.println("Dump DB version:\t" + dumpDBVersion);
                    System.exit(1);
                }
            }
            catch (Exception e) {
                System.err.println();
                System.err.println("Error during DB version check: " + e.getMessage());
                System.err.println();
                if (!this.verbose) break block18;
                System.err.println("Stack trace for the error was (for debug purposes):");
                System.err.println("--------------------------------------");
                e.printStackTrace(System.err);
                System.err.println("--------------------------------------");
                System.err.println();
            }
        }
    }

    private void checkTablesArePresentAndEmpty() throws SQLException {
        this.checkTableIsPresentAndEmpty(WorkflowJobBean.class);
        this.checkTableIsPresentAndEmpty(WorkflowActionBean.class);
        this.checkTableIsPresentAndEmpty(CoordinatorJobBean.class);
        this.checkTableIsPresentAndEmpty(CoordinatorActionBean.class);
        this.checkTableIsPresentAndEmpty(BundleJobBean.class);
        this.checkTableIsPresentAndEmpty(BundleActionBean.class);
        this.checkTableIsPresentAndEmpty(SLARegistrationBean.class);
        this.checkTableIsPresentAndEmpty(SLASummaryBean.class);
    }

    private <E> void checkTableIsPresentAndEmpty(Class<E> entityClass) throws SQLException {
        OpenJPAEntityManagerSPI entityManagerDelegate = (OpenJPAEntityManagerSPI)this.entityManager.getDelegate();
        Connection connection = (Connection)entityManagerDelegate.getConnection();
        DatabaseMetaData metaData = connection.getMetaData();
        String tableName = OozieDBImportCLI.findTableName(this.entityManager, entityClass);
        try (ResultSet rs = metaData.getTables(null, null, tableName, null);){
            Preconditions.checkState((boolean)rs.next(), (Object)String.format("Table [%s] does not exist for class [%s].", tableName, entityClass.getSimpleName()));
        }
        long entityCount = this.getEntityCount(entityClass);
        Preconditions.checkState((entityCount == 0L ? 1 : 0) != 0, (Object)String.format("There are already [%d] entries in table [%s] for class [%s], should be empty.", entityCount, tableName, entityClass.getSimpleName()));
    }

    private <E> long getEntityCount(Class<E> entityClass) {
        return (Long)this.entityManager.createQuery(String.format("SELECT COUNT(e) FROM %s e", entityClass.getSimpleName()), Long.class).getSingleResult();
    }

    private <E> void importOneInputFileToOneEntityTable(Class<E> entityClass, String importFileName) {
        BatchTransactionHandler batchTransactionHandler = new BatchTransactionHandler();
        BatchEntityPersister batchEntityPersister = new BatchEntityPersister(entityClass, importFileName, batchTransactionHandler);
        ArrayList batch = Lists.newArrayList();
        Gson gson = new Gson();
        ZipEntry importEntry = this.mainZipFile.getEntry(importFileName);
        if (importEntry != null) {
            long lineIndex = 1L;
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(this.mainZipFile.getInputStream(importEntry), Charsets.UTF_8));){
                String line;
                while ((line = reader.readLine()) != null) {
                    Object newEntity = gson.fromJson(line, entityClass);
                    batch.add(newEntity);
                    if (lineIndex % (long)IMPORT_BATCH_SIZE == 0L) {
                        System.out.println(String.format("Batch is full, persisting. [lineIndex=%s;batch.size=%s]", lineIndex, batch.size()));
                        batchEntityPersister.persist(batch);
                    }
                    ++lineIndex;
                }
            }
            catch (IOException e) {
                this.rollbackAndThrow(importFileName, batchTransactionHandler, e);
            }
            catch (JsonSyntaxException e) {
                if (this.verbose) {
                    System.err.println(String.format("JSON error. [lineIndex=%s;e.message=%s]", lineIndex, e.getMessage()));
                }
                this.rollbackAndThrow(importFileName, batchTransactionHandler, (Exception)((Object)e));
            }
        }
        if (!batch.isEmpty()) {
            System.out.println(String.format("Persisting last batch. [batch.size=%s]", batch.size()));
            batchEntityPersister.persist(batch);
        }
        String tableName = OozieDBImportCLI.findTableName(this.entityManager, entityClass);
        System.out.println(String.format("%s row(s) imported to table %s.", batchTransactionHandler.getTotalPersistedCount(), tableName));
        if (batchTransactionHandler.getTotalSkippedCount() > 0) {
            System.err.println(String.format("[%s] row(s) skipped while importing to table [%s]. Will remove all the rows of all the tables to get clean data.", batchTransactionHandler.getTotalSkippedCount(), tableName));
            this.cleanupNecessary = true;
        }
    }

    private boolean cleanupIfNecessary() {
        if (!this.cleanupNecessary) {
            System.out.println("Cleanup not necessary, no entities skipped.");
            return false;
        }
        System.out.println(String.format("Performing cleanup of [%d] tables due to skipped entities.", this.entityClasses.size()));
        for (Class<?> entityClass : this.entityClasses) {
            String tableName = OozieDBImportCLI.findTableName(this.entityManager, entityClass);
            System.out.println(String.format("Cleaning up table [%s].", tableName));
            BatchTransactionHandler batchTransactionHandler = new BatchTransactionHandler();
            batchTransactionHandler.begin();
            this.entityManager.createQuery(String.format("DELETE FROM %s e", entityClass.getSimpleName())).executeUpdate();
            batchTransactionHandler.commit();
            System.out.println(String.format("Table [%s] cleaned up.", tableName));
        }
        System.out.println(String.format("Cleanup of [%d] tables due to skipped entities performed.", this.entityClasses.size()));
        return true;
    }

    private <E> void rollbackAndThrow(String importFileName, BatchTransactionHandler<E> batchTransactionHandler, Exception cause) {
        batchTransactionHandler.rollbackIfActive();
        throw new RuntimeException(String.format("Import failed from json [zippedFileName=%s;e.message=%s].", importFileName, cause.getMessage()), cause);
    }

    static <E> String findTableName(EntityManager entityManager, Class<E> entityClass) {
        Metamodel meta = entityManager.getMetamodel();
        EntityType entityType = meta.entity(entityClass);
        Table t = entityClass.getAnnotation(Table.class);
        String tableName = t == null ? entityType.getName().toUpperCase() : t.name();
        return tableName;
    }

    private class BatchEntityPersister<E> {
        private final Class<E> entityClass;
        private final String importFileName;
        private final BatchTransactionHandler<E> batchTransactionHandler;

        private BatchEntityPersister(Class<E> entityClass, String importFileName, BatchTransactionHandler<E> batchTransactionHandler) {
            this.entityClass = entityClass;
            this.importFileName = importFileName;
            this.batchTransactionHandler = batchTransactionHandler;
        }

        void persist(List<E> batch) {
            if (batch.isEmpty()) {
                System.out.println("No entities to import.");
                return;
            }
            while (!batch.isEmpty()) {
                try {
                    OozieDBImportCLI.this.entityClasses.add(this.entityClass);
                    this.batchTransactionHandler.begin();
                    ListIterator<E> iterator = batch.listIterator();
                    while (iterator.hasNext()) {
                        E entityToPersist = iterator.next();
                        iterator.remove();
                        this.batchTransactionHandler.persist(entityToPersist);
                    }
                    this.batchTransactionHandler.commit();
                }
                catch (RollbackException re) {
                    for (E pendingEntity : this.batchTransactionHandler.getPendingEntities()) {
                        this.batchTransactionHandler.persistAndTryCommit(pendingEntity);
                    }
                }
                catch (Exception e) {
                    OozieDBImportCLI.this.rollbackAndThrow(this.importFileName, this.batchTransactionHandler, e);
                }
            }
        }
    }

    private class BatchTransactionHandler<E> {
        private EntityTransaction currentTransaction;
        private int totalPersistedCount = 0;
        private int totalSkippedCount = 0;
        private List<E> pendingEntities = Lists.newArrayList();

        private BatchTransactionHandler() {
        }

        void begin() {
            this.currentTransaction = OozieDBImportCLI.this.entityManager.getTransaction();
            this.currentTransaction.begin();
            this.pendingEntities.clear();
        }

        void commit() {
            Preconditions.checkNotNull((Object)this.currentTransaction, (Object)"TX should be open.");
            this.currentTransaction.commit();
            this.totalPersistedCount += this.pendingEntities.size();
            this.pendingEntities.clear();
        }

        void rollbackIfActive() {
            if (this.currentTransaction == null) {
                return;
            }
            if (this.currentTransaction.isActive()) {
                this.currentTransaction.rollback();
            }
            this.pendingEntities.clear();
        }

        void persist(E newEntity) {
            Preconditions.checkNotNull((Object)this.currentTransaction, (Object)"TX should be open.");
            OozieDBImportCLI.this.entityManager.persist(newEntity);
            this.pendingEntities.add(newEntity);
            if (this.pendingEntities.size() == IMPORT_BATCH_SIZE) {
                this.commit();
                this.begin();
            }
        }

        void persistAndTryCommit(E pendingEntity) {
            try {
                this.currentTransaction.begin();
                OozieDBImportCLI.this.entityManager.persist(pendingEntity);
                this.currentTransaction.commit();
                ++this.totalPersistedCount;
            }
            catch (RollbackException re) {
                if (OozieDBImportCLI.this.verbose) {
                    System.err.println(String.format("Cannot persist entity, skipping. [re.failedObject=%s]", re.getFailedObject()));
                }
                ++this.totalSkippedCount;
            }
        }

        int getTotalPersistedCount() {
            return this.totalPersistedCount;
        }

        int getTotalSkippedCount() {
            return this.totalSkippedCount;
        }

        List<E> getPendingEntities() {
            return this.pendingEntities;
        }
    }
}

