# Copyright 2022 99cloud
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# flake8: noqa
# fmt: off

from skyline_apiserver.schemas.policy_manager import Operation

from . import base

list_rules = (
    base.Rule(
        name="admin_or_owner",
        check_str=("is_admin:True or (role:admin and is_admin_project:True) or project_id:%(project_id)s"),
        description="DEPRECATED: This rule will be removed in the Yoga release.            Default rule for most non-Admin APIs.",
    ),
    base.Rule(
        name="system_or_domain_or_project_admin",
        check_str=("(role:admin and system_scope:all) or (role:admin and domain_id:%(domain_id)s) or (role:admin and project_id:%(project_id)s)"),
        description="DEPRECATED: This rule will be removed in the Yoga release.            Default rule for admins of cloud, domain or a project.",
    ),
    base.Rule(
        name="context_is_admin",
        check_str=("role:admin"),
        description="Decides what is required for the 'is_admin:True' check to succeed.",
    ),
    base.Rule(
        name="admin_api",
        check_str=("is_admin:True or (role:admin and is_admin_project:True)"),
        description="Default rule for most Admin APIs.",
    ),
    base.Rule(
        name="xena_system_admin_or_project_reader",
        check_str=("(role:admin) or (role:reader and project_id:%(project_id)s)"),
        description="NOTE: this purely role-based rule recognizes only project scope",
    ),
    base.Rule(
        name="xena_system_admin_or_project_member",
        check_str=("(role:admin) or ((role:member or role:_member_) and project_id:%(project_id)s)"),
        description="NOTE: this purely role-based rule recognizes only project scope",
    ),
    base.Rule(
        name="volume_extension:volume_type_encryption",
        check_str=("rule:admin_api"),
        description="DEPRECATED: This rule will be removed in the Yoga release.",
    ),
    base.APIRule(
        name="volume:attachment_create",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create attachment.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/attachments")],
    ),
    base.APIRule(
        name="volume:attachment_update",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update attachment.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/attachments/{attachment_id}")],
    ),
    base.APIRule(
        name="volume:attachment_delete",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete attachment.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/attachments/{attachment_id}")],
    ),
    base.APIRule(
        name="volume:attachment_complete",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Mark a volume attachment process as completed (in-use)",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/attachments/{attachment_id}/action (os-complete)")],
    ),
    base.APIRule(
        name="volume:multiattach_bootable_volume",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Allow multiattach of bootable volumes.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/attachments")],
    ),
    base.APIRule(
        name="message:get_all",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List messages.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/messages")],
    ),
    base.APIRule(
        name="message:get",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show message.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/messages/{message_id}")],
    ),
    base.APIRule(
        name="message:delete",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete message.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/messages/{message_id}")],
    ),
    base.APIRule(
        name="clusters:get_all",
        check_str=("rule:admin_api"),
        description="List clusters.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/clusters"), Operation(method="GET", path="/clusters/detail")],
    ),
    base.APIRule(
        name="clusters:get",
        check_str=("rule:admin_api"),
        description="Show cluster.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/clusters/{cluster_id}")],
    ),
    base.APIRule(
        name="clusters:update",
        check_str=("rule:admin_api"),
        description="Update cluster.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/clusters/{cluster_id}")],
    ),
    base.APIRule(
        name="workers:cleanup",
        check_str=("rule:admin_api"),
        description="Clean up workers.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/workers/cleanup")],
    ),
    base.APIRule(
        name="volume:get_snapshot_metadata",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show snapshot's metadata or one specified metadata with a given key.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/snapshots/{snapshot_id}/metadata"), Operation(method="GET", path="/snapshots/{snapshot_id}/metadata/{key}")],
    ),
    base.APIRule(
        name="volume:update_snapshot_metadata",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update snapshot's metadata or one specified metadata with a given key.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/snapshots/{snapshot_id}/metadata"), Operation(method="PUT", path="/snapshots/{snapshot_id}/metadata/{key}")],
    ),
    base.APIRule(
        name="volume:delete_snapshot_metadata",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete snapshot's specified metadata with a given key.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/snapshots/{snapshot_id}/metadata/{key}")],
    ),
    base.APIRule(
        name="volume:get_all_snapshots",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List snapshots.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/snapshots"), Operation(method="GET", path="/snapshots/detail")],
    ),
    base.APIRule(
        name="volume_extension:extended_snapshot_attributes",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List or show snapshots with extended attributes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/snapshots/{snapshot_id}"), Operation(method="GET", path="/snapshots/detail")],
    ),
    base.APIRule(
        name="volume:create_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/snapshots")],
    ),
    base.APIRule(
        name="volume:get_snapshot",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show snapshot.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/snapshots/{snapshot_id}")],
    ),
    base.APIRule(
        name="volume:update_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update snapshot.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/snapshots/{snapshot_id}")],
    ),
    base.APIRule(
        name="volume:delete_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete snapshot.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/snapshots/{snapshot_id}")],
    ),
    base.APIRule(
        name="volume_extension:snapshot_admin_actions:reset_status",
        check_str=("rule:admin_api"),
        description="Reset status of a snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/snapshots/{snapshot_id}/action (os-reset_status)")],
    ),
    base.APIRule(
        name="snapshot_extension:snapshot_actions:update_snapshot_status",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update database fields of snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/snapshots/{snapshot_id}/action (update_snapshot_status)")],
    ),
    base.APIRule(
        name="volume_extension:snapshot_admin_actions:force_delete",
        check_str=("rule:admin_api"),
        description="Force delete a snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/snapshots/{snapshot_id}/action (os-force_delete)")],
    ),
    base.APIRule(
        name="snapshot_extension:list_manageable",
        check_str=("rule:admin_api"),
        description="List (in detail) of snapshots which are available to manage.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/manageable_snapshots"), Operation(method="GET", path="/manageable_snapshots/detail")],
    ),
    base.APIRule(
        name="snapshot_extension:snapshot_manage",
        check_str=("rule:admin_api"),
        description="Manage an existing snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/manageable_snapshots")],
    ),
    base.APIRule(
        name="snapshot_extension:snapshot_unmanage",
        check_str=("rule:admin_api"),
        description="Stop managing a snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/snapshots/{snapshot_id}/action (os-unmanage)")],
    ),
    base.APIRule(
        name="backup:get_all",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List backups.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/backups"), Operation(method="GET", path="/backups/detail")],
    ),
    base.APIRule(
        name="backup:backup_project_attribute",
        check_str=("rule:admin_api"),
        description="List backups or show backup with project attributes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/backups/{backup_id}"), Operation(method="GET", path="/backups/detail")],
    ),
    base.APIRule(
        name="backup:create",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create backup.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/backups")],
    ),
    base.APIRule(
        name="backup:get",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show backup.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/backups/{backup_id}")],
    ),
    base.APIRule(
        name="backup:update",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update backup.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/backups/{backup_id}")],
    ),
    base.APIRule(
        name="backup:delete",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete backup.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/backups/{backup_id}")],
    ),
    base.APIRule(
        name="backup:restore",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Restore backup.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/backups/{backup_id}/restore")],
    ),
    base.APIRule(
        name="backup:backup-import",
        check_str=("rule:admin_api"),
        description="Import backup.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/backups/{backup_id}/import_record")],
    ),
    base.APIRule(
        name="backup:export-import",
        check_str=("rule:admin_api"),
        description="Export backup.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/backups/{backup_id}/export_record")],
    ),
    base.APIRule(
        name="volume_extension:backup_admin_actions:reset_status",
        check_str=("rule:admin_api"),
        description="Reset status of a backup.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/backups/{backup_id}/action (os-reset_status)")],
    ),
    base.APIRule(
        name="volume_extension:backup_admin_actions:force_delete",
        check_str=("rule:admin_api"),
        description="Force delete a backup.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/backups/{backup_id}/action (os-force_delete)")],
    ),
    base.APIRule(
        name="group:get_all",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List groups.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/groups"), Operation(method="GET", path="/groups/detail")],
    ),
    base.APIRule(
        name="group:create",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create group.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups")],
    ),
    base.APIRule(
        name="group:get",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show group.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/groups/{group_id}")],
    ),
    base.APIRule(
        name="group:update",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update group.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/groups/{group_id}")],
    ),
    base.APIRule(
        name="group:group_project_attribute",
        check_str=("rule:admin_api"),
        description="List groups or show group with project attributes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/groups/{group_id}"), Operation(method="GET", path="/groups/detail")],
    ),
    base.APIRule(
        name="group:group_types:create",
        check_str=("rule:admin_api"),
        description="Create a group type.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/group_types/")],
    ),
    base.APIRule(
        name="group:group_types:update",
        check_str=("rule:admin_api"),
        description="Update a group type.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/group_types/{group_type_id}")],
    ),
    base.APIRule(
        name="group:group_types:delete",
        check_str=("rule:admin_api"),
        description="Delete a group type.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/group_types/{group_type_id}")],
    ),
    base.APIRule(
        name="group:access_group_types_specs",
        check_str=("rule:admin_api"),
        description="Show group type with type specs attributes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/group_types/{group_type_id}")],
    ),
    base.APIRule(
        name="group:group_types_specs:get",
        check_str=("rule:admin_api"),
        description="Show a group type spec.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/group_types/{group_type_id}/group_specs/{g_spec_id}")],
    ),
    base.APIRule(
        name="group:group_types_specs:get_all",
        check_str=("rule:admin_api"),
        description="List group type specs.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/group_types/{group_type_id}/group_specs")],
    ),
    base.APIRule(
        name="group:group_types_specs:create",
        check_str=("rule:admin_api"),
        description="Create a group type spec.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/group_types/{group_type_id}/group_specs")],
    ),
    base.APIRule(
        name="group:group_types_specs:update",
        check_str=("rule:admin_api"),
        description="Update a group type spec.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/group_types/{group_type_id}/group_specs/{g_spec_id}")],
    ),
    base.APIRule(
        name="group:group_types_specs:delete",
        check_str=("rule:admin_api"),
        description="Delete a group type spec.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/group_types/{group_type_id}/group_specs/{g_spec_id}")],
    ),
    base.APIRule(
        name="group:get_all_group_snapshots",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List group snapshots.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/group_snapshots"), Operation(method="GET", path="/group_snapshots/detail")],
    ),
    base.APIRule(
        name="group:create_group_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create group snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/group_snapshots")],
    ),
    base.APIRule(
        name="group:get_group_snapshot",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show group snapshot.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/group_snapshots/{group_snapshot_id}")],
    ),
    base.APIRule(
        name="group:delete_group_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete group snapshot.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/group_snapshots/{group_snapshot_id}")],
    ),
    base.APIRule(
        name="group:update_group_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update group snapshot.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/group_snapshots/{group_snapshot_id}")],
    ),
    base.APIRule(
        name="group:group_snapshot_project_attribute",
        check_str=("rule:admin_api"),
        description="List group snapshots or show group snapshot with project attributes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/group_snapshots/{group_snapshot_id}"), Operation(method="GET", path="/group_snapshots/detail")],
    ),
    base.APIRule(
        name="group:reset_group_snapshot_status",
        check_str=("rule:admin_api"),
        description="Reset status of group snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/group_snapshots/{g_snapshot_id}/action (reset_status)")],
    ),
    base.APIRule(
        name="group:delete",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete group.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups/{group_id}/action (delete)")],
    ),
    base.APIRule(
        name="group:reset_status",
        check_str=("rule:admin_api"),
        description="Reset status of group.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups/{group_id}/action (reset_status)")],
    ),
    base.APIRule(
        name="group:enable_replication",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Enable replication.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups/{group_id}/action (enable_replication)")],
    ),
    base.APIRule(
        name="group:disable_replication",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Disable replication.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups/{group_id}/action (disable_replication)")],
    ),
    base.APIRule(
        name="group:failover_replication",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Fail over replication.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups/{group_id}/action (failover_replication)")],
    ),
    base.APIRule(
        name="group:list_replication_targets",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="List failover replication.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/groups/{group_id}/action (list_replication_targets)")],
    ),
    base.APIRule(
        name="volume_extension:qos_specs_manage:get_all",
        check_str=("rule:admin_api"),
        description="List qos specs or list all associations.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/qos-specs"), Operation(method="GET", path="/qos-specs/{qos_id}/associations")],
    ),
    base.APIRule(
        name="volume_extension:qos_specs_manage:get",
        check_str=("rule:admin_api"),
        description="Show qos specs.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/qos-specs/{qos_id}")],
    ),
    base.APIRule(
        name="volume_extension:qos_specs_manage:create",
        check_str=("rule:admin_api"),
        description="Create qos specs.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/qos-specs")],
    ),
    base.APIRule(
        name="volume_extension:qos_specs_manage:update",
        check_str=("rule:admin_api"),
        description="Update qos specs (including updating association).",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/qos-specs/{qos_id}"), Operation(method="GET", path="/qos-specs/{qos_id}/disassociate_all"), Operation(method="GET", path="/qos-specs/{qos_id}/associate"), Operation(method="GET", path="/qos-specs/{qos_id}/disassociate")],
    ),
    base.APIRule(
        name="volume_extension:qos_specs_manage:delete",
        check_str=("rule:admin_api"),
        description="delete qos specs or unset one specified qos key.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/qos-specs/{qos_id}"), Operation(method="PUT", path="/qos-specs/{qos_id}/delete_keys")],
    ),
    base.APIRule(
        name="volume_extension:quota_classes:get",
        check_str=("rule:admin_api"),
        description="Show project quota class.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/os-quota-class-sets/{project_id}")],
    ),
    base.APIRule(
        name="volume_extension:quota_classes:update",
        check_str=("rule:admin_api"),
        description="Update project quota class.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/os-quota-class-sets/{project_id}")],
    ),
    base.APIRule(
        name="volume_extension:quotas:show",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show project quota (including usage and default).",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/os-quota-sets/{project_id}"), Operation(method="GET", path="/os-quota-sets/{project_id}/default"), Operation(method="GET", path="/os-quota-sets/{project_id}?usage=True")],
    ),
    base.APIRule(
        name="volume_extension:quotas:update",
        check_str=("rule:admin_api"),
        description="Update project quota.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/os-quota-sets/{project_id}")],
    ),
    base.APIRule(
        name="volume_extension:quotas:delete",
        check_str=("rule:admin_api"),
        description="Delete project quota.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/os-quota-sets/{project_id}")],
    ),
    base.APIRule(
        name="volume_extension:capabilities",
        check_str=("rule:admin_api"),
        description="Show backend capabilities.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/capabilities/{host_name}")],
    ),
    base.APIRule(
        name="volume_extension:services:index",
        check_str=("rule:admin_api"),
        description="List all services.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/os-services")],
    ),
    base.APIRule(
        name="volume_extension:services:update",
        check_str=("rule:admin_api"),
        description="Update service, including failover_host, thaw, freeze, disable, enable, set-log and get-log actions.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/os-services/{action}")],
    ),
    base.APIRule(
        name="volume:freeze_host",
        check_str=("rule:admin_api"),
        description="Freeze a backend host.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/os-services/freeze")],
    ),
    base.APIRule(
        name="volume:thaw_host",
        check_str=("rule:admin_api"),
        description="Thaw a backend host.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/os-services/thaw")],
    ),
    base.APIRule(
        name="volume:failover_host",
        check_str=("rule:admin_api"),
        description="Failover a backend host.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/os-services/failover_host")],
    ),
    base.APIRule(
        name="scheduler_extension:scheduler_stats:get_pools",
        check_str=("rule:admin_api"),
        description="List all backend pools.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/scheduler-stats/get_pools")],
    ),
    base.APIRule(
        name="volume_extension:hosts",
        check_str=("rule:admin_api"),
        description="List, update or show hosts for a project.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/os-hosts"), Operation(method="PUT", path="/os-hosts/{host_name}"), Operation(method="GET", path="/os-hosts/{host_id}")],
    ),
    base.APIRule(
        name="limits_extension:used_limits",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show limits with used limit attributes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/limits")],
    ),
    base.APIRule(
        name="volume_extension:list_manageable",
        check_str=("rule:admin_api"),
        description="List (in detail) of volumes which are available to manage.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/manageable_volumes"), Operation(method="GET", path="/manageable_volumes/detail")],
    ),
    base.APIRule(
        name="volume_extension:volume_manage",
        check_str=("rule:admin_api"),
        description="Manage existing volumes.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/manageable_volumes")],
    ),
    base.APIRule(
        name="volume_extension:volume_unmanage",
        check_str=("rule:admin_api"),
        description="Stop managing a volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-unmanage)")],
    ),
    base.APIRule(
        name="volume_extension:type_create",
        check_str=("rule:admin_api"),
        description="Create volume type.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/types")],
    ),
    base.APIRule(
        name="volume_extension:type_update",
        check_str=("rule:admin_api"),
        description="Update volume type.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/types")],
    ),
    base.APIRule(
        name="volume_extension:type_delete",
        check_str=("rule:admin_api"),
        description="Delete volume type.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/types")],
    ),
    base.APIRule(
        name="volume_extension:type_get",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Get one specific volume type.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}")],
    ),
    base.APIRule(
        name="volume_extension:type_get_all",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List volume types.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/")],
    ),
    base.APIRule(
        name="volume_extension:access_types_extra_specs",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Include the volume type's extra_specs attribute in the volume type list or show requests.  The ability to make these calls is governed by other policies.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}"), Operation(method="GET", path="/types")],
    ),
    base.APIRule(
        name="volume_extension:access_types_qos_specs_id",
        check_str=("rule:admin_api"),
        description="Include the volume type's QoS specifications ID attribute in the volume type list or show requests.  The ability to make these calls is governed by other policies.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}"), Operation(method="GET", path="/types")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_encryption:create",
        check_str=("rule:admin_api"),
        description="Create volume type encryption.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/types/{type_id}/encryption")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_encryption:get",
        check_str=("rule:admin_api"),
        description="Show a volume type's encryption type, show an encryption specs item.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}/encryption"), Operation(method="GET", path="/types/{type_id}/encryption/{key}")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_encryption:update",
        check_str=("rule:admin_api"),
        description="Update volume type encryption.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/types/{type_id}/encryption/{encryption_id}")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_encryption:delete",
        check_str=("rule:admin_api"),
        description="Delete volume type encryption.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/types/{type_id}/encryption/{encryption_id}")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_access",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Adds the boolean field 'os-volume-type-access:is_public' to the responses for these API calls.  The ability to make these calls is governed by other policies.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types"), Operation(method="GET", path="/types/{type_id}"), Operation(method="POST", path="/types")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_access:addProjectAccess",
        check_str=("rule:admin_api"),
        description="Add volume type access for project.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/types/{type_id}/action (addProjectAccess)")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_access:removeProjectAccess",
        check_str=("rule:admin_api"),
        description="Remove volume type access for project.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/types/{type_id}/action (removeProjectAccess)")],
    ),
    base.APIRule(
        name="volume_extension:volume_type_access:get_all_for_type",
        check_str=("rule:admin_api"),
        description="List private volume type access detail, that is, list the projects that have access to this volume type.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}/os-volume-type-access")],
    ),
    base.APIRule(
        name="volume:extend",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Extend a volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-extend)")],
    ),
    base.APIRule(
        name="volume:extend_attached_volume",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Extend a attached volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-extend)")],
    ),
    base.APIRule(
        name="volume:revert_to_snapshot",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Revert a volume to a snapshot.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (revert)")],
    ),
    base.APIRule(
        name="volume_extension:volume_admin_actions:reset_status",
        check_str=("rule:admin_api"),
        description="Reset status of a volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-reset_status)")],
    ),
    base.APIRule(
        name="volume:retype",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Retype a volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-retype)")],
    ),
    base.APIRule(
        name="volume:update_readonly_flag",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update a volume's readonly flag.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-update_readonly_flag)")],
    ),
    base.APIRule(
        name="volume_extension:volume_admin_actions:force_delete",
        check_str=("rule:admin_api"),
        description="Force delete a volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-force_delete)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:upload_public",
        check_str=("rule:admin_api"),
        description="Upload a volume to image with public visibility.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-volume_upload_image)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:upload_image",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Upload a volume to image.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-volume_upload_image)")],
    ),
    base.APIRule(
        name="volume_extension:volume_admin_actions:force_detach",
        check_str=("rule:admin_api"),
        description="Force detach a volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-force_detach)")],
    ),
    base.APIRule(
        name="volume_extension:volume_admin_actions:migrate_volume",
        check_str=("rule:admin_api"),
        description="migrate a volume to a specified host.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-migrate_volume)")],
    ),
    base.APIRule(
        name="volume_extension:volume_admin_actions:migrate_volume_completion",
        check_str=("rule:admin_api"),
        description="Complete a volume migration.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-migrate_volume_completion)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:initialize_connection",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Initialize volume attachment.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-initialize_connection)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:terminate_connection",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Terminate volume attachment.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-terminate_connection)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:roll_detaching",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Roll back volume status to 'in-use'.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-roll_detaching)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:reserve",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Mark volume as reserved.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-reserve)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:unreserve",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Unmark volume as reserved.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-unreserve)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:begin_detaching",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Begin detach volumes.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-begin_detaching)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:attach",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Add attachment metadata.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-attach)")],
    ),
    base.APIRule(
        name="volume_extension:volume_actions:detach",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Clear attachment metadata.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-detach)")],
    ),
    base.APIRule(
        name="volume:reimage",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Reimage a volume in 'available' or 'error' status.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-reimage)")],
    ),
    base.APIRule(
        name="volume:reimage_reserved",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Reimage a volume in 'reserved' status.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-reimage)")],
    ),
    base.APIRule(
        name="volume:get_all_transfers",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List volume transfer.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/os-volume-transfer"), Operation(method="GET", path="/os-volume-transfer/detail"), Operation(method="GET", path="/volume_transfers"), Operation(method="GET", path="/volume-transfers/detail")],
    ),
    base.APIRule(
        name="volume:create_transfer",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create a volume transfer.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/os-volume-transfer"), Operation(method="POST", path="/volume_transfers")],
    ),
    base.APIRule(
        name="volume:get_transfer",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show one specified volume transfer.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/os-volume-transfer/{transfer_id}"), Operation(method="GET", path="/volume-transfers/{transfer_id}")],
    ),
    base.APIRule(
        name="volume:accept_transfer",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Accept a volume transfer.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/os-volume-transfer/{transfer_id}/accept"), Operation(method="POST", path="/volume-transfers/{transfer_id}/accept")],
    ),
    base.APIRule(
        name="volume:delete_transfer",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete volume transfer.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/os-volume-transfer/{transfer_id}"), Operation(method="DELETE", path="/volume-transfers/{transfer_id}")],
    ),
    base.APIRule(
        name="volume:get_volume_metadata",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show volume's metadata or one specified metadata with a given key.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/{volume_id}/metadata"), Operation(method="GET", path="/volumes/{volume_id}/metadata/{key}"), Operation(method="POST", path="/volumes/{volume_id}/action  (os-show_image_metadata)")],
    ),
    base.APIRule(
        name="volume:create_volume_metadata",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create volume metadata.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/metadata")],
    ),
    base.APIRule(
        name="volume:update_volume_metadata",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Replace a volume's metadata dictionary or update a single metadatum with a given key.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/volumes/{volume_id}/metadata"), Operation(method="PUT", path="/volumes/{volume_id}/metadata/{key}")],
    ),
    base.APIRule(
        name="volume:delete_volume_metadata",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete a volume's metadatum with the given key.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/volumes/{volume_id}/metadata/{key}")],
    ),
    base.APIRule(
        name="volume_extension:volume_image_metadata:show",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Include a volume's image metadata in volume detail responses.  The ability to make these calls is governed by other policies.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/detail"), Operation(method="GET", path="/volumes/{volume_id}")],
    ),
    base.APIRule(
        name="volume_extension:volume_image_metadata:set",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Set image metadata for a volume",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-set_image_metadata)")],
    ),
    base.APIRule(
        name="volume_extension:volume_image_metadata:remove",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Remove specific image metadata from a volume",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-unset_image_metadata)")],
    ),
    base.APIRule(
        name="volume:update_volume_admin_metadata",
        check_str=("rule:admin_api"),
        description="Update volume admin metadata. This permission is required to complete these API calls, though the ability to make these calls is governed by other policies.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes/{volume_id}/action (os-update_readonly_flag)"), Operation(method="POST", path="/volumes/{volume_id}/action (os-attach)")],
    ),
    base.APIRule(
        name="volume_extension:types_extra_specs:index",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List type extra specs.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}/extra_specs")],
    ),
    base.APIRule(
        name="volume_extension:types_extra_specs:create",
        check_str=("rule:admin_api"),
        description="Create type extra specs.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/types/{type_id}/extra_specs")],
    ),
    base.APIRule(
        name="volume_extension:types_extra_specs:show",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show one specified type extra specs.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types/{type_id}/extra_specs/{extra_spec_key}")],
    ),
    base.APIRule(
        name="volume_extension:types_extra_specs:read_sensitive",
        check_str=("rule:admin_api"),
        description="Include extra_specs fields that may reveal sensitive information about the deployment that should not be exposed to end users in various volume-type responses that show extra_specs. The ability to make these calls is governed by other policies.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/types"), Operation(method="GET", path="/types/{type_id}"), Operation(method="GET", path="/types/{type_id}/extra_specs"), Operation(method="GET", path="/types/{type_id}/extra_specs/{extra_spec_key}")],
    ),
    base.APIRule(
        name="volume_extension:types_extra_specs:update",
        check_str=("rule:admin_api"),
        description="Update type extra specs.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/types/{type_id}/extra_specs/{extra_spec_key}")],
    ),
    base.APIRule(
        name="volume_extension:types_extra_specs:delete",
        check_str=("rule:admin_api"),
        description="Delete type extra specs.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/types/{type_id}/extra_specs/{extra_spec_key}")],
    ),
    base.APIRule(
        name="volume:create",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes")],
    ),
    base.APIRule(
        name="volume:create_from_image",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create volume from image.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes")],
    ),
    base.APIRule(
        name="volume:get",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show volume.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/{volume_id}")],
    ),
    base.APIRule(
        name="volume:get_all",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List volumes or get summary of volumes.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes"), Operation(method="GET", path="/volumes/detail"), Operation(method="GET", path="/volumes/summary")],
    ),
    base.APIRule(
        name="volume:update",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Update volume or update a volume's bootable status.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/volumes"), Operation(method="POST", path="/volumes/{volume_id}/action (os-set_bootable)")],
    ),
    base.APIRule(
        name="volume:delete",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Delete volume.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/volumes/{volume_id}")],
    ),
    base.APIRule(
        name="volume:force_delete",
        check_str=("rule:admin_api"),
        description="Force Delete a volume.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/volumes/{volume_id}")],
    ),
    base.APIRule(
        name="volume_extension:volume_host_attribute",
        check_str=("rule:admin_api"),
        description="List or show volume with host attribute.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/{volume_id}"), Operation(method="GET", path="/volumes/detail")],
    ),
    base.APIRule(
        name="volume_extension:volume_tenant_attribute",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="List or show volume with tenant attribute.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/{volume_id}"), Operation(method="GET", path="/volumes/detail")],
    ),
    base.APIRule(
        name="volume_extension:volume_mig_status_attribute",
        check_str=("rule:admin_api"),
        description="List or show volume with migration status attribute.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/{volume_id}"), Operation(method="GET", path="/volumes/detail")],
    ),
    base.APIRule(
        name="volume_extension:volume_encryption_metadata",
        check_str=("rule:xena_system_admin_or_project_reader"),
        description="Show volume's encryption metadata.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/volumes/{volume_id}/encryption"), Operation(method="GET", path="/volumes/{volume_id}/encryption/{encryption_key}")],
    ),
    base.APIRule(
        name="volume:multiattach",
        check_str=("rule:xena_system_admin_or_project_member"),
        description="Create multiattach capable volume.",
        scope_types=["project"],
        operations=[Operation(method="POST", path="/volumes")],
    ),
    base.APIRule(
        name="volume_extension:default_set_or_update",
        check_str=("rule:admin_api"),
        description="Set or update default volume type.",
        scope_types=["project"],
        operations=[Operation(method="PUT", path="/default-types")],
    ),
    base.APIRule(
        name="volume_extension:default_get",
        check_str=("rule:admin_api"),
        description="Get default types.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/default-types/{project-id}")],
    ),
    base.APIRule(
        name="volume_extension:default_get_all",
        check_str=("rule:admin_api"),
        description="Get all default types. WARNING: Changing this might open up too much information regarding cloud deployment.",
        scope_types=["project"],
        operations=[Operation(method="GET", path="/default-types/")],
    ),
    base.APIRule(
        name="volume_extension:default_unset",
        check_str=("rule:admin_api"),
        description="Unset default type.",
        scope_types=["project"],
        operations=[Operation(method="DELETE", path="/default-types/{project-id}")],
    ),
)

__all__ = ("list_rules",)
