// Copyright 2021 99cloud
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import fs from 'fs';
import { getBase64, getText } from './file';

describe('test util/file.js', () => {
  it('getBase64', async () => {
    const result = await fs.readFileSync('src/asset/image/animbus.png');
    const base64 = await getBase64(new File([result], 'animbus.png'));
    expect(base64).toBe(
      `data:;base64,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`
    );
  });

  it('getText', async () => {
    const originTxt = await fs.readFileSync('src/asset/template/index.html', {
      encoding: 'utf-8',
    });
    const txt = await getText(new File([originTxt], 'index.html'));
    expect(txt).toBe(originTxt);
  });
});
