// Copyright (c) 2010-2012 OpenStack Foundation
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
// implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"encoding/binary"
	"encoding/hex"
	"errors"
	"github.com/sirupsen/logrus"
)

// Encodes a volume key.
func EncodeVolumeKey(index uint32) (val []byte) {
	buf := make([]byte, binary.MaxVarintLen32)

	n := binary.PutUvarint(buf, uint64(index))

	val = buf[:n]
	log.WithFields(logrus.Fields{"value": val}).Debug("encoded volume key")
	return
}

func DecodeVolumeKey(val []byte) (index uint32, err error) {
	index32, n := binary.Uvarint(val)
	if n <= 0 {
		err = errors.New("failed to decode index")
		return
	}

	index = uint32(index32)
	return
}

// volumeType is an int32 to match the type generated by protobuf for enums
func EncodeVolumeValue(partition int64, volumeType int32, nextOffset, usedSpace, state int64) (val []byte) {
	buf := make([]byte, binary.MaxVarintLen64*5)
	bufLen := 0

	n := binary.PutUvarint(buf, uint64(partition))
	bufLen += n

	n = binary.PutUvarint(buf[bufLen:], uint64(volumeType))
	bufLen += n

	n = binary.PutUvarint(buf[bufLen:], uint64(nextOffset))
	bufLen += n

	n = binary.PutUvarint(buf[bufLen:], uint64(usedSpace))
	bufLen += n

	n = binary.PutUvarint(buf[bufLen:], uint64(state))
	bufLen += n

	val = buf[:bufLen]
	log.WithFields(logrus.Fields{"value": val}).Debug("encoded volume value")
	return
}

func DecodeVolumeValue(val []byte) (partition int64, volumeType int32, nextOffset, usedSpace, state int64, err error) {
	position := 0

	partition64, n := binary.Uvarint(val)
	if n <= 0 {
		err = errors.New("failed to decode partition")
	}
	position += n

	volumeType64, n := binary.Uvarint(val[position:])
	if n <= 0 {
		err = errors.New("failed to decode nextOffset")
	}
	position += n

	nextOffset64, n := binary.Uvarint(val[position:])
	if n <= 0 {
		err = errors.New("failed to decode nextOffset")
	}
	position += n

	usedSpace64, n := binary.Uvarint(val[position:])
	if n <= 0 {
		err = errors.New("failed to decode usedSpace")
	}
	position += n

	state64, n := binary.Uvarint(val[position:])
	if n <= 0 {
		err = errors.New("failed to decode state")
	}

	partition = int64(partition64)
	volumeType = int32(volumeType64)
	nextOffset = int64(nextOffset64)
	usedSpace = int64(usedSpace64)
	state = int64(state64)
	return
}

// Encodes an object key. the key is the md5 hash string + the filename.
// Turn the 32 characters hash to a 16 byte array. Leave the filename as
// is for now. We could gain more space encoding the filename (varint timestamp + encoded file extension),
// but there are special cases to handle (the "delta")
func EncodeObjectKey(key []byte) ([]byte, error) {
	var err error

	if len(key) < 32 {
		err = errors.New("object key len < 32, cannot encode")
		return nil, err
	}

	dst := make([]byte, 16+len(key[32:]))
	n, err := hex.Decode(dst, key[:32])
	if err != nil {
		err = errors.New("failed to encode object hash")
		return dst, err
	}

	if n != 16 {
		err = errors.New("encoded object hash is not 16 bytes long")
		return dst, err
	}

	// copy the filename
	copy(dst[16:], key[32:])

	return dst, err
}

// Decodes object key
// This is the most called function of the project. The profiler showed that it did
// the most allocations on the heap (after cgo, which is something else to look at..)
// Now expect the buffer from the caller.
// decodedKey size must be 32+len(encodedKey[16:]), because we will decode the 16 bytes
// hash to a 32 bytes string, with the rest unchanged.
func DecodeObjectKey(encodedKey []byte, decodedKey []byte) error {
	if len(encodedKey) < 16 {
		err := errors.New("DecodeObjectKey called with encodedKey of len < 16")
		return err
	}
	if len(decodedKey) < 32+len(encodedKey[16:]) {
		err := errors.New("DecodeObjectKey called with decodedKey too small")
		return err
	}

	hex.Encode(decodedKey, encodedKey[:16])
	copy(decodedKey[32:], encodedKey[16:])

	return nil
}

// Encodes an object file value.
func EncodeObjectValue(volumeIndex uint32, offset uint64) (val []byte) {
	buf := make([]byte, binary.MaxVarintLen64*2)
	bufLen := 0

	n := binary.PutUvarint(buf, uint64(volumeIndex))
	bufLen += n

	n = binary.PutUvarint(buf[n:], offset)
	bufLen += n

	val = buf[:bufLen]
	log.WithFields(logrus.Fields{"value": val}).Debug("encoded object value")
	return
}

func DecodeObjectValue(val []byte) (volumeIndex uint32, offset uint64, err error) {
	log.WithFields(logrus.Fields{"value": val}).Debug("Decode object value")
	volumeIndex64, n := binary.Uvarint(val)
	if n <= 0 {
		log.WithFields(logrus.Fields{"index": n}).Debug("failed to decode volumeIndex")
		err = errors.New("failed to decode volumeIndex")
	}

	offset64, n := binary.Uvarint(val[n:])
	if n <= 0 {
		log.WithFields(logrus.Fields{"offset": n}).Debug("failed to decode offset")
		err = errors.New("failed to decode offset")
		return
	}

	volumeIndex = uint32(volumeIndex64)
	offset = uint64(offset64)
	return
}
