# Copyright (c) 2019 NTT DATA
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import base64
import datetime
import fixtures
import json
import os
import re
import requests
import shutil
import sys
import unittest
from unittest import mock
import urllib
from urllib import error as urlerr
import yaml

from glance_store import exceptions as store_exceptions
from oslo_config import cfg
from oslo_log import log as logging
from oslo_serialization import jsonutils

from tacker import auth
from tacker.common import coordination
from tacker.common import csar_utils
from tacker.common import driver_manager
from tacker.common import exceptions
from tacker.common.rpc import BackingOffClient
from tacker.conductor import conductor_server
import tacker.conf
from tacker import context
from tacker import context as t_context
from tacker.db.db_sqlalchemy import models
from tacker.glance_store import store as glance_store
from tacker import objects
from tacker.objects import fields
from tacker.plugins.common import constants
from tacker.tests import constants as test_constants
from tacker.tests.unit import base as unit_base
from tacker.tests.unit.conductor import fakes
from tacker.tests.unit.db.base import SqlTestCase
from tacker.tests.unit.db import utils as db_utils
from tacker.tests.unit.objects import fakes as fake_obj
from tacker.tests.unit.vnflcm import fakes as vnflcm_fakes
from tacker.tests.unit.vnfm.infra_drivers.openstack.fixture_data import client
from tacker.tests.unit.vnfm.infra_drivers.openstack.fixture_data import \
    fixture_data_utils as fd_utils
import tacker.tests.unit.vnfm.test_nfvo_client as nfvo_client
from tacker.tests import utils
from tacker.tests import uuidsentinel
from tacker.vnfm import nfvo_client as test_nfvo_client
from tacker.vnfm import vim_client


CONF = tacker.conf.CONF


class FakeVnfLcmDriver(mock.Mock):
    def modify_vnf(self, context, vnf_lcm_opoccs, body_data,
                   vnfd_pkg_data, vnfd_id):
        return datetime.datetime(
            1900, 1, 1, 1, 1, 1, tzinfo=datetime.timezone.utc)


class FakeVNFMPlugin(mock.Mock):
    pass


class MockResponse:
    def __init__(self, json_data):
        self.json_data = json_data

    def json(self):
        return self.json_data


class TestConductor(SqlTestCase, unit_base.FixturedTestCase):
    client_fixture_class = client.ClientFixture
    sdk_connection_fixure_class = client.SdkConnectionFixture

    def setUp(self):
        super(TestConductor, self).setUp()
        self.addCleanup(mock.patch.stopall)
        self.context = context.get_admin_context()
        self._mock_vnflcm_driver()
        self._mock_vnfm_plugin()
        self.conductor = conductor_server.Conductor('host')
        self.vnf_package = self._create_vnf_package()
        self.instance_uuid = uuidsentinel.instance_id
        self.temp_dir = self.useFixture(fixtures.TempDir()).path
        self.body_data = self._create_body_data()
        self.vnf_lcm_opoccs = self._create_vnf_lcm_opoccs()
        self.vnfd_pkg_data = self._create_vnfd_pkg_data()
        cfg.CONF.set_override('retry_wait', 0, group='vnf_lcm')
        self.addCleanup(cfg.CONF.clear_override, 'retry_wait', group='vnf_lcm')
        self.cctxt_mock = mock.MagicMock()

    def _nfv_sample(self, name):
        return utils.test_etc_sample("etsi/nfv", name)

    def _mock_vnfm_plugin(self):
        self.vnfm_plugin = mock.Mock(wraps=FakeVNFMPlugin())
        fake_vnfm_plugin = mock.Mock()
        fake_vnfm_plugin.return_value = self.vnfm_plugin
        self._mock(
            'tacker.vnfm.plugin.VNFMPlugin', fake_vnfm_plugin)

    def _mock_vnflcm_driver(self):
        self.vnflcm_driver = mock.Mock(wraps=FakeVnfLcmDriver())
        fake_vnflcm_driver = mock.Mock()
        fake_vnflcm_driver.return_value = self.vnflcm_driver
        self._mock(
            'tacker.vnflcm.vnflcm_driver.VnfLcmDriver', fake_vnflcm_driver)

    def _create_vnf_package(self):
        vnfpkgm = objects.VnfPackage(context=self.context,
                                     **fakes.VNF_PACKAGE_DATA)
        vnfpkgm.create()
        return vnfpkgm

    def _create_vnf_package_vnfd(self):
        return fakes.get_vnf_package_vnfd()

    def _create_subscriptions(self, auth_params=None, filter_params=None):
        class DummyLcmSubscription:
            def __init__(self, auth_params=None):
                if auth_params:
                    self.authentication = json.dumps(
                        auth_params).encode()
                if filter_params:
                    self.filter = json.dumps(filter_params).encode()

                self.id = uuidsentinel.lcm_subscription_id
                self.tenant_id = uuidsentinel.tenant_id
                self.callback_uri = 'https://localhost/callback'.encode()

            def __getattr__(self, name):
                try:
                    return object.__getattr__(self, name)
                except AttributeError:
                    return None

        return [DummyLcmSubscription(auth_params)]

    def _create_body_data(self):
        body_data = {}
        body_data['vnf_instance_name'] = "new_instance_name"
        body_data['vnf_instance_description'] = "new_instance_discription"
        body_data['vnfd_id'] = "2c69a161-0000-4b0f-bcf8-391f8fc76600"
        body_data['vnf_configurable_properties'] = {"test": "test_value"}
        body_data['vnfc_info_modifications_delete_ids'] = ["test1"]
        body_data['vnf_pkg_id'] = uuidsentinel.vnf_pkg_id
        body_data['metadata'] = {"test_meta": "test_meta_value"}
        body_data['vim_connection_info'] = [
            {
                "id": "d24f9796-a8e9-4cb0-85ce-5920dc100001",
                "vim_id": "85355a9f-9ecf-481c-bf65-451677a9ee82",
                "vim_type": "test",
                "interface_info": {},
                "access_info": {},
                "extra": {"test": "test1"}
            }
        ]

        return body_data

    def _create_vnf_lcm_opoccs(self):
        vnf_lcm_opoccs = {
            'vnf_instance_id': uuidsentinel.vnf_instance_id,
            'id': uuidsentinel.id,
            'state_entered_time': datetime.datetime(
                1900, 1, 1, 1, 1, 1,
                tzinfo=datetime.timezone.utc),
            'operationParams': 'operationParams',
            'tenant_id': uuidsentinel.tenant_id
        }
        return vnf_lcm_opoccs

    def _create_vnfd_pkg_data(self):
        vnfd_pkg_data = {}
        vnfd_pkg_data['vnf_provider'] = fakes.return_vnf_package_vnfd().get(
            'vnf_provider')
        vnfd_pkg_data['vnf_product_name'] =\
            fakes.return_vnf_package_vnfd().get('vnf_product_name')
        vnfd_pkg_data['vnf_software_version'] =\
            fakes.return_vnf_package_vnfd().get('vnf_software_version')
        vnfd_pkg_data['vnfd_version'] = fakes.return_vnf_package_vnfd().get(
            'vnfd_version')
        vnfd_pkg_data['package_uuid'] = fakes.return_vnf_package_vnfd().get(
            'package_uuid')
        return vnfd_pkg_data

    def assert_auth_basic(
            self,
            acutual_request,
            expected_user_name,
            expected_password):
        actual_auth = acutual_request._request.headers.get("Authorization")
        expected_auth = base64.b64encode(
            '{}:{}'.format(
                expected_user_name,
                expected_password).encode('utf-8')).decode()
        self.assertEqual("Basic " + expected_auth, actual_auth)

    def assert_auth_client_credentials(self, acutual_request, expected_token):
        actual_auth = acutual_request._request.headers.get(
            "Authorization")
        self.assertEqual("Bearer " + expected_token, actual_auth)

    @mock.patch.object(BackingOffClient, 'prepare')
    @mock.patch.object(glance_store, 'store_csar')
    @mock.patch.object(conductor_server, 'revert_upload_vnf_package')
    def test_upload_vnf_package_from_uri(self,
                                         mock_revert,
                                         mock_store,
                                         mock_prepare):
        mock_prepare.return_value = self.cctxt_mock
        address_information = "http://test.zip"
        mock_store.return_value = (
            'location', 0, 'checksum', 'multihash', 'loc_meta')
        self.conductor.upload_vnf_package_from_uri(self.context,
                                                   self.vnf_package,
                                                   address_information,
                                                   user_name=None,
                                                   password=None)
        mock_store.assert_called()
        mock_prepare.assert_called()
        self.cctxt_mock.cast.assert_called_once_with(
            self.context, 'load_csar_data', vnf_package=self.vnf_package)
        self.assertEqual('multihash', self.vnf_package.hash)
        self.assertEqual('location', self.vnf_package.location_glance_store)

    @mock.patch.object(csar_utils, 'load_csar_data')
    @mock.patch.object(glance_store, 'load_csar')
    @mock.patch.object(objects.vnf_package.VnfPackagesList, 'get_by_filters')
    def test_init_csar_files(self, mock_pkg_list,
                             mock_load_csar, mock_load_csar_data):
        fake_csar = os.path.join(self.temp_dir, self.vnf_package.id)
        self.vnf_package.location = fake_csar
        mock_pkg_list.return_value = [self.vnf_package]
        mock_load_csar.return_value = True
        mock_load_csar_data.return_value = (mock.ANY, mock.ANY, mock.ANY)

        self.conductor.init_csar_files()

        mock_load_csar.assert_called()
        mock_load_csar_data.assert_called()

    @mock.patch.object(csar_utils, 'load_csar_data')
    @mock.patch.object(glance_store, 'load_csar')
    @mock.patch.object(objects.vnf_package.VnfPackagesList, 'get_by_filters')
    def test_init_csar_files_exception(self, mock_pkg_list,
                                       mock_load_csar, mock_load_csar_data):
        fake_csar = os.path.join(self.temp_dir, self.vnf_package.id)
        self.vnf_package.location = fake_csar
        mock_pkg_list.return_value = [self.vnf_package]
        mock_load_csar.return_value = True
        mock_load_csar_data.side_effect = exceptions.InvalidCSAR

        self.assertRaises(exceptions.TackerException,
                          self.conductor.init_csar_files)

    @mock.patch.object(BackingOffClient, 'prepare')
    @mock.patch.object(glance_store, 'delete_csar')
    def test_delete_vnf_package(self, mock_delete_csar, mock_prepare):
        self.vnf_package.onboarding_state = 'ONBOARDED'
        mock_prepare.return_value = self.cctxt_mock
        self.conductor.delete_vnf_package(self.context, self.vnf_package)

        mock_delete_csar.assert_called()
        mock_prepare.assert_called()
        self.cctxt_mock.cast.assert_called_once_with(
            self.context, 'delete_csar', vnf_package=self.vnf_package)

    def test_get_vnf_package_vnfd_with_tosca_meta_file_in_csar(self):
        fake_csar = fakes.create_fake_csar_dir(self.vnf_package.id,
                                               self.temp_dir)
        expected_data = fakes.get_expected_vnfd_data()
        result = self.conductor.get_vnf_package_vnfd(self.context,
                                                     self.vnf_package)
        self.assertEqual(expected_data, result)
        shutil.rmtree(fake_csar)

    def test_get_vnf_package_vnfd_with_single_yaml_csar(self):
        fake_csar = fakes.create_fake_csar_dir(
            self.vnf_package.id, self.temp_dir, csar_without_tosca_meta=True)
        result = self.conductor.get_vnf_package_vnfd(self.context,
                                                     self.vnf_package)
        # only one key present in the result shows that it contains only one
        # yaml file
        self.assertEqual(1, len(result.keys()))
        shutil.rmtree(fake_csar)

    @mock.patch.object(glance_store, 'load_csar')
    def test_get_vnf_package_vnfd_download_from_glance_store(self,
                                                             mock_load_csar):
        fake_csar = os.path.join(self.temp_dir, self.vnf_package.id)
        cfg.CONF.set_override('vnf_package_csar_path', self.temp_dir,
                              group='vnf_package')
        fake_csar_zip, _ = utils.create_csar_with_unique_vnfd_id(
            self._nfv_sample('sample_vnfpkg_tosca_vnfd'))
        mock_load_csar.return_value = fake_csar_zip
        expected_data = fakes.get_expected_vnfd_data(zip_file=fake_csar_zip)
        result = self.conductor.get_vnf_package_vnfd(self.context,
                                                     self.vnf_package)
        self.assertEqual(expected_data, result)
        shutil.rmtree(fake_csar)
        os.remove(fake_csar_zip)

    @mock.patch.object(glance_store, 'load_csar')
    def test_get_vnf_package_vnfd_exception_from_glance_store(self,
                                                              mock_load_csar):
        mock_load_csar.side_effect = store_exceptions.NotFound
        self.assertRaises(exceptions.FailedToGetVnfdData,
                          self.conductor.get_vnf_package_vnfd, self.context,
                          self.vnf_package)

    @mock.patch.object(conductor_server.Conductor, '_read_vnfd_files')
    def test_get_vnf_package_vnfd_exception_from_read_vnfd_files(
            self, mock_read_vnfd_files):
        fake_csar = fakes.create_fake_csar_dir(self.vnf_package.id,
                                               self.temp_dir)
        mock_read_vnfd_files.side_effect = yaml.YAMLError
        self.assertRaises(exceptions.FailedToGetVnfdData,
                          self.conductor.get_vnf_package_vnfd, self.context,
                          self.vnf_package)
        shutil.rmtree(fake_csar)

    def _create_and_upload_vnf_package(self):
        vnf_package = objects.VnfPackage(context=self.context,
                                         **fake_obj.vnf_package_data)
        vnf_package.create()

        vnf_pack_vnfd = fake_obj.get_vnf_package_vnfd_data(
            vnf_package.id, uuidsentinel.vnfd_id)

        vnf_pack_vnfd_obj = objects.VnfPackageVnfd(
            context=self.context, **vnf_pack_vnfd)
        vnf_pack_vnfd_obj.create()

        vnf_package.onboarding_state = "ONBOARDED"
        vnf_package.save()

        return vnf_pack_vnfd_obj

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance(self, mock_vnf_by_id,
            mock_des, mock_vnfd_dict,
            mock_get_lock, mock_save,
            mock_build_info, mock_change_vnf_status,
            mock_update_vnf_attributes):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, mock.ANY, vnf_dict, instantiate_vnf_req)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        mock_change_vnf_status. \
            assert_called_once_with(self.context, vnf_instance.id,
                                    mock.ANY, 'PENDING_CREATE')
        mock_update_vnf_attributes.assert_called_once()

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.vnflcm.utils._get_affected_resources')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_without_virtual_storage(
            self, mock_vnf_by_id,
            mock_des, mock_vnfd_dict,
            mock_get_lock, mock_save,
            mock_build_info, mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_affected_resources,
            mock_send_notification):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        mock_affected_resources.return_value = {}
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, mock.ANY, vnf_dict, instantiate_vnf_req)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        mock_change_vnf_status. \
            assert_called_once_with(self.context, vnf_instance.id,
                                    mock.ANY, 'PENDING_CREATE')
        mock_update_vnf_attributes.assert_called_once()
        self.assertIsNone(
            mock_send_notification.call_args[0][1].get("affectedVnfcs"))
        self.assertIsNone(
            mock_send_notification.call_args[0][1].get(
                "affectedVirtualStorages"))
        self.assertIsNone(
            mock_send_notification.call_args[0][1].get("affectedVirtualLinks"))

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._check_res_add_remove_rsc')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_grant(self,
            mock_vnf_by_id,
            mock_des,
            mock_vnfd_dict,
            mock_get_lock,
            mock_grants,
            mock_exec,
            mock_save,
            mock_check,
            mock_build_info,
            mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_send_notification):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = (
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data))

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
            flavour=instantiate_vnf_req.flavour_id,
            vnfd_name="etsi_vnfd_group")
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}
        resAddResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa0',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa1',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['vimAssets'] = vimAssets
        mock_grants.return_value = MockResponse(json_data=grant_dict)
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
            instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(mock_send_notification.call_count, 2)
        self.assertEqual(mock_change_vnf_status.call_count, 1)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._check_res_add_remove_rsc')
    @mock.patch.object(objects.VnfLcmOpOcc, 'save')
    @mock.patch.object(conductor_server.Conductor, '_get_grant_execute')
    @mock.patch.object(test_nfvo_client.GrantRequest, 'grants')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, 'get_by_id')
    def test_instantiate_grant_req_with_affinity(self, mock_vnf_by_id,
            mock_des, mock_vnfd_dict, mock_get_lock, mock_grants, mock_exec,
            mock_save, mock_check, mock_build_info, mock_change_vnf_status,
            mock_update_vnf_attributes, mock_send_notification):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = (
            objects.VnfLcmOpOcc(context=self.context, **lcm_op_occs_data))

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
            flavour=instantiate_vnf_req.flavour_id,
            vnfd_name='etsi_vnfd_affinity')
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnfd_key = f'vnfd_{instantiate_vnf_req.flavour_id}'
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        vimAssets = {
            'computeResourceFlavours': [
                {'vimConnectionId': uuidsentinel.vim_id,
                'vnfdVirtualComputeDescId': 'CDU1',
                'vimFlavourId': 'm1.tiny'}
            ],
            'softwareImages': [
                {'vimConnectionId': uuidsentinel.vim_id,
                 'vnfdSoftwareImageId': 'VDU1',
                 'vimSoftwareImageId': 'cirros'}
            ]
        }
        resAddResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa0',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa1',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        grant_dict = {
            'id': 'c213e465-8220-487e-9464-f79104e81e96',
            'vnfInstanceId': vnf_instance.id,
            'vnfLcmOpOccId': vnf_lcm_op_occs_id,
            'addResources': resAddResource,
            'vimAssets': vimAssets
        }
        mock_grants.return_value = MockResponse(json_data=grant_dict)

        log_name = 'tacker.conductor.conductor_server'
        with self.assertLogs(logger=log_name, level=logging.INFO) as cm:
            self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(mock_send_notification.call_count, 2)
        self.assertEqual(mock_change_vnf_status.call_count, 1)

        chk_log = re.sub(r'INFO:.+:grant start grant_request', '',
            cm.output[0]
        )
        chk_place = json.loads(chk_log)[0]['placementConstraints']
        self.assertEqual(chk_place[0]['affinityOrAntiAffinity'], 'AFFINITY')
        self.assertEqual(chk_place[0]['scope'], 'ZONE')
        self.assertEqual(chk_place[1]['affinityOrAntiAffinity'],
            'ANTI_AFFINITY')
        self.assertEqual(chk_place[1]['scope'], 'NFVI_NODE')

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_with_vim_id_update(self, mock_vnf_by_id,
            mock_des, mock_vnfd_dict,
            mock_get_lock, mock_save,
            mock_build_info, mock_change_vnf_status,
            mock_update_vnf_attributes):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        instantiate_vnf_req.vim_connection_info[0].vim_id = uuidsentinel.vim_id
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        vim_id = instantiate_vnf_req.vim_connection_info[0].vim_id
        mock_update_vnf_attributes.assert_called_with(self.context,
                mock.ANY, mock.ANY, mock.ANY, constants.ACTIVE,
            vim_id=vim_id)

    @mock.patch('tacker.vnflcm.utils._get_vim')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, 'save')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, 'get_by_id')
    def test_instantiate_vnf_instance_without_vim_connection_info(
            self, mock_vnf_by_id, mock_des, mock_vnfd_dict, mock_get_lock,
            mock_save, mock_build_info, mock_change_vnf_status,
            mock_update_vnf_attributes, mock_get_vim):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = (
            objects.VnfLcmOpOcc(context=self.context, **lcm_op_occs_data))
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        instantiate_vnf_req.vim_connection_info = []
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
            flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        default_vim_id = uuidsentinel.vim_id
        mock_get_vim.return_value = {'vim_id': default_vim_id}
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
            instantiate_vnf_req, vnf_lcm_op_occs_id)
        mock_update_vnf_attributes.assert_called_with(self.context,
            mock.ANY, mock.ANY, mock.ANY, constants.ACTIVE,
            vim_id=default_vim_id)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._check_res_add_remove_rsc')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_grant(self,
            mock_vnf_by_id,
            mock_save,
            mock_check,
            mock_des, mock_vnfd_dict, mock_grants,
            mock_exec, mock_get_lock,
            mock_change_vnf_status,
            mock_build_info):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
            **lcm_op_occs_data)
        mock_exec.return_value = True
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}
        resAddResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa0',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa1',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['vimAssets'] = vimAssets
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, vnf_instance, vnf_dict,
            instantiate_vnf_req)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._check_res_add_remove_rsc')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_grant_resource_exception(self,
            mock_vnf_by_id,
            mock_save,
            mock_check,
            mock_des, mock_vnfd_dict, mock_grants,
            mock_exec, mock_get_lock,
            mock_build_info,
            mock_send):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        # mock_package_in_use.return_value = False
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
            **lcm_op_occs_data)
        mock_exec.return_value = True
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}
        resAddResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa0',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['vimAssets'] = vimAssets
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)
        self.assertRaises(exceptions.ValidationError,
            self.conductor.instantiate,
            self.context, vnf_instance, vnf_dict,
            instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._check_res_add_remove_rsc')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_grant_exception(self,
            mock_vnf_by_id,
            mock_save,
            mock_check,
            mock_des, mock_vnfd_dict, mock_grants,
            mock_exec, mock_get_lock,
            mock_build_info,
            mock_send):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        # mock_package_in_use.return_value = False
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
            **lcm_op_occs_data)
        mock_exec.return_value = True
        mock_grants.side_effect = \
            requests.exceptions.HTTPError("MockException")
        self.assertRaises(requests.exceptions.HTTPError,
            self.conductor.instantiate,
            self.context, vnf_instance, vnf_dict,
            instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfPackage, 'is_package_in_use')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_already_instantiated(
            self, mock_vnf_by_id, mock_log, mock_package_in_use, mock_get_lock,
            mock_save):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.instantiate(self.context, vnf_instance,
                                   instantiate_vnf_req,
                                   vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_not_called()
        mock_package_in_use.assert_not_called()
        expected_log = 'Vnf instance %(id)s is already in %(state)s state.'
        mock_log.error.assert_called_once_with(
            expected_log, {
                'id': vnf_instance.id,
                'state': fields.VnfInstanceState.INSTANTIATED})

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfPackage, 'is_package_in_use')
    @mock.patch.object(objects.LccnSubscriptionRequest,
        'vnf_lcm_subscriptions_get')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_with_vnf_package_in_use(
            self,
            mock_vnf_by_id,
            mock_vnf_lcm_subscriptions_get,
            mock_vnf_package_in_use,
            mock_get_lock,
            mock_save):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        m_vnf_lcm_subscriptions = \
            [mock.MagicMock(**fakes.get_vnf_lcm_subscriptions())]
        mock_vnf_lcm_subscriptions_get.return_value = \
            m_vnf_lcm_subscriptions
        mock_vnf_package_in_use.return_value = True
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.instantiate(self.context, vnf_instance,
                                   instantiate_vnf_req,
                                   vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, mock.ANY, instantiate_vnf_req)
        mock_vnf_package_in_use.assert_called_once()

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.LccnSubscriptionRequest,
        'vnf_lcm_subscriptions_get')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch('tacker.vnflcm.utils._get_affected_resources')
    def test_instantiate_vnf_instance_failed_with_exception(
            self, mock_res, mock_vnf_by_id, mock_log,
            mock_des, mock_vnfd_dict,
            mock_vnf_lcm_subscriptions_get,
            mock_get_lock, mock_save,
            mock_send_notification, mock_build_info,
            mock_change_vnf_status,
            mock_update_vnf_attributes):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        m_vnf_lcm_subscriptions = \
            [mock.MagicMock(**fakes.get_vnf_lcm_subscriptions())]
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        mock_vnf_lcm_subscriptions_get.return_value = \
            m_vnf_lcm_subscriptions
        mock_update_vnf_attributes.side_effect = Exception
        mock_res.return_value = {}
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, vnf_instance, vnf_dict, instantiate_vnf_req)
        self.assertIsNone(
            mock_send_notification.call_args[0][1].get("affectedVnfcs"))
        self.assertIsNone(
            mock_send_notification.call_args[0][1].get(
                "affectedVirtualStorages"))
        self.assertIsNone(
            mock_send_notification.call_args[0][1].get("affectedVirtualLinks"))
        mock_change_vnf_status.assert_called_with(self.context,
            vnf_instance.id, mock.ANY, 'ERROR')
        mock_update_vnf_attributes.assert_called_once()

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.LccnSubscriptionRequest,
        'vnf_lcm_subscriptions_get')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch('tacker.vnflcm.utils._get_affected_resources')
    def test_lcm_notification_exception_in_exception(
            self, mock_res, mock_vnf_by_id, mock_log,
            mock_des, mock_vnfd_dict,
            mock_vnf_lcm_subscriptions_get,
            mock_get_lock, mock_save,
            mock_build_info,
            mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_send_notification):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        mock_vnf_lcm_subscriptions_get.return_value = \
            [mock.MagicMock(**fakes.get_vnf_lcm_subscriptions())]
        mock_res.return_value = {}
        mock_change_vnf_status.side_effect = Exception
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context,
                vnf_instance.id, mock.ANY, 'ERROR')
        self.assertEqual("FAILED_TEMP",
            mock_send_notification.call_args[0][1].get("operationState"))

        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        mock_change_vnf_status.side_effect = Exception
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context,
                            vnf_instance.id, mock.ANY, 'ERROR')
        self.assertEqual("FAILED_TEMP",
            mock_send_notification.call_args[0][1].get("operationState"))

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={"key": "value"})
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
            vnf_instance, terminate_vnf_req, vnf_dict)
        mock_change_vnf_status.assert_called_with(self.context,
                vnf_instance.id, mock.ANY, 'ERROR')
        self.assertEqual("FAILED_TEMP",
            mock_send_notification.call_args[0][1].get("operationState"))

        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()
        mock_change_vnf_status.side_effect = Exception
        self.conductor.change_ext_conn(
            self.context,
            vnf_instance,
            vnf_dict,
            change_ext_conn_req,
            vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context,
                vnf_instance.id, mock.ANY, 'ERROR')
        self.assertEqual("FAILED_TEMP",
            mock_send_notification.call_args[0][1].get("operationState"))

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_error_point_notify_processing(
            self, mock_vnf_by_id, mock_des, mock_vnfd_dict, mock_get_lock,
            mock_save, mock_build_info, mock_change_vnf_status,
            mock_update_vnf_attributes):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.NOTIFY_PROCESSING
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, mock.ANY, vnf_dict, instantiate_vnf_req)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        mock_change_vnf_status. \
            assert_called_once_with(self.context, vnf_instance.id,
                                    mock.ANY, 'PENDING_CREATE')
        mock_update_vnf_attributes.assert_called_once()

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._build_instantiated_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_error_point_vnf_config_end(
            self, mock_vnf_by_id, mock_des, mock_vnfd_dict, mock_get_lock,
            mock_save, mock_build_info, mock_update_vnf_attributes):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.VNF_CONFIG_END
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver.instantiate_vnf.assert_called_once_with(
            self.context, vnf_instance, vnf_dict, instantiate_vnf_req)

    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_instantiate_vnf_instance_error_point_notify_completed(
            self, mock_vnf_by_id, mock_des, mock_vnfd_dict,
            mock_get_lock, mock_save):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        instantiate_vnf_req = vnflcm_fakes.get_instantiate_vnf_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf_etsi(instance_id=self.instance_uuid,
                                       flavour=instantiate_vnf_req.flavour_id)
        vnf_dict['before_error_point'] = fields.ErrorPoint.NOTIFY_COMPLETED
        vnfd_key = 'vnfd_' + instantiate_vnf_req.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        self.conductor.instantiate(self.context, vnf_instance, vnf_dict,
                                   instantiate_vnf_req, vnf_lcm_op_occs_id)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()

    @mock.patch('tacker.vnflcm.utils._get_vim')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    def test_terminate_vnf_instance(self, mock_get_lock,
            mock_send_notification, mock_change_vnf_status, mock_get_vim):
        inst_vnf_info = fd_utils.get_vnf_instantiated_info()
        vnf_instance = (fd_utils.
            get_vnf_instance_object(instantiated_vnf_info=inst_vnf_info))

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={'key': 'value'})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = {
            **db_utils.get_dummy_vnf(instance_id=self.instance_uuid),
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        mock_get_vim.return_value = {'vim_id': uuidsentinel.vim_id,
            'vim_type': 'openstack'}
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance, terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, vnf_instance, terminate_vnf_req, vnf_dict)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        self.assertEqual(mock_send_notification.call_count, 2)
        self.assertEqual(mock_change_vnf_status.call_count, 2)

    @mock.patch('tacker.vnflcm.utils._get_vim')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    def test_terminate_vnf_instance_error_point_notify_processing(
            self, mock_get_lock, mock_send_notification,
            mock_change_vnf_status, mock_get_vim):
        inst_vnf_info = fd_utils.get_vnf_instantiated_info()
        vnf_instance = (fd_utils.
            get_vnf_instance_object(instantiated_vnf_info=inst_vnf_info))

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={'key': 'value'})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.NOTIFY_PROCESSING
        mock_get_vim.return_value = {'vim_id': uuidsentinel.vim_id,
            'vim_type': 'openstack'}
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance, terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, vnf_instance, terminate_vnf_req, vnf_dict)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        self.assertEqual(mock_send_notification.call_count, 2)
        self.assertEqual(mock_change_vnf_status.call_count, 2)

    @mock.patch('tacker.vnflcm.utils._get_vim')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    def test_terminate_vnf_instance_error_point_internal_processing(
            self, mock_get_lock, mock_send_notification,
            mock_change_vnf_status, mock_get_vim):
        inst_vnf_info = fd_utils.get_vnf_instantiated_info()
        vnf_instance = (fd_utils.
            get_vnf_instance_object(instantiated_vnf_info=inst_vnf_info))

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={'key': 'value'})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INTERNAL_PROCESSING
        mock_get_vim.return_value = {'vim_id': uuidsentinel.vim_id,
            'vim_type': 'openstack'}
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance, terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, vnf_instance, terminate_vnf_req, vnf_dict)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        self.assertEqual(mock_send_notification.call_count, 2)
        self.assertEqual(mock_change_vnf_status.call_count, 1)

    @mock.patch('tacker.vnflcm.utils._get_vim')
    @mock.patch('tacker.vnflcm.vnflcm_driver.VnfLcmDriver'
                '.terminate_vnf')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    def test_terminate_vnf_instance_error_point_notify_completed(self,
            mock_get_lock, mock_send_notification, mock_change_vnf_status,
            mock_vnflcm_driver_terminate_vnf, mock_get_vim):
        inst_vnf_info = fd_utils.get_vnf_instantiated_info()
        vnf_instance = (fd_utils.
            get_vnf_instance_object(instantiated_vnf_info=inst_vnf_info))

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={'key': 'value'})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.NOTIFY_COMPLETED
        mock_get_vim.return_value = {'vim_id': uuidsentinel.vim_id,
            'vim_type': 'openstack'}
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance, terminate_vnf_req, vnf_dict)

        self.assertEqual(mock_vnflcm_driver_terminate_vnf.call_count, 0)
        self.vnflcm_driver._vnf_instance_update.assert_called_once()
        self.assertEqual(mock_send_notification.call_count, 2)
        self.assertEqual(mock_change_vnf_status.call_count, 0)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_terminate_vnf_instance_grant(self, mock_vnf_by_id,
                                          mock_grants,
                                          mock_exec,
                                          mock_get_lock,
                                          mock_change_vnf_status):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        # mock_package_in_use.return_value = True
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple',
            vnf_instance_id=vnf_instance.id)
        vnf_instance.instantiated_vnf_info.reinitialize()

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL)
        vnfLcmOpOccId = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.VNF_CONFIG_END

        mock_exec.return_value = True
        resRemResource = []
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnfLcmOpOccId
        grant_dict['removeResources'] = []
        grant_dict['removeResources'].extend(resRemResource)
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)
        self.conductor.terminate(self.context, vnfLcmOpOccId,
                                vnf_instance, terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, mock.ANY, terminate_vnf_req, vnf_dict)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_terminate_vnf_instance_grant_1(self, mock_vnf_by_id,
                                            mock_grants,
                                            mock_exec,
                                            mock_get_lock,
                                            mock_change_vnf_status):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        # mock_package_in_use.return_value = True
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple',
            vnf_instance_id=vnf_instance.id)
        vnf_instance.instantiated_vnf_info.reinitialize()
        vnfc_obj = objects.VnfcResourceInfo()
        vnfc_obj.id = '2c6e5cc7-240d-4458-a683-1fe648351280'
        vnfc_obj.vdu_id = 'VDU1'
        vnfc_obj.storage_resource_ids = \
            ['faf14707-da7c-4eec-be99-8099fa1e9fa0']
        compute_resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce2')
        vnfc_obj.compute_resource = compute_resource
        cp_obj = objects.VnfcCpInfo()
        cp_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fa9'
        cp_obj.cpd_id = 'PORT1'
        cp_obj.vnf_link_port_id = 'faf14707-da7c-4eec-be99-8099fa1e9fb9'
        vnfc_obj.vnfc_cp_info = [cp_obj]
        st_obj = objects.VirtualStorageResourceInfo()
        st_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fa0'
        st_obj.virtual_storage_desc_id = 'ST1'
        storage_resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce4')
        st_obj.storage_resource = storage_resource
        vl_obj = objects.VnfVirtualLinkResourceInfo()
        vl_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fa1'
        vl_obj.vnf_virtual_link_desc_id = 'VL1'
        network_resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce5')
        vl_obj.network_resource = network_resource
        port_obj = objects.VnfLinkPortInfo()
        port_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fb9'
        port_obj.cp_instance_id = 'faf14707-da7c-4eec-be99-8099fa1e9fa9'
        resource_handle = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce3')
        port_obj.resource_handle = resource_handle
        vl_obj.vnf_link_ports = [port_obj]
        vnf_instance.instantiated_vnf_info.vnfc_resource_info = [vnfc_obj]
        vnf_instance.instantiated_vnf_info.virtual_storage_resource_info = \
            [st_obj]
        vnf_instance.instantiated_vnf_info.vnf_virtual_link_resource_info = \
            [vl_obj]

        resRemResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fb9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa0',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa1',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemResource.append(resource)

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL)
        vnfLcmOpOccId = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.VNF_CONFIG_END

        mock_exec.return_value = True
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnfLcmOpOccId
        grant_dict['removeResources'] = []
        grant_dict['removeResources'].extend(resRemResource)
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)
        self.conductor.terminate(self.context, vnfLcmOpOccId,
                                vnf_instance, terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, mock.ANY, terminate_vnf_req, vnf_dict)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_terminate_vnf_instance_grant_resource_exception(self,
                                          mock_vnf_by_id,
                                          mock_save,
                                          mock_grants,
                                          mock_exec,
                                          mock_get_lock,
                                          mock_send):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        # mock_package_in_use.return_value = True
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple',
            vnf_instance_id=vnf_instance.id)
        vnf_instance.instantiated_vnf_info.reinitialize()

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL)
        vnfLcmOpOccId = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL

        mock_exec.return_value = True
        resRemResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemResource.append(resource)
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnfLcmOpOccId
        grant_dict['removeResources'] = []
        grant_dict['removeResources'].extend(resRemResource)
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)

        self.assertRaises(exceptions.ValidationError,
            self.conductor.terminate,
            self.context, vnfLcmOpOccId,
            vnf_instance, terminate_vnf_req, vnf_dict)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_terminate_vnf_instance_grant_exception(self,
                                          mock_vnf_by_id,
                                          mock_save,
                                          mock_grants,
                                          mock_exec,
                                          mock_get_lock,
                                          mock_send):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        # mock_package_in_use.return_value = True
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple',
            vnf_instance_id=vnf_instance.id)
        vnf_instance.instantiated_vnf_info.reinitialize()

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL)
        vnfLcmOpOccId = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL

        mock_exec.return_value = True
        mock_grants.side_effect = \
            requests.exceptions.HTTPError("MockException")
        self.assertRaises(requests.exceptions.HTTPError,
            self.conductor.terminate,
            self.context, vnfLcmOpOccId,
            vnf_instance, terminate_vnf_req, vnf_dict)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.LOG')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, '_get_grant_execute')
    @mock.patch.object(test_nfvo_client.GrantRequest, 'grants')
    @mock.patch.object(objects.VnfLcmOpOcc, 'save')
    @mock.patch.object(objects.VnfLcmOpOcc, 'get_by_id')
    def test_terminate_vnf_instance_grant_and_vnf_instance_update_exception(
            self, mock_vnf_by_id, mock_save, mock_grants, mock_exec,
            mock_get_lock, mock_send, mock_log):
        def _vnf_instance_update(self, context, vnf_instance, **kwargs):
            raise Exception('test_ex')
        FakeVnfLcmDriver._vnf_instance_update = _vnf_instance_update

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance_data['instantiation_state'] = (
            fields.VnfInstanceState.INSTANTIATED)
        vnf_instance = objects.VnfInstance(context=self.context,
            **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple',
            vnf_instance_id=vnf_instance.id)
        vnf_instance.instantiated_vnf_info.reinitialize()

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL)
        vnfLcmOpOccId = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL

        mock_exec.return_value = True
        mock_grants.side_effect = (
            requests.exceptions.HTTPError('MockException'))
        self.assertRaises(requests.exceptions.HTTPError,
            self.conductor.terminate, self.context, vnfLcmOpOccId,
            vnf_instance, terminate_vnf_req, vnf_dict)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'), 'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)
        expected_log = 'Failed to revert task state for vnf instance'
        mock_log.error.assert_called_once_with(
            f'{expected_log} {vnf_instance.id}. Error: test_ex')

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    def test_terminate_vnf_instance_exception(self, mock_get_lock,
                                    mock_send_notification,
                                    mock_change_vnf_status):
        inst_vnf_info = fd_utils.get_vnf_instantiated_info()
        vnf_instance = fd_utils. \
            get_vnf_instance_object(instantiated_vnf_info=inst_vnf_info)

        mock_send_notification.side_effect = (Exception, None)
        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={"key": "value"})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        vnf_dict['before_error_point'] = fields.ErrorPoint.INITIAL
        vnf_dict['current_error_point'] = fields.ErrorPoint.INITIAL

        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance, terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_not_called()
        mock_change_vnf_status.assert_called_once_with(self.context,
            vnf_instance.id, mock.ANY, 'ERROR')
        self.assertEqual(mock_send_notification.call_count, 2)

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfPackage, 'is_package_in_use')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    def test_terminate_vnf_instance_already_not_instantiated(self,
            mock_log, mock_package_in_use, mock_get_lock,
            mock_send_notification):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        mock_package_in_use.return_value = True
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.NOT_INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={"key": "value"})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance,
                                 terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_not_called()
        mock_package_in_use.assert_not_called()
        expected_log = ('Terminate action cannot be performed on vnf %(id)s '
                        'which is in %(state)s state.')
        mock_log.error.assert_called_once_with(
            expected_log, {
                'id': vnf_instance.id,
                'state': fields.VnfInstanceState.NOT_INSTANTIATED})

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfPackage, 'is_package_in_use')
    def test_terminate_vnf_instance_with_usage_state_not_in_use(self,
            mock_vnf_package_is_package_in_use, mock_get_lock,
            mock_send_notification):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        mock_vnf_package_is_package_in_use.return_value = False
        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={"key": "value"})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance,
                                 terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, mock.ANY, terminate_vnf_req,
            vnf_lcm_op_occs_id)
        mock_vnf_package_is_package_in_use.assert_called_once()

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfPackage, 'is_package_in_use')
    def test_terminate_vnf_instance_with_usage_state_already_in_use(self,
            mock_vnf_package_is_package_in_use, mock_get_lock,
            mock_send_notification):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        mock_vnf_package_is_package_in_use.return_value = True
        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={"key": "value"})
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance,
                                 terminate_vnf_req, vnf_dict)

        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, mock.ANY, terminate_vnf_req,
            vnf_lcm_op_occs_id)
        mock_vnf_package_is_package_in_use.assert_called_once()

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_send_lcm_op_occ_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfPackage, 'is_package_in_use')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    def test_terminate_vnf_instance_failed_to_update_usage_state(
            self, mock_log, mock_is_package_in_use, mock_get_lock,
            mock_send_notification):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        terminate_vnf_req = objects.TerminateVnfRequest(
            termination_type=fields.VnfInstanceTerminationType.GRACEFUL,
            additional_params={"key": "value"})
        mock_is_package_in_use.side_effect = Exception
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = db_utils.get_dummy_vnf(instance_id=self.instance_uuid)
        self.conductor.terminate(self.context, vnf_lcm_op_occs_id,
                                 vnf_instance,
                                 terminate_vnf_req, vnf_dict)
        self.vnflcm_driver.terminate_vnf.assert_called_once_with(
            self.context, mock.ANY, terminate_vnf_req,
            vnf_lcm_op_occs_id)
        expected_msg = "Failed to update usage_state of vnf package %s"
        mock_log.error.assert_called_once_with(expected_msg,
                                               vnf_package_vnfd.package_uuid)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes_stack_param')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_add_additional_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_heal_vnf_instance(self, mock_vnf_by_id,
            mock_get_lock, mock_save, mock_change_vnf_status,
            mock_update_insta_vnf_info, mock_add_additional_vnf_info,
            mock_update_vnf_attributes_stack_param):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            'fake': 'fake_dict',
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(mock_change_vnf_status.call_count, 2)
        mock_update_insta_vnf_info. \
            assert_called_once_with(self.context, vnf_instance, heal_vnf_req)
        mock_add_additional_vnf_info. \
            assert_called_once_with(self.context, vnf_instance)
        mock_update_vnf_attributes_stack_param.assert_called_once_with(
            self.context, vnf_dict, vnf_instance.id, heal_vnf_req,
            vnf_instance.instantiated_vnf_info)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes_stack_param')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_add_additional_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_heal_vnf_instance_error_point_notify_processing(
            self, mock_vnf_by_id, mock_get_lock, mock_save,
            mock_change_vnf_status, mock_update_insta_vnf_info,
            mock_add_additional_vnf_info,
            mock_update_vnf_attributes_stack_param):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            'fake': 'fake_dict',
            'before_error_point': fields.ErrorPoint.NOTIFY_PROCESSING,
            'status': ''
        }
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(mock_change_vnf_status.call_count, 2)
        mock_update_insta_vnf_info. \
            assert_called_once_with(self.context, vnf_instance, heal_vnf_req)
        mock_add_additional_vnf_info. \
            assert_called_once_with(self.context, vnf_instance)
        mock_update_vnf_attributes_stack_param.assert_called_once_with(
            self.context, vnf_dict, vnf_instance.id, heal_vnf_req,
            vnf_instance.instantiated_vnf_info)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes_stack_param')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_add_additional_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_heal_vnf_instance_error_point_internal_processing(
            self, mock_vnf_by_id, mock_get_lock, mock_save,
            mock_change_vnf_status, mock_update_insta_vnf_info,
            mock_add_additional_vnf_info,
            mock_update_vnf_attributes_stack_param):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            'fake': 'fake_dict',
            'before_error_point': fields.ErrorPoint.INTERNAL_PROCESSING,
            'status': ''
        }
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(mock_change_vnf_status.call_count, 1)
        mock_update_insta_vnf_info. \
            assert_called_once_with(self.context, vnf_instance, heal_vnf_req)
        mock_add_additional_vnf_info. \
            assert_called_once_with(self.context, vnf_instance)
        mock_update_vnf_attributes_stack_param.assert_called_once_with(
            self.context, vnf_dict, vnf_instance.id, heal_vnf_req,
            vnf_instance.instantiated_vnf_info)

    @mock.patch('tacker.vnflcm.vnflcm_driver.VnfLcmDriver'
                '.heal_vnf')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_add_additional_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_heal_vnf_instance_error_point_notify_completed(
            self, mock_vnf_by_id, mock_get_lock, mock_save,
            mock_change_vnf_status, mock_update_insta_vnf_info,
            mock_add_additional_vnf_info, mock_vnflcm_driver_heal_vnf):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = \
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            'fake': 'fake_dict',
            'before_error_point': fields.ErrorPoint.NOTIFY_COMPLETED,
            'status': ''
        }
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(mock_vnflcm_driver_heal_vnf.call_count, 0)
        self.assertEqual(mock_change_vnf_status.call_count, 0)
        self.assertEqual(mock_update_insta_vnf_info.call_count, 0)
        self.assertEqual(mock_add_additional_vnf_info.call_count, 0)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes_stack_param')
    @mock.patch('tacker.vnflcm.utils'
                '._build_instantiated_vnf_info')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._add_additional_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnflcm_interface')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._get_placement')
    def test_heal_vnf_instance_grant(self,
                                     mock_placement,
                                     mock_d1,
                                     mock_d2,
                                     mock_d3,
                                     mock_vim,
                                     mock_act,
                                     mock_grants,
                                     mock_exec,
                                     mock_vnf_by_id,
                                     mock_get_lock,
                                     mock_save,
                                     mock_add_vnf_info,
                                     mock_update_vnf_info,
                                     mock_change_status,
                                     mock_vnfd_dict,
                                     mock_build_info,
                                     mock_update_vnf_attributes_stack_param):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple')
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            **db_utils.get_dummy_vnf_etsi(
                instance_id=self.instance_uuid, flavour='simple'),
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        mock_exec.return_value = True
        mock_act.return_value = None
        vnfd_key = 'vnfd_' + vnf_instance.instantiated_vnf_info.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}
        resAddResource = []
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['vimAssets'] = vimAssets
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        mock_d1.return_value = []
        mock_placement.return_value = []
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        mock_add_vnf_info.assert_called_once()
        mock_update_vnf_info.assert_called_once()
        mock_update_vnf_attributes_stack_param.assert_called_once()
        self.vnflcm_driver.heal_vnf.assert_called_once_with(
            self.context, mock.ANY, vnf_dict, heal_vnf_req)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes_stack_param')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._add_additional_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnflcm_interface')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._get_placement')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._check_res_add_remove_rsc')
    def test_heal_vnf_instance_grant_1(self,
                                       mock_check,
                                       mock_placement,
                                       mock_d1,
                                       mock_d2,
                                       mock_d3,
                                       mock_vim,
                                       mock_act,
                                       mock_grants,
                                       mock_exec,
                                       mock_vnf_by_id,
                                       mock_get_lock,
                                       mock_save,
                                       mock_add_vnf_info,
                                       mock_update_vnf_info,
                                       mock_change_status,
                                       mock_vnfd_dict,
                                       mock_update_vnf_attribute_stack_param):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple',
            vnf_instance_id=vnf_instance.id)
        vnf_instance.instantiated_vnf_info.reinitialize()
        vnfc_obj = objects.VnfcResourceInfo()
        vnfc_obj.id = '2c6e5cc7-240d-4458-a683-1fe648351280'
        vnfc_obj.vdu_id = 'VDU1'
        vnfc_obj.storage_resource_ids = \
            ['faf14707-da7c-4eec-be99-8099fa1e9fa0']
        compute_resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce2')
        vnfc_obj.compute_resource = compute_resource
        cp_obj = objects.VnfcCpInfo()
        cp_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fa9'
        cp_obj.cpd_id = 'PORT1'
        cp_obj.vnf_link_port_id = 'faf14707-da7c-4eec-be99-8099fa1e9fb9'
        vnfc_obj.vnfc_cp_info = [cp_obj]
        st_obj = objects.VirtualStorageResourceInfo()
        st_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fa0'
        st_obj.virtual_storage_desc_id = 'ST1'
        storage_resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce4')
        st_obj.storage_resource = storage_resource
        vl_obj = objects.VnfVirtualLinkResourceInfo()
        vl_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fa1'
        vl_obj.vnf_virtual_link_desc_id = 'VL1'
        network_resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce5')
        vl_obj.network_resource = network_resource
        port_obj = objects.VnfLinkPortInfo()
        port_obj.id = 'faf14707-da7c-4eec-be99-8099fa1e9fb9'
        port_obj.cp_instance_id = 'faf14707-da7c-4eec-be99-8099fa1e9fa9'
        resource_handle = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce3')
        port_obj.resource_handle = resource_handle
        vl_obj.vnf_link_ports = [port_obj]
        vnf_instance.instantiated_vnf_info.vnfc_resource_info = [vnfc_obj]
        vnf_instance.instantiated_vnf_info.virtual_storage_resource_info = \
            [st_obj]
        vnf_instance.instantiated_vnf_info.vnf_virtual_link_resource_info = \
            [vl_obj]
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            **db_utils.get_dummy_vnf_etsi(
                instance_id=self.instance_uuid, flavour='simple'),
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        vnfd_yaml = self._nfv_sample('test_heal_grant_unit/sample_vnfd.yaml')
        with open(vnfd_yaml) as f:
            mock_vnfd_dict.return_value = yaml.safe_load(f)
        mock_vnfd_dict.return_value['imports'] = []
        etsi_common_file_path = self._nfv_sample(
            'common/Definitions/etsi_nfv_sol001_common_types.yaml')
        etsi_vnfd_file_path = self._nfv_sample(
            'common/Definitions/etsi_nfv_sol001_vnfd_types.yaml')
        etsi_vnfd_file_path_tmp = self._nfv_sample(
            'common/Definitions/etsi_nfv_sol001_vnfd_types_tmp.yaml')
        with open(etsi_vnfd_file_path) as f:
            data = yaml.safe_load(f)
            del data['imports']
            data['imports'] = []
            data['imports'].append(etsi_common_file_path)
        with open(etsi_vnfd_file_path_tmp, "w", encoding="utf-8") as f:
            yaml.dump(data, f)
        types_file_path = self._nfv_sample(
            'test_heal_grant_unit/helloworld3_types.yaml')
        types_file_tmp_path = self._nfv_sample(
            'test_heal_grant_unit/helloworld3_types_tmp.yaml')
        with open(types_file_path) as f:
            data = yaml.safe_load(f)
            data['imports'] = []
            data['imports'].append(etsi_common_file_path)
            data['imports'].append(etsi_vnfd_file_path_tmp)
        with open(types_file_tmp_path, "w", encoding="utf-8") as f:
            yaml.dump(data, f)
        mock_vnfd_dict.return_value['imports'].append(etsi_common_file_path)
        mock_vnfd_dict.return_value['imports'].append(etsi_vnfd_file_path_tmp)
        mock_vnfd_dict.return_value['imports'].append(types_file_tmp_path)
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        mock_exec.return_value = True
        mock_act.return_value = None
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}
        resAddResource = []
        resRemResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa0'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': 'c91e0e86-e555-71fc-0f6b-e3d2945f2fd6'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': '5d84a03c-07e9-4978-b0cd-6450482189b7'}
        resRemResource.append(resource)
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351281',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351282',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['removeResources'] = []
        grant_dict['removeResources'].extend(resRemResource)
        grant_dict['vimAssets'] = vimAssets
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        mock_d1.return_value = ['ST1']
        res_str = '[{"id_type": "RES_MGMT", "resource_id": ' + \
            '"2c6e5cc7-240d-4458-a683-1fe648351200", ' + \
            '"vim_connection_id": ' + \
            '"2a63bee3-0c43-4568-bcfa-b0cb733e064c"}]'
        placemnt = models.PlacementConstraint(
            id='c2947d8a-2c67-4e8f-ad6f-c0889b351c17',
            vnf_instance_id='7ddc38c3-a116-48b0-bfc1-68d7f306f467',
            affinity_or_anti_affinity='ANTI_AFFINITY',
            scope='ZONE',
            server_group_name='my_compute_placement_policy',
            resource=res_str,
            deleted_at=datetime.datetime.min)
        mock_placement.return_value = [placemnt]
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        mock_add_vnf_info.assert_called_once()
        mock_update_vnf_info.assert_called_once()
        mock_update_vnf_attribute_stack_param.assert_called_once()
        self.vnflcm_driver.heal_vnf.assert_called_once_with(
            self.context, mock.ANY, vnf_dict, heal_vnf_req)
        os.remove(types_file_tmp_path)
        os.remove(etsi_vnfd_file_path_tmp)

    @mock.patch('tacker.vnflcm.utils'
                '._build_instantiated_vnf_info')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._add_additional_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnflcm_interface')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._get_placement')
    def test_heal_vnf_instance_grant_resource_exception(self,
                                     mock_placement,
                                     mock_d1,
                                     mock_d2,
                                     mock_d3,
                                     mock_vim,
                                     mock_act,
                                     mock_grants,
                                     mock_exec,
                                     mock_vnf_by_id,
                                     mock_get_lock,
                                     mock_save,
                                     mock_add_vnf_info,
                                     mock_update_vnf_info,
                                     mock_send,
                                     mock_vnfd_dict,
                                     mock_build_info):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple')
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnfd_key = 'vnfd_' + vnf_instance.instantiated_vnf_info.flavour_id
        vnf_dict = {
            **db_utils.get_dummy_vnf_etsi(
                instance_id=self.instance_uuid, flavour='simple'),
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        mock_exec.return_value = True
        mock_act.return_value = None
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}
        resAddResource = []
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351281',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = vnf_instance.id
        grant_dict['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['vimAssets'] = vimAssets
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        mock_d1.return_value = []
        mock_placement.return_value = []
        self.assertRaises(exceptions.ValidationError,
            self.conductor.heal,
            self.context, vnf_instance, vnf_dict,
            heal_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.vnflcm.utils'
                '._build_instantiated_vnf_info')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._add_additional_vnf_info')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch('tacker.vnflcm.utils._get_vnflcm_interface')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._get_placement')
    def test_heal_vnf_instance_grant_exception(self,
                                     mock_placement,
                                     mock_d1,
                                     mock_d2,
                                     mock_d3,
                                     mock_vim,
                                     mock_act,
                                     mock_grants,
                                     mock_exec,
                                     mock_vnf_by_id,
                                     mock_get_lock,
                                     mock_save,
                                     mock_add_vnf_info,
                                     mock_update_vnf_info,
                                     mock_send,
                                     mock_vnfd_dict,
                                     mock_build_info):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = \
            fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = objects.InstantiatedVnfInfo(
            flavour_id='simple')
        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            **db_utils.get_dummy_vnf_etsi(
                instance_id=self.instance_uuid, flavour='simple'),
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        vnfd_key = 'vnfd_' + vnf_instance.instantiated_vnf_info.flavour_id
        vnfd_yaml = vnf_dict['vnfd']['attributes'].get(vnfd_key, '')
        mock_vnfd_dict.return_value = yaml.safe_load(vnfd_yaml)
        vnf_lcm_op_occs_id = 'a9c36d21-21aa-4692-8922-7999bbcae08c'
        mock_exec.return_value = True
        mock_act.return_value = None
        mock_grants.side_effect = \
            requests.exceptions.HTTPError("MockException")

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        mock_d1.return_value = []
        mock_placement.return_value = []
        self.assertRaises(requests.exceptions.HTTPError,
            self.conductor.heal,
            self.context, vnf_instance, vnf_dict,
            heal_vnf_req, vnf_lcm_op_occs_id)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes_stack_param')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_send_lcm_op_occ_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_add_additional_vnf_info')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    def test_heal_vnf_instance_exception(self,
            mock_log, mock_get_lock, mock_add_additional_vnf_info,
            mock_change_vnf_status, mock_update_insta_vnf_info,
            mock_send_notification, mock_update_vnf_attributes_stack_param):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)

        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.NOT_INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        mock_add_additional_vnf_info.side_effect = Exception

        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            'fake': 'fake_dict',
            'before_error_point': fields.ErrorPoint.INITIAL,
            'status': ''
        }
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context,
            vnf_instance.id, mock.ANY, constants.ERROR)
        mock_update_insta_vnf_info.assert_called_with(self.context,
            vnf_instance, heal_vnf_req)
        mock_update_vnf_attributes_stack_param.assert_called_once_with(
            self.context, vnf_dict, vnf_instance.id, heal_vnf_req,
            vnf_instance.instantiated_vnf_info)
        self.assertEqual(mock_send_notification.call_count, 2)

    @unittest.skip("Such test is no longer feasible.")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    def test_heal_vnf_instance_already_not_instantiated(
            self, mock_log, mock_get_lock):
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)

        vnf_instance_data['instantiation_state'] =\
            fields.VnfInstanceState.NOT_INSTANTIATED
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        heal_vnf_req = objects.HealVnfRequest(cause="healing request")
        vnf_dict = {
            'fake': 'fake_dict',
            'status': ''
        }
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        self.conductor.heal(self.context, vnf_instance, vnf_dict,
                            heal_vnf_req, vnf_lcm_op_occs_id)

        self.vnflcm_driver.heal_vnf.assert_not_called()
        expected_log = ('Heal action cannot be performed on vnf %(id)s '
                        'which is in %(state)s state.')
        mock_log.error.assert_called_once_with(
            expected_log, {
                'id': vnf_instance.id,
                'state': fields.VnfInstanceState.NOT_INSTANTIATED})

    @mock.patch.object(os, 'remove')
    @mock.patch.object(shutil, 'rmtree')
    @mock.patch.object(os.path, 'exists')
    @mock.patch.object(objects.VnfPackagesList, 'get_by_filters')
    def test_run_cleanup_vnf_packages(self, mock_get_by_filter,
                                      mock_exists, mock_rmtree,
                                      mock_remove):
        vnf_package_data = {'algorithm': None, 'hash': None,
                            'location_glance_store': None,
                            'onboarding_state': 'CREATED',
                            'operational_state': 'DISABLED',
                            'tenant_id': uuidsentinel.tenant_id,
                            'usage_state': 'NOT_IN_USE',
                            'user_data': {'abc': 'xyz'}
                            }

        vnfpkgm = objects.VnfPackage(context=self.context, **vnf_package_data)
        vnfpkgm.create()
        vnfpkgm.destroy(self.context)

        mock_get_by_filter.return_value = [vnfpkgm]
        mock_exists.return_value = True
        conductor_server.Conductor('host')._run_cleanup_vnf_packages(
            self.context)
        mock_get_by_filter.assert_called()
        mock_rmtree.assert_called()
        mock_remove.assert_called()

    @mock.patch.object(sys, 'exit')
    @mock.patch.object(conductor_server.LOG, 'error')
    @mock.patch.object(glance_store, 'initialize_glance_store')
    @mock.patch.object(os.path, 'isdir')
    def test_init_host(self, mock_isdir, mock_initialize_glance_store,
                       mock_log_error, mock_exit):
        mock_isdir.return_value = False
        self.conductor.init_host()
        mock_log_error.assert_called()
        mock_exit.assert_called_with(1)
        self.assertIn("Config option 'vnf_package_csar_path' is not configured"
                      " correctly. VNF package CSAR path directory %s doesn't"
                      " exist", mock_log_error.call_args[0][0])

    @mock.patch.object(urllib.request, 'urlopen')
    def test_upload_vnf_package_from_uri_with_invalid_auth(self,
                                                           mock_url_open):
        address_information = "http://localhost/test.zip"
        user_name = "username"
        password = "password"
        mock_url_open.side_effect = urlerr.HTTPError(
            url='', code=401, msg='HTTP Error 401 Unauthorized', hdrs={},
            fp=None)
        self.assertRaises(exceptions.VNFPackageURLInvalid,
                          self.conductor.upload_vnf_package_from_uri,
                          self.context,
                          self.vnf_package,
                          address_information,
                          user_name=user_name,
                          password=password)
        self.assertEqual('CREATED', self.vnf_package.onboarding_state)

    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    def test_scale_in(
            self,
            mock_grants,
            mock_d1,
            mock_d2,
            mock_d3,
            mock_vim,
            mock_vnf_by_id,
            mock_get_lock):
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        vnf_info = fakes._get_vnf()
        vnf_lcm_op_occ = objects.VnfLcmOpOcc(
            state_entered_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                              tzinfo=datetime.timezone.utc),
            start_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc),
            vnf_instance_id=uuidsentinel.vnf_instance_id,
            operation='SCALE',
            operation_state='ACTIVE',
            is_automatic_invocation=False,
            operation_params='{"type": "SCALE_IN", "aspect_id": "SP1"}',
            error_point=0,
            id=test_constants.UUID,
            created_at=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc))
        vnf_info['vnf_lcm_op_occ'] = vnf_lcm_op_occ
        vnf_instance = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED,
            scale_status="scale_status")
        scale_vnf_request = fakes.scale_request("SCALE_IN", 1)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        mock_d1.return_value = ([], [], "", "")
        vnf_info['addResources'] = []
        vnf_info['removeResources'] = []
        vnf_info['affinity_list'] = []
        vnf_info['placement_constraint_list'] = []
        vnf_info['before_error_point'] = fields.ErrorPoint.INITIAL
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnf_instance_id'] = uuidsentinel.vnf_instance_id
        grant_dict['vnf_lcm_op_occ_id'] = test_constants.UUID
        mock_grants.return_value = MockResponse(
            json_data=jsonutils.dumps(grant_dict))

        self.conductor.scale(
            self.context,
            vnf_info,
            vnf_instance,
            scale_vnf_request)
        self.vnflcm_driver.scale_vnf.assert_called_once_with(
            self.context, vnf_info, mock.ANY, scale_vnf_request)
        self.assertEqual(0, mock_grants.call_count)

    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    def test_scale_in_grants(
            self,
            mock_grants,
            moch_exec,
            mock_d1,
            mock_d2,
            mock_d3,
            mock_vim,
            mock_vnf_by_id,
            mock_get_lock):
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        vnf_info = fakes._get_vnf()
        vnf_lcm_op_occ = objects.VnfLcmOpOcc(
            state_entered_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                              tzinfo=datetime.timezone.utc),
            start_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc),
            vnf_instance_id=uuidsentinel.vnf_instance_id,
            operation='SCALE',
            operation_state='ACTIVE',
            is_automatic_invocation=False,
            operation_params='{"type": "SCALE_IN", "aspect_id": "SP1"}',
            error_point=0,
            id=test_constants.UUID,
            created_at=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc))
        vnf_info['vnf_lcm_op_occ'] = vnf_lcm_op_occ
        vnf_instance = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED,
            scale_status="scale_status")
        scale_vnf_request = fakes.scale_request("SCALE_IN", 1)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        removeResources = []
        resRemoveResource = []
        resource = objects.ResourceDefinition(
            id='2c6e5cc7-240d-4458-a683-1fe648351280',
            type='COMPUTE',
            vdu_id='VDU1',
            resource_template_id='VDU1')
        resource.resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce2')
        removeResources.append(resource)
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemoveResource.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='LINKPORT',
            vdu_id='VDU1',
            resource_template_id='PORT1')
        resource.resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce3')
        removeResources.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemoveResource.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='STORAGE',
            vdu_id='VDU1',
            resource_template_id='ST1')
        resource.resource = objects.ResourceHandle(
            vim_connection_id=uuidsentinel.vim_id,
            resource_id='6e1c286d-c023-4b34-8369-831c6e84cce4')
        removeResources.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resRemoveResource.append(resource)
        mock_d1.return_value = ([], [], "", "")
        vnf_info['addResources'] = []
        vnf_info['removeResources'] = removeResources
        vnf_info['affinity_list'] = []
        vnf_info['placement_constraint_list'] = []
        vnf_info['before_error_point'] = fields.ErrorPoint.INITIAL
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = uuidsentinel.vnf_instance_id
        grant_dict['vnfLcmOpOccId'] = test_constants.UUID
        grant_dict['removeResources'] = []
        grant_dict['removeResources'].extend(resRemoveResource)
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)
        moch_exec.return_value = True

        self.conductor.scale(
            self.context,
            vnf_info,
            vnf_instance,
            scale_vnf_request)
        self.vnflcm_driver.scale_vnf.assert_called_once_with(
            self.context, vnf_info, mock.ANY, scale_vnf_request)
        self.assertEqual(1, mock_grants.call_count)

    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    def test_scale_grants_out(
            self,
            mock_grants,
            moch_exec,
            mock_d1,
            mock_d2,
            mock_d3,
            mock_vim,
            mock_vnf_by_id,
            mock_get_lock):
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        vnf_info = fakes._get_vnf()
        vnf_lcm_op_occ = objects.VnfLcmOpOcc(
            context=self.context,
            state_entered_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                              tzinfo=datetime.timezone.utc),
            start_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc),
            vnf_instance_id=vnf_instance.id,
            operation='SCALE',
            operation_state='ACTIVE',
            is_automatic_invocation=False,
            operation_params='{"type": "SCALE_OUT", "aspect_id": "SP1"}',
            is_cancel_pending=False,
            error_point=0,
            id='00e1314d-2a82-40bd-b318-cc881243842d',
            tenant_id='01db9967-ba45-4f1d-962d-7cbb825448f1',
            created_at=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc))
        vnf_lcm_op_occ.create()
        vnf_info['vnf_lcm_op_occ'] = vnf_lcm_op_occ
        vnf_instance = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED,
            scale_status="scale_status")
        scale_vnf_request = fakes.scale_request("SCALE_OUT", 1)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        addResources = []
        resAddResource = []
        resource = objects.ResourceDefinition(
            id='2c6e5cc7-240d-4458-a683-1fe648351280',
            type='COMPUTE',
            vdu_id='VDU1',
            resource_template_id='VDU1')
        addResources.append(resource)
        resource = {
            'resourceDefinitionId': '2c6e5cc7-240d-4458-a683-1fe648351280',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='LINKPORT',
            vdu_id='VDU1',
            resource_template_id='PORT1')
        addResources.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='STORAGE',
            vdu_id='VDU1',
            resource_template_id='ST1')
        addResources.append(resource)
        resource = {
            'resourceDefinitionId': 'faf14707-da7c-4eec-be99-8099fa1e9fa9',
            'vimConnectionId': uuidsentinel.vim_id,
            'zoneId': '5e4da3c3-4a55-412a-b624-843921f8b51d'}
        resAddResource.append(resource)
        placement = objects.PlacementConstraint(
            affinity_or_anti_affinity='ANTI_AFFINITY',
            scope='ZONE',
            resource=[],
            fallback_best_effort=True)
        resource = objects.ConstraintResourceRef(
            id_type='RES_MGMT',
            resource_id='2c6e5cc7-240d-4458-a683-1fe648351200',
            vim_connection_id=uuidsentinel.vim_id)
        placement.resource.append(resource)
        resource = objects.ConstraintResourceRef(
            id_type='GRANT',
            resource_id='2c6e5cc7-240d-4458-a683-1fe648351280')
        placement.resource.append(resource)
        vimAssets = {'computeResourceFlavours': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdVirtualComputeDescId': 'CDU1',
             'vimFlavourId': 'm1.tiny'}],
            'softwareImages': [
            {'vimConnectionId': uuidsentinel.vim_id,
             'vnfdSoftwareImageId': 'VDU1',
             'vimSoftwareImageId': 'cirros'}]}

        mock_d1.return_value = ([], [], "", "")
        vnf_info['addResources'] = addResources
        vnf_info['removeResources'] = []
        vnf_info['affinity_list'] = []
        vnf_info['placement_constraint_list'] = []
        vnf_info['placement_constraint_list'].append(placement)
        vnf_info['before_error_point'] = fields.ErrorPoint.INITIAL
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnfInstanceId'] = uuidsentinel.vnf_instance_id
        grant_dict['vnfLcmOpOccId'] = test_constants.UUID
        grant_dict['addResources'] = []
        grant_dict['addResources'].extend(resAddResource)
        grant_dict['vimAssets'] = vimAssets
        moch_exec.return_value = True
        json_data = grant_dict
        mock_grants.return_value = MockResponse(json_data=json_data)

        self.conductor.scale(
            self.context,
            vnf_info,
            vnf_instance,
            scale_vnf_request)
        self.vnflcm_driver.scale_vnf.assert_called_once_with(
            self.context, vnf_info, mock.ANY, scale_vnf_request)
        self.assertEqual(1, mock_grants.call_count)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    def test_scale_grants_out_resource_exception(
            self,
            mock_grants,
            moch_exec,
            mock_d1,
            mock_d2,
            mock_d3,
            mock_vim,
            mock_vnf_by_id,
            mock_get_lock,
            mock_send):
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        vnf_info = fakes._get_vnf()
        vnf_lcm_op_occ = objects.VnfLcmOpOcc(
            context=self.context,
            state_entered_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                              tzinfo=datetime.timezone.utc),
            start_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc),
            vnf_instance_id=vnf_instance.id,
            operation='SCALE',
            operation_state='ACTIVE',
            is_automatic_invocation=False,
            operation_params='{"type": "SCALE_OUT", "aspect_id": "SP1"}',
            is_cancel_pending=False,
            error_point=0,
            tenant_id='7cd42301-a3a5-47ad-a7bd-87b02540503b',
            id='00e1314d-2a82-40bd-b318-cc881243843d',
            created_at=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc))
        vnf_lcm_op_occ.create()
        vnf_info['vnf_lcm_op_occ'] = vnf_lcm_op_occ
        vnf_instance = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED,
            scale_status="scale_status")
        scale_vnf_request = fakes.scale_request("SCALE_OUT", 1)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        addResources = []
        resource = objects.ResourceDefinition(
            id='2c6e5cc7-240d-4458-a683-1fe648351280',
            type='COMPUTE',
            vdu_id='VDU1',
            resource_template_id='VDU1')
        addResources.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='LINKPORT',
            vdu_id='VDU1',
            resource_template_id='PORT1')
        addResources.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='STORAGE',
            vdu_id='VDU1',
            resource_template_id='ST1')
        addResources.append(resource)
        mock_d1.return_value = ([], [], "", "")
        vnf_info['addResources'] = addResources
        vnf_info['removeResources'] = []
        vnf_info['affinity_list'] = []
        vnf_info['placement_constraint_list'] = []
        vnf_info['before_error_point'] = fields.ErrorPoint.INITIAL
        grant_dict = {}
        grant_dict['id'] = 'c213e465-8220-487e-9464-f79104e81e96'
        grant_dict['vnf_instance_id'] = uuidsentinel.vnf_instance_id
        grant_dict['vnf_lcm_op_occ_id'] = test_constants.UUID
        moch_exec.return_value = True
        mock_grants.return_value = MockResponse(json_data=grant_dict)

        self.assertRaises(exceptions.ValidationError,
            self.conductor.scale,
            self.context, vnf_info, vnf_instance, scale_vnf_request)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(vim_client.VimClient, "get_vim")
    @mock.patch.object(driver_manager.DriverManager, "__init__")
    @mock.patch.object(driver_manager.DriverManager, "register")
    @mock.patch.object(driver_manager.DriverManager, "invoke")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    def test_scale_grants_exception(
            self,
            mock_grants,
            moch_exec,
            mock_d1,
            mock_d2,
            mock_d3,
            mock_vim,
            mock_vnf_by_id,
            mock_get_lock,
            mock_send):
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()

        vnf_info = fakes._get_vnf()
        vnf_lcm_op_occ = objects.VnfLcmOpOcc(
            context=self.context,
            state_entered_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                              tzinfo=datetime.timezone.utc),
            start_time=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc),
            vnf_instance_id=vnf_instance.id,
            operation='SCALE',
            operation_state='ACTIVE',
            is_automatic_invocation=False,
            operation_params='{"type": "SCALE_OUT", "aspect_id": "SP1"}',
            error_point=0,
            id='00e1314d-2a82-40bd-b318-cc881243843d',
            tenant_id='00e1314d-2a82-40bd-b318-cc881243843r',
            created_at=datetime.datetime(2000, 1, 1, 1, 1, 1,
                                         tzinfo=datetime.timezone.utc))
        vnf_lcm_op_occ.create()
        vnf_info['vnf_lcm_op_occ'] = vnf_lcm_op_occ
        vnf_instance = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED,
            scale_status="scale_status")
        scale_vnf_request = fakes.scale_request("SCALE_OUT", 1)

        mock_vim.return_value = vnflcm_fakes.get_dummy_openstack_vim_obj()
        addResources = []
        resource = objects.ResourceDefinition(
            id='2c6e5cc7-240d-4458-a683-1fe648351280',
            type='COMPUTE',
            vdu_id='VDU1',
            resource_template_id='VDU1')
        addResources.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='LINKPORT',
            vdu_id='VDU1',
            resource_template_id='PORT1')
        addResources.append(resource)
        resource = objects.ResourceDefinition(
            id='faf14707-da7c-4eec-be99-8099fa1e9fa9',
            type='STORAGE',
            vdu_id='VDU1',
            resource_template_id='ST1')
        addResources.append(resource)
        mock_d1.return_value = ([], [], "", "")
        vnf_info['addResources'] = addResources
        vnf_info['removeResources'] = []
        vnf_info['affinity_list'] = []
        vnf_info['placement_constraint_list'] = []
        vnf_info['before_error_point'] = fields.ErrorPoint.INITIAL
        moch_exec.return_value = True
        mock_grants.side_effect = \
            requests.exceptions.HTTPError("MockException")

        self.assertRaises(requests.exceptions.HTTPError,
            self.conductor.scale,
            self.context, vnf_info, vnf_instance, scale_vnf_request)
        print("test_log rollback")
        print(mock_send.call_args)
        self.assertEqual(
            mock_send.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_not_found_subscription(self,
                                                   mock_subscriptions_get):
        mock_subscriptions_get.return_value = None
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfLcmOperationOccurrenceNotification'}

        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, -1)
        mock_subscriptions_get.assert_called()

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_vnf_lcm_operation_occurrence(self,
                                                    mock_subscriptions_get,
                                                    mock_vnf_by_id):
        self.requests_mock.register_uri('POST',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=204)

        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)
        sub_1 = self._create_subscriptions(
            filter_params={
                "vnfInstanceSubscriptionFilter": {
                    "vnfdIds": [uuidsentinel.vnfd_id]
                }
            }
        )
        sub_2 = self._create_subscriptions()
        sub_2[0].tenant_id = uuidsentinel.tenant_id_2
        sub_3 = self._create_subscriptions()
        sub_4 = self._create_subscriptions(
            filter_params={
                "vnfInstanceSubscriptionFilter": {
                    "vnfdIds": [uuidsentinel.vnfd_id_2]
                }
            }
        )
        sub_5 = self._create_subscriptions(
            filter_params={
                "vnfInstanceSubscriptionFilter": {}
            }
        )
        sub_sum = sub_1 + sub_2 + sub_3 + sub_4 + sub_5
        mock_subscriptions_get.return_value = sub_sum
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfLcmOperationOccurrenceNotification',
            'operationTypes': 'SCALE',
            'operationStates': 'RESULT',
            '_links': {}}

        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, 0)
        mock_subscriptions_get.assert_called()

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(3, req_count)

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_vnf_identifier_creation(self,
                                                    mock_subscriptions_get,
                                                    mock_vnf_by_id):
        self.requests_mock.register_uri(
            'POST',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=204)

        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)
        mock_subscriptions_get.return_value = self._create_subscriptions()
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfIdentifierCreationNotification',
            'links': {}}

        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, 0)
        mock_subscriptions_get.assert_called()

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(1, req_count)

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_with_auth_basic(self, mock_subscriptions_get,
            mock_vnf_by_id):
        self.requests_mock.register_uri('POST',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=204)

        auth_user_name = 'test_user'
        auth_password = 'test_password'
        mock_subscriptions_get.return_value = self._create_subscriptions(
            {'authType': 'BASIC',
            'paramsBasic': {'userName': auth_user_name,
                            'password': auth_password}})
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfIdentifierCreationNotification',
            'links': {}}

        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, 0)
        mock_subscriptions_get.assert_called()

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(1, req_count)
        self.assert_auth_basic(
            history[0],
            auth_user_name,
            auth_password)

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_with_auth_client_credentials(
            self, mock_subscriptions_get, mock_vnf_by_id):
        auth.auth_manager = auth._AuthManager()
        self.requests_mock.register_uri(
            'POST',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=204)

        auth_user_name = 'test_user'
        auth_password = 'test_password'
        token_endpoint = 'https://oauth2/tokens'
        self.requests_mock.register_uri(
            'POST', token_endpoint, json={
                'access_token': 'test_token', 'token_type': 'bearer'},
            headers={'Content-Type': 'application/json'},
            status_code=200)

        mock_subscriptions_get.return_value = self._create_subscriptions(
            {'authType': ['OAUTH2_CLIENT_CREDENTIALS'],
            'paramsOauth2ClientCredentials': {
                'clientId': auth_user_name,
                'clientPassword': auth_password,
                'tokenEndpoint': token_endpoint}})

        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfIdentifierCreationNotification',
            'links': {}}

        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, 0)
        mock_subscriptions_get.assert_called()

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost", 'https://oauth2')
        self.assertEqual(2, req_count)
        self.assert_auth_basic(history[0], auth_user_name, auth_password)
        self.assert_auth_client_credentials(history[1], "test_token")

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_retry_notification(self,
                                              mock_subscriptions_get,
                                              mock_vnf_by_id):
        self.requests_mock.register_uri('POST',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=400)

        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)
        mock_subscriptions_get.return_value = self._create_subscriptions()
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfIdentifierCreationNotification',
            'links': {}}

        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, 0)
        mock_subscriptions_get.assert_called()

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(3, req_count)

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_send_error(self,
                                        mock_subscriptions_get,
                                        mock_vnf_by_id):
        self.requests_mock.register_uri(
            'POST',
            "https://localhost/callback",
            exc=requests.exceptions.HTTPError("MockException"))

        mock_subscriptions_get.return_value = self._create_subscriptions()
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfIdentifierCreationNotification',
            'links': {}}
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)
        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, 0)
        mock_subscriptions_get.assert_called()

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(1, req_count)

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_internal_server_error(
            self, mock_subscriptions_get,
            mock_vnf_by_id):
        mock_subscriptions_get.side_effect = Exception("MockException")
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationTypes': 'VnfIdentifierCreationNotification',
            'links': {}}
        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)
        result = self.conductor.send_notification(self.context, notification)

        self.assertEqual(result, -2)
        mock_subscriptions_get.assert_called()

    @mock.patch.object(objects.VnfInstance, 'get_by_id')
    @mock.patch.object(objects.LccnSubscriptionRequest,
                       'vnf_lcm_subscriptions_get')
    def test_send_notification_timeout(self, mock_subscriptions_get,
            mock_vnf_by_id):
        self.requests_mock.register_uri(
            'POST',
            "https://localhost/callback",
            exc=requests.Timeout)

        mock_vnf_by_id.return_value = fakes.return_vnf_instance(
            fields.VnfInstanceState.INSTANTIATED)

        mock_subscriptions_get.return_value = self._create_subscriptions()
        notification = {
            'vnfInstanceId': uuidsentinel.vnf_instance_id,
            'notificationType': 'VnfIdentifierCreationNotification',
            'links': {}}

        result = self.conductor.send_notification(self.context,
            notification)

        # return value when timeout for POST method is 0
        self.assertEqual(result, 0)

    def test_get_notification(self):
        cfg.CONF.set_override('test_callback_uri', True,
                              group='vnf_lcm')
        self.requests_mock.register_uri('GET',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=204)

        callback_uri = 'https://localhost/callback'

        vnf_lcm_subscription = objects.\
            LccnSubscriptionRequest(context=self.context)
        vnf_lcm_subscription.id = uuidsentinel.lcm_subscription_id
        vnf_lcm_subscription.callback_uri = callback_uri

        result = self.conductor.test_notification(self.context,
            vnf_lcm_subscription)

        # return value when successful for GET method is 0
        self.assertEqual(result, 0)

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(1, req_count)
        cfg.CONF.set_override('test_callback_uri', False,
                              group='vnf_lcm')

    def test_get_notification_callback_uri_false(self):
        cfg.CONF.set_override('test_callback_uri', False,
                              group='vnf_lcm')
        self.requests_mock.register_uri('GET',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=204)

        callback_uri = 'https://localhost/callback'

        vnf_lcm_subscription = objects.\
            LccnSubscriptionRequest(context=self.context)
        vnf_lcm_subscription.id = uuidsentinel.lcm_subscription_id
        vnf_lcm_subscription.callback_uri = callback_uri

        result = self.conductor.test_notification(self.context,
            vnf_lcm_subscription)

        # return value when successful for GET method is 0
        self.assertEqual(result, 0)

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(0, req_count)

    def test_get_notification_unknown_auth_type(self):
        cfg.CONF.set_override('test_callback_uri', True,
                              group='vnf_lcm')
        callback_uri = 'https://localhost/callback'

        vnf_lcm_subscription = objects.LccnSubscriptionRequest(
            context=self.context)
        vnf_lcm_subscription.id = uuidsentinel.lcm_subscription_id
        vnf_lcm_subscription.callback_uri = callback_uri
        vnf_lcm_subscription.authentication = jsonutils.dumps(
            {'authType': 'OTHER_AUTH',
             'otherParams': {'userName': "test_user",
                             'password': "test_password"}}
        )
        error = self.assertRaises(
            exceptions.NotificationProcessingError,
            self.conductor.test_notification,
            self.context, vnf_lcm_subscription)
        self.assertEqual("Unknown auth_type OTHER_AUTH", str(error))

    def test_get_notification_retry(self):
        cfg.CONF.set_override('test_callback_uri', True, group='vnf_lcm')
        self.requests_mock.register_uri('GET',
            "https://localhost/callback",
            headers={
                'Content-Type': 'application/json'},
            status_code=400)

        callback_uri = 'https://localhost/callback'

        vnf_lcm_subscription = objects.\
            LccnSubscriptionRequest(context=self.context)
        vnf_lcm_subscription.id = uuidsentinel.lcm_subscription_id
        vnf_lcm_subscription.callback_uri = callback_uri

        result = self.conductor.test_notification(self.context,
            vnf_lcm_subscription)

        # return value when error occurs for GET method is -1
        self.assertEqual(result, -1)

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(3, req_count)
        cfg.CONF.set_override('test_callback_uri', False,
                              group='vnf_lcm')

    def test_get_notification_timeout(self):
        cfg.CONF.set_override('test_callback_uri', True, group='vnf_lcm')
        self.requests_mock.register_uri(
            'GET',
            "https://localhost/callback",
            exc=requests.Timeout)

        callback_uri = 'https://localhost/callback'

        vnf_lcm_subscription = objects.\
            LccnSubscriptionRequest(context=self.context)
        vnf_lcm_subscription.id = uuidsentinel.lcm_subscription_id
        vnf_lcm_subscription.callback_uri = callback_uri

        result = self.conductor.test_notification(self.context,
            vnf_lcm_subscription)

        # return value when error for GET method is -1
        self.assertEqual(result, -1)

        history = self.requests_mock.request_history
        req_count = nfvo_client._count_mock_history(
            history, "https://localhost")
        self.assertEqual(3, req_count)
        cfg.CONF.set_override('test_callback_uri', False,
                              group='vnf_lcm')

    @mock.patch.object(objects.vnf_instance.VnfInstance, "get_by_id")
    @mock.patch.object(t_context.get_admin_context().session, "add")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "create")
    def test_update(self, mock_create, mock_save, mock_add, mock_get_by_id):
        mock_create.return_value = "OK"
        mock_add.return_value = "OK"
        mock_save.return_value = "OK"
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.vnf_instance.VnfInstance(context=self.context,
            **vnf_instance_data)
        mock_get_by_id.return_value = vnf_instance
        vnfd_id = "2c69a161-0000-4b0f-bcf8-391f8fc76600"

        self.conductor.update(
            self.context,
            self.vnf_lcm_opoccs,
            self.body_data,
            self.vnfd_pkg_data,
            vnfd_id)

    @mock.patch.object(objects.vnf_instance.VnfInstance, "get_by_id")
    @mock.patch.object(t_context.get_admin_context().session, "add")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.vnf_instance.VnfInstance, "update")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "create")
    def test_update_lcm_with_vnf_pkg_id(self, mock_create,
                                        mock_update, mock_save,
                                        mock_add, mock_get_by_id):
        mock_create.return_value = "OK"
        mock_update.return_value = datetime.datetime(
            1900, 1, 1, 1, 1, 1, tzinfo=datetime.timezone.utc)
        mock_add.return_value = "OK"
        mock_save.return_value = "OK"
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.vnf_instance.VnfInstance(context=self.context,
            **vnf_instance_data)
        mock_get_by_id.return_value = vnf_instance
        vnfd_id = "2c69a161-0000-4b0f-bcf8-391f8fc76600"

        self.conductor.update(
            self.context,
            self.vnf_lcm_opoccs,
            self.body_data,
            self.vnfd_pkg_data,
            vnfd_id)

    @mock.patch.object(objects.vnf_instance.VnfInstance, "get_by_id")
    @mock.patch.object(t_context.get_admin_context().session, "add")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.vnf_instance.VnfInstance, "update")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "create")
    def test_update_lcm_without_vnf_instance_name(self, mock_create,
                                        mock_update, mock_save,
                                        mock_add, mock_get_by_id):
        mock_create.return_value = "OK"
        mock_update.return_value = datetime.datetime(
            1900, 1, 1, 1, 1, 1, tzinfo=datetime.timezone.utc)
        mock_add.return_value = "OK"
        mock_save.return_value = "OK"
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.vnf_instance.VnfInstance(context=self.context,
            **vnf_instance_data)
        mock_get_by_id.return_value = vnf_instance
        vnfd_id = "2c69a161-0000-4b0f-bcf8-391f8fc76600"
        self.body_data.pop('vnf_instance_name')
        self.conductor.update(
            self.context,
            self.vnf_lcm_opoccs,
            self.body_data,
            self.vnfd_pkg_data,
            vnfd_id)

    @mock.patch.object(objects.vnf_instance.VnfInstance, "get_by_id")
    @mock.patch.object(t_context.get_admin_context().session, "add")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "save")
    @mock.patch.object(objects.vnf_instance.VnfInstance, "update")
    @mock.patch.object(objects.vnf_lcm_op_occs.VnfLcmOpOcc, "create")
    def test_update_lcm_without_vnf_instance_description(self, mock_create,
                                                  mock_update, mock_save,
                                                  mock_add, mock_get_by_id):
        mock_create.return_value = "OK"
        mock_update.return_value = datetime.datetime(
            1900, 1, 1, 1, 1, 1, tzinfo=datetime.timezone.utc)
        mock_add.return_value = "OK"
        mock_save.return_value = "OK"
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.vnf_instance.VnfInstance(context=self.context,
            **vnf_instance_data)
        mock_get_by_id.return_value = vnf_instance
        vnfd_id = "2c69a161-0000-4b0f-bcf8-391f8fc76600"
        self.body_data.pop('vnf_instance_description')
        self.conductor.update(
            self.context,
            self.vnf_lcm_opoccs,
            self.body_data,
            self.vnfd_pkg_data,
            vnfd_id)

    @mock.patch.object(BackingOffClient, 'prepare')
    @mock.patch.object(objects.vnf_package.VnfPackage, 'save')
    @mock.patch.object(glance_store, 'delete_csar')
    @mock.patch.object(csar_utils, 'load_csar_data')
    @mock.patch.object(glance_store, 'load_csar')
    def test_revert_upload_vnf_package_in_load_csar_data(self,
            mock_load_csar,
            mock_load_csar_data,
            mock_delete_csar,
            mock_save,
            mock_prepare):
        self.vnf_package.onboarding_state = "PROCESSING"
        self.vnf_package.save()
        mock_load_csar.return_value = '/var/lib/tacker/5f5d99c6-844a-4c3' \
                                      '1-9e6d-ab21b87dcfff.zip'
        mock_load_csar_data.side_effect = exceptions.InvalidCSAR
        mock_prepare.return_value = self.cctxt_mock

        self.assertRaises(exceptions.InvalidCSAR,
            self.conductor.load_csar_data,
            self.context, self.vnf_package)
        mock_delete_csar.assert_called()
        mock_prepare.assert_called()
        self.cctxt_mock.cast.assert_called_once_with(
            self.context, 'delete_csar', vnf_package=self.vnf_package)
        self.assertEqual(self.vnf_package.onboarding_state, "CREATED")

    @mock.patch.object(conductor_server.Conductor, '_onboard_vnf_package')
    @mock.patch.object(csar_utils, 'load_csar_data')
    @mock.patch.object(glance_store, 'load_csar')
    def test_load_csar_data(self, mock_load_csar,
            mock_load_csar_data, mock_onboard):
        self.vnf_package.onboarding_state = "PROCESSING"
        self.vnf_package.save()
        vnf_data = fakes.get_vnf_package_vnfd()
        vnf_data = self._rename_vnfdata_keys(vnf_data)
        mock_load_csar_data.return_value = (vnf_data, [], [])
        mock_load_csar.return_value = '/var/lib/tacker/5f5d99c6-844a-4c3' \
                                      '1-9e6d-ab21b87dcfff.zip'
        self.conductor.load_csar_data(self.context, self.vnf_package)
        mock_load_csar.assert_called()
        mock_load_csar_data.assert_called()
        mock_onboard.assert_called()

    @mock.patch.object(conductor_server.Conductor, '_onboard_vnf_package')
    @mock.patch.object(csar_utils, 'load_csar_data')
    @mock.patch.object(glance_store, 'load_csar')
    def test_load_csar_data_with_onboarding_state_created(self, mock_load_csar,
            mock_load_csar_data, mock_onboard):
        self.vnf_package.onboarding_state = "PROCESSING"
        self.vnf_package.save()
        vnf_data = fakes.get_vnf_package_vnfd()
        vnf_data = self._rename_vnfdata_keys(vnf_data)
        mock_load_csar_data.return_value = (vnf_data, [], [])

        def f():
            vnf_pkg = objects.VnfPackage.get_by_id(
                self.context, self.vnf_package.id)
            vnf_pkg.onboarding_state = "CREATED"
            vnf_pkg.save()
            return (mock.ANY, mock.ANY, mock.ANY)

        mock_load_csar_data.return_value = f()
        mock_load_csar.return_value = '/var/lib/tacker/5f5d99c6-844a-4c3' \
                                      '1-9e6d-ab21b87dcfff.zip'
        self.conductor.load_csar_data(self.context, self.vnf_package)

        mock_load_csar.assert_called()
        mock_load_csar_data.assert_called()
        vnf_pkg = objects.VnfPackage.get_by_id(
            self.context, self.vnf_package.id)
        self.assertEqual(vnf_pkg.onboarding_state, "CREATED")
        mock_onboard.assert_not_called()

    @mock.patch.object(conductor_server.Conductor, '_onboard_vnf_package')
    @mock.patch.object(csar_utils, 'load_csar_data')
    @mock.patch.object(glance_store, 'load_csar')
    def test_load_csar_data_downloading_not_completed(self, mock_load_csar,
            mock_load_csar_data, mock_onboard):
        self.vnf_package.onboarding_state = "PROCESSING"
        self.vnf_package.downloading = 1
        self.vnf_package.save()
        vnf_data = fakes.get_vnf_package_vnfd()
        vnf_data = self._rename_vnfdata_keys(vnf_data)
        mock_load_csar_data.return_value = (vnf_data, [], [])
        mock_load_csar.return_value = '/var/lib/tacker/5f5d99c6-844a-4c3' \
                                      '1-9e6d-ab21b87dcfff.zip'
        self.conductor.load_csar_data(self.context, self.vnf_package)
        mock_load_csar.assert_called()
        mock_load_csar_data.assert_called()
        mock_onboard.assert_not_called()
        vnf_pkg = objects.VnfPackage.get_by_id(
            self.context, self.vnf_package.id)
        self.assertEqual(vnf_pkg.onboarding_state, "PROCESSING")
        self.assertEqual(vnf_pkg.downloading, 1)

    @mock.patch.object(BackingOffClient, 'prepare')
    def test_onboard_vnf_package_through_upload_vnf_package_content(
            self, mock_prepare):
        mock_prepare.return_value = self.cctxt_mock
        self.conductor.upload_vnf_package_content(
            self.context, self.vnf_package)
        mock_prepare.assert_called()
        self.cctxt_mock.cast.assert_called_once_with(
            self.context, 'load_csar_data',
            vnf_package=self.vnf_package)

    def _rename_vnfdata_keys(self, vnf_data):
        vnf_data["descriptor_id"] = vnf_data.pop("id")
        vnf_data["provider"] = vnf_data.pop("vnf_provider")
        vnf_data["product_name"] = vnf_data.pop("vnf_product_name")
        vnf_data["software_version"] = vnf_data.pop("vnf_software_version")
        vnf_data["descriptor_version"] = vnf_data.pop("vnfd_version")
        return vnf_data

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info_change_ext_conn')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_change_ext_conn(
            self,
            mock_vnf_by_id,
            mock_exec,
            mock_get_lock,
            mock_save,
            mock_send_notification,
            mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_update_instantiated_vnf_info_change_ext_conn):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        mock_vnf_by_id.return_value = objects.VnfLcmOpOcc(
            context=self.context, **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)

        vnf_instance.create()
        vnf_instance.instantiation_state = fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = fakes.get_instantiated_vnf_info()
        vnf_dict = {"before_error_point": 0, "status": ""}
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()

        op_states = []

        def _store_operation(context, notification_data):
            op_states.append(notification_data['operationState'])

        mock_send_notification.side_effect = _store_operation

        # Test condition settings.
        mock_exec.return_value = False

        self.conductor.change_ext_conn(
            self.context,
            vnf_instance,
            vnf_dict,
            change_ext_conn_req,
            vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context, mock.ANY,
            constants.ACTIVE, constants.PENDING_CHANGE_EXT_CONN)
        mock_update_vnf_attributes.assert_called_with(self.context, mock.ANY,
            mock.ANY, mock.ANY, constants.ACTIVE)

        self.assertEqual(2, mock_send_notification.call_count)
        self.assertEqual('PROCESSING', op_states[0])
        self.assertEqual('COMPLETED', op_states[1])

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info_change_ext_conn')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    def test_change_ext_conn_grant(
            self,
            mock_grants,
            mock_exec,
            mock_vnf_by_id,
            mock_get_lock,
            mock_save,
            mock_send_notification,
            mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_update_instantiated_vnf_info_change_ext_conn):
        cfg.CONF.set_override(
            'base_url',
            'http://127.0.0.1:9990/grant/v1/grants',
            group='connect_grant')
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        mock_vnf_by_id.return_value = objects.VnfLcmOpOcc(
            context=self.context, **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = fakes.get_instantiated_vnf_info()
        vnf_dict = {"before_error_point": 0, "status": ""}
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()
        vnf_virtual_link = (
            vnf_instance.instantiated_vnf_info.vnf_virtual_link_resource_info)

        # Test condition settings.
        mock_exec.return_value = True
        res_grant = dict()
        res_grant['id'] = uuidsentinel.grant_id
        res_grant['vnfInstanceId'] = vnf_instance.id
        res_grant['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        res_grant['updateResources'] = [
            {
                'resourceDefinitionId':
                    vnf_virtual_link[0].vnf_link_ports[0].id,
            },
            {
                'resourceDefinitionId':
                    vnf_virtual_link[0].vnf_link_ports[1].id,
            },
            {
                'resourceDefinitionId':
                    vnf_virtual_link[1].vnf_link_ports[0].id,
            },
            {
                'resourceDefinitionId':
                    vnf_virtual_link[1].vnf_link_ports[2].id,
            },
        ]
        mock_grants.return_value = MockResponse(json_data=res_grant)

        op_states = []

        def _store_operation(context, notification_data):
            op_states.append(notification_data['operationState'])

        mock_send_notification.side_effect = _store_operation

        self.conductor.change_ext_conn(
            self.context,
            vnf_instance,
            vnf_dict,
            change_ext_conn_req,
            vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context,
            mock.ANY, constants.ACTIVE, constants.PENDING_CHANGE_EXT_CONN)
        mock_update_vnf_attributes.assert_called_with(self.context, mock.ANY,
            mock.ANY, mock.ANY, constants.ACTIVE)

        self.assertEqual(2, mock_send_notification.call_count)
        self.assertEqual('PROCESSING', op_states[0])
        self.assertEqual('COMPLETED', op_states[1])

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info_change_ext_conn')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_change_ext_conn_grant_exception_http_error(
            self,
            mock_vnf_by_id,
            mock_grants,
            mock_exec,
            mock_get_lock,
            mock_save,
            mock_send_notification,
            mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_update_instantiated_vnf_info_change_ext_conn):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        mock_vnf_by_id.return_value = objects.VnfLcmOpOcc(
            context=self.context, **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = fakes.get_instantiated_vnf_info()
        vnf_dict = {"before_error_point": 0, "status": ""}
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()

        # Test condition settings.
        mock_exec.return_value = True
        mock_grants.side_effect = (
            requests.exceptions.HTTPError("MockException"))

        self.assertRaises(requests.exceptions.HTTPError,
            self.conductor.change_ext_conn,
            self.context, vnf_instance, vnf_dict,
            change_ext_conn_req, vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_not_called()
        mock_update_instantiated_vnf_info_change_ext_conn.assert_not_called()
        self.assertEqual(
            mock_send_notification.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_instantiated_vnf_info_change_ext_conn')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(conductor_server.Conductor, "_get_grant_execute")
    @mock.patch.object(test_nfvo_client.GrantRequest, "grants")
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_change_ext_conn_grant_exception_validation_error(
            self,
            mock_vnf_by_id,
            mock_grants,
            mock_exec,
            mock_get_lock,
            mock_save,
            mock_send_notification,
            mock_change_vnf_status,
            mock_update_vnf_attributes,
            mock_update_instantiated_vnf_info_change_ext_conn):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        mock_vnf_by_id.return_value = objects.VnfLcmOpOcc(
            context=self.context, **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = fakes.get_instantiated_vnf_info()
        vnf_dict = {"before_error_point": 0, "status": ""}
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()

        # Test condition settings.
        mock_exec.return_value = True
        res_upd_resource = []
        resource = {
            'resourceDefinitionId': uuidsentinel.rsc_dummy,
            'vimConnectionId': '1ffcd358-bee3-4bfb-bc3a-920db09f5da5',
        }
        res_upd_resource.append(resource)
        res_grant = dict()
        res_grant['id'] = uuidsentinel.grant_id
        res_grant['vnfInstanceId'] = vnf_instance.id
        res_grant['vnfLcmOpOccId'] = vnf_lcm_op_occs_id
        res_grant['updateResources'] = []
        res_grant['updateResources'].extend(res_upd_resource)
        mock_grants.return_value = MockResponse(json_data=res_grant)

        self.assertRaises(exceptions.ValidationError,
            self.conductor.change_ext_conn,
            self.context, vnf_instance, vnf_dict,
            change_ext_conn_req, vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_not_called()
        mock_update_vnf_attributes.assert_not_called()
        mock_update_instantiated_vnf_info_change_ext_conn.assert_not_called()
        self.assertEqual(
            mock_send_notification.call_args[0][1].get('operationState'),
            'ROLLED_BACK')
        self.vnflcm_driver._vnf_instance_update.assert_called_once_with(
            self.context, vnf_instance, task_state=None)

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info_change_ext_conn')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch.object(objects.LccnSubscriptionRequest,
        'vnf_lcm_subscriptions_get')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch('tacker.vnflcm.utils._convert_desired_capacity')
    @mock.patch('tacker.conductor.conductor_server.LOG')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_change_ext_conn_failed_with_exception(
            self,
            mock_vnf_by_id,
            mock_log,
            mock_des,
            mock_vnfd_dict,
            mock_vnf_lcm_subscriptions_get,
            mock_get_lock,
            mock_save,
            mock_update_vnf_info_change_ext_conn,
            mock_update_vnf_attributes,
            mock_change_vnf_status):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        mock_vnf_by_id.return_value = (
            objects.VnfLcmOpOcc(context=self.context,
                                **lcm_op_occs_data))

        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        vnf_dict = {"before_error_point": 0,
                    "current_error_point": 6, "status": ""}
        m_vnf_lcm_subscriptions = (
            [mock.MagicMock(**fakes.get_vnf_lcm_subscriptions())])
        mock_vnf_lcm_subscriptions_get.return_value = (
            m_vnf_lcm_subscriptions)
        mock_update_vnf_info_change_ext_conn.side_effect = Exception
        self.conductor.change_ext_conn(
            self.context,
            vnf_instance,
            vnf_dict,
            change_ext_conn_req,
            vnf_lcm_op_occs_id)
        mock_change_vnf_status.assert_called_with(self.context,
            mock.ANY, mock.ANY, constants.ERROR)
        self.vnflcm_driver.change_ext_conn_vnf.assert_called_once_with(
            self.context, vnf_instance, vnf_dict, change_ext_conn_req)
        mock_update_vnf_info_change_ext_conn.assert_called_once()

    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '.send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info_change_ext_conn')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_ext_conn_grant')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_change_ext_conn_retry_error_point_1(
            self,
            mock_vnf_by_id,
            mock_vnfd_dict,
            mock_get_lock,
            mock_save,
            mock_change_ext_conn_grant,
            mock_update_vnf_info_change_ext_conn,
            mock_update_vnf_attributes,
            mock_change_vnf_status,
            mock_send_notification):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        mock_vnf_by_id.return_value = objects.VnfLcmOpOcc(
            context=self.context, **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = fakes.get_instantiated_vnf_info()
        vnf_dict = {"before_error_point": 1, "status": ""}
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()

        self.conductor.change_ext_conn(
            self.context,
            vnf_instance,
            vnf_dict,
            change_ext_conn_req,
            vnf_lcm_op_occs_id)
        self.vnflcm_driver.change_ext_conn_vnf.assert_called_once_with(
            self.context, vnf_instance, vnf_dict, change_ext_conn_req)
        mock_change_vnf_status.assert_called_with(self.context, mock.ANY,
            constants.ACTIVE, constants.PENDING_CHANGE_EXT_CONN)
        self.assertEqual(mock_change_ext_conn_grant.call_count, 0)
        mock_update_vnf_attributes.assert_called_once()

    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                'send_notification')
    @mock.patch('tacker.conductor.conductor_server.Conductor.'
                '_change_vnf_status')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_vnf_attributes')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._update_instantiated_vnf_info_change_ext_conn')
    @mock.patch('tacker.conductor.conductor_server.Conductor'
                '._change_ext_conn_grant')
    @mock.patch.object(objects.VnfLcmOpOcc, "save")
    @mock.patch.object(coordination.Coordinator, 'get_lock')
    @mock.patch('tacker.vnflcm.utils._get_vnfd_dict')
    @mock.patch.object(objects.VnfLcmOpOcc, "get_by_id")
    def test_change_ext_conn_retry_error_point_7(
            self,
            mock_vnf_by_id,
            mock_vnfd_dict,
            mock_get_lock,
            mock_save,
            mock_change_ext_conn_grant,
            mock_update_vnf_info_change_ext_conn,
            mock_update_vnf_attributes,
            mock_change_vnf_status,
            mock_send_notification):
        lcm_op_occs_data = fakes.get_lcm_op_occs_data()
        vnf_lcm_op_occs_id = uuidsentinel.vnf_lcm_op_occs_id
        mock_vnf_by_id.return_value = objects.VnfLcmOpOcc(
            context=self.context, **lcm_op_occs_data)
        vnf_package_vnfd = self._create_and_upload_vnf_package()
        vnf_instance_data = fake_obj.get_vnf_instance_data(
            vnf_package_vnfd.vnfd_id)
        vnf_instance = objects.VnfInstance(context=self.context,
                                           **vnf_instance_data)
        vnf_instance.create()
        vnf_instance.instantiation_state = fields.VnfInstanceState.INSTANTIATED
        vnf_instance.save()
        vnf_instance.instantiated_vnf_info = fakes.get_instantiated_vnf_info()
        vnf_dict = {"before_error_point": 7, "status": ""}
        change_ext_conn_req = fakes.get_change_ext_conn_request_obj()

        self.conductor.change_ext_conn(
            self.context,
            vnf_instance,
            vnf_dict,
            change_ext_conn_req,
            vnf_lcm_op_occs_id)
        self.vnflcm_driver.change_ext_conn_vnf.assert_called_once_with(
            self.context, vnf_instance, vnf_dict, change_ext_conn_req)
        mock_change_vnf_status.assert_not_called()
        self.assertEqual(mock_change_ext_conn_grant.call_count, 0)
        mock_update_vnf_attributes.assert_called_once()

    def test_sync_db(self):
        self.conductor._sync_db()
        self.vnflcm_driver.sync_db.assert_called_once()
