# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import flask
from flask import current_app as app

import flask_restful
from flask_restful import fields

from tuning_box import db
from tuning_box import errors


def iter_environment_level_values(environment, levels):
    app.logger.debug("Getting environment level values. Env: %s, "
                     "levels: %s", environment.id, levels)
    env_levels = db.EnvironmentHierarchyLevel.get_for_environment(environment)
    app.logger.debug("Environment levels got. Env: %s, levels: %s",
                     environment.id, [l.name for l in env_levels])

    if len(env_levels) < len(levels):
        raise errors.TuningboxNotFound(
            "Levels {0} can't be matched with "
            "environment {1} levels: {2}".format(
                [l[0] for l in levels],
                environment.id,
                [l.name for l in env_levels]
            )
        )
    level_pairs = zip(env_levels, levels)
    for env_level, (level_name, level_value) in level_pairs:
        if env_level.name != level_name:
            raise errors.TuningboxNotFound(
                "Unexpected level name '{0}'. Expected '{1}'.".format(
                    level_name, env_level.name)
            )

        level_value_db = db.get_or_create(
            db.EnvironmentHierarchyLevelValue,
            level=env_level,
            value=level_value,
        )
        yield level_value_db


def get_environment_level_value(environment, levels):
    level_value = None
    for level_value in iter_environment_level_values(environment, levels):
        pass
    return level_value


environment_hierarchy_level_fields = {
    'id': fields.Integer,
    'name': fields.String,
    'environment_id': fields.Integer,
    'parent': fields.String(attribute='parent.name'),
    'values': fields.List(fields.String(attribute='value'))
}


class EnvironmentHierarchyLevelsCollection(flask_restful.Resource):
    method_decorators = [
        flask_restful.marshal_with(environment_hierarchy_level_fields)
    ]

    def get(self, environment_id):
        env = db.get_or_404(db.Environment, environment_id)
        return db.EnvironmentHierarchyLevel.get_for_environment(env)


class EnvironmentHierarchyLevels(flask_restful.Resource):
    method_decorators = [
        flask_restful.marshal_with(environment_hierarchy_level_fields)
    ]

    def _get_query_params(self, environment_id, id_or_name):
        params = {'environment_id': environment_id}
        if isinstance(id_or_name, int):
            params['id'] = id_or_name
        else:
            params['name'] = id_or_name
        return params

    def get(self, environment_id, id_or_name):
        params = self._get_query_params(environment_id, id_or_name)
        level = db.find_or_404(db.EnvironmentHierarchyLevel, **params)
        return level

    @db.with_transaction
    def _do_update(self, environment_id, id_or_name):
        params = self._get_query_params(environment_id, id_or_name)
        level = db.find_or_404(db.EnvironmentHierarchyLevel, **params)
        level.name = flask.request.json.get('name', level.name)

    def put(self, environment_id, id_or_name):
        return self.patch(environment_id, id_or_name)

    def patch(self, environment_id, id_or_name):
        self._do_update(environment_id, id_or_name)
        return None, 204
