# Copyright (c) 2016 b<>com
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
import logging

from django.utils.safestring import mark_safe

from django.utils.translation import gettext_lazy as _
import horizon.exceptions
import horizon.tables
import horizon.tabs
from horizon.utils import memoized
import horizon.workflows

from watcher_dashboard.api import watcher
from watcher_dashboard.content.strategies import tables
from watcher_dashboard.content.strategies import tabs as wtabs

LOG = logging.getLogger(__name__)


class IndexView(horizon.tables.DataTableView):
    table_class = tables.StrategiesTable
    template_name = 'infra_optim/strategies/index.html'
    page_title = _("Strategies")

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['strategies_count'] = self.get_strategies_count()
        return context

    def get_data(self):
        strategies = []
        search_opts = self.get_filters()
        try:
            strategies = watcher.Strategy.list(self.request, **search_opts)
        except Exception as exc:
            LOG.exception(exc)
            horizon.exceptions.handle(
                self.request,
                _("Unable to retrieve strategy information."))
        return strategies

    def get_strategies_count(self):
        return len(self.get_data())

    def get_filters(self):
        filters = {}
        filter_action = self.table._meta._filter_action
        if filter_action:
            filter_field = self.table.get_filter_field()
            if filter_action.is_api_filter(filter_field):
                filter_string = self.table.get_filter_string()
                if filter_field and filter_string:
                    filters[filter_field] = filter_string
        return filters


class DetailView(horizon.tabs.TabbedTableView):
    tab_group_class = wtabs.StrategyDetailTabs
    template_name = 'infra_optim/strategies/details.html'
    redirect_url = 'horizon:admin:strategies:index'
    page_title = _("Strategy Details: {{ strategy.name }}")

    @memoized.memoized_method
    def _get_data(self):
        strategy_uuid = None
        try:
            strategy_uuid = self.kwargs['strategy_uuid']
            strategy = watcher.Strategy.get(self.request, strategy_uuid)
        except Exception as exc:
            LOG.exception(exc)
            msg = _('Unable to retrieve details for strategy "%s".') \
                % strategy_uuid
            horizon.exceptions.handle(
                self.request, msg,
                redirect=self.redirect_url)
        return strategy

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        strategy = self._get_data()
        context["strategy"] = strategy
        # Render parameters_spec in a readable format for the template
        context["strategy_parameters_spec_pretty"] = (
            self._render_pretty_parameters_spec(
                getattr(strategy, 'parameters_spec', None)
            )
        )
        return context

    def get_tabs(self, request, *args, **kwargs):
        strategy = self._get_data()
        return self.tab_group_class(request, strategy=strategy, **kwargs)

    def _render_pretty_parameters_spec(self, spec):
        """Return a human-friendly rendering of parameters_spec.

        Rules:
        - If spec is a JSON string, parse it first.
        - If spec is a JSON schema, prefer the 'properties' mapping.
        - If result is a dict/list, pretty-print as indented YAML in a <pre>.
        - If result is a scalar (str/int/bool), return it as-is.
        - On any error or empty value, return None.
        """
        try:
            obj = spec
            if not obj:
                return None

            # Parse JSON-encoded strings if needed
            if isinstance(obj, str):
                try:
                    obj = json.loads(obj)
                except Exception:
                    # Keep raw string if not JSON
                    return obj

            # If JSON schema-like, extract properties for readability
            if isinstance(obj, dict) and 'properties' in obj \
                    and isinstance(obj.get('properties'), dict):
                obj = obj['properties']

            # Pretty print dict/list as YAML-like preformatted block
            if isinstance(obj, dict | list):
                try:
                    import yaml  # Lazy import
                    dumped = yaml.safe_dump(
                        obj,
                        default_flow_style=False,
                        sort_keys=False,
                    )
                except Exception:
                    dumped = json.dumps(obj, indent=2, ensure_ascii=False)
                return mark_safe(  # noqa: S308  # nosec B703,B308
                    f'<pre style="margin:0">{dumped}</pre>'
                )

            # Scalars
            return obj
        except Exception:
            return None
