# -*- encoding: utf-8 -*-
# Copyright (c) 2015 b<>com
#
# Authors: Jean-Emile DARTOIS <jean-emile.dartois@b-com.com>
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

"""
A :ref:`Solution <solution_definition>` is a set of
:ref:`Actions <action_definition>` generated by a
:ref:`Strategy <strategy_definition>` (i.e., an algorithm) in order to achieve
the :ref:`Goal <goal_definition>` of an :ref:`Audit <audit_definition>`.

A :ref:`Solution <solution_definition>` is different from an
:ref:`Action Plan <action_plan_definition>` because it contains the
non-scheduled list of :ref:`Actions <action_definition>` which is produced by a
:ref:`Strategy <strategy_definition>`. In other words, the list of Actions in
a :ref:`Solution <solution_definition>` has not yet been re-ordered by the
:ref:`Watcher Planner <watcher_planner_definition>`.

Note that some algorithms (i.e. :ref:`Strategies <strategy_definition>`) may
generate several :ref:`Solutions <solution_definition>`. This gives rise to the
problem of determining which :ref:`Solution <solution_definition>` should be
applied.

Two approaches to dealing with this can be envisaged:

-  **fully automated mode**: only the :ref:`Solution <solution_definition>`
   with the highest ranking (i.e., the highest
   :ref:`Optimization Efficiency <efficiency_definition>`)
   will be sent to the :ref:`Watcher Planner <watcher_planner_definition>` and
   translated into concrete :ref:`Actions <action_definition>`.
-  **manual mode**: several :ref:`Solutions <solution_definition>` are proposed
   to the :ref:`Administrator <administrator_definition>` with a detailed
   measurement of the estimated
   :ref:`Optimization Efficiency <efficiency_definition>` and he/she decides
   which one will be launched.
"""

import abc
import six


@six.add_metaclass(abc.ABCMeta)
class BaseSolution(object):
    def __init__(self):
        self._origin = None
        self._model = None
        self._efficacy = 0

    @property
    def efficacy(self):
        return self._efficacy

    @efficacy.setter
    def efficacy(self, e):
        self._efficacy = e

    @property
    def model(self):
        return self._model

    @model.setter
    def model(self, m):
        self._model = m

    @property
    def origin(self):
        return self._origin

    @origin.setter
    def origin(self, m):
        self._origin = m

    @abc.abstractmethod
    def add_action(self,
                   action_type,
                   resource_id,
                   input_parameters=None):
        """Add a new Action in the Action Plan

        :param action_type: the unique id of an action type defined in
            entry point 'watcher_actions'
        :param resource_id: the unique id of the resource to which the
            `Action` applies.
        :param input_parameters: An array of input parameters provided as
            key-value pairs of strings. Each key-pair contains names and
            values that match what was previously defined in the `Action`
            type schema.
        """
        raise NotImplementedError()

    @abc.abstractproperty
    def actions(self):
        raise NotImplementedError()
