# -*- encoding: utf-8 -*-
# Copyright (c) 2015 b<>com
#
# Authors: Jean-Emile DARTOIS <jean-emile.dartois@b-com.com>
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
# implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os

import mock

from watcher.decision_engine.model.collector import base
from watcher.decision_engine.model import element
from watcher.decision_engine.model import model_root as modelroot


class FakerModelCollector(base.BaseClusterDataModelCollector):

    def __init__(self, config=None, osc=None):
        if config is None:
            config = mock.Mock(period=777)
        super(FakerModelCollector, self).__init__(config)

    @property
    def notification_endpoints(self):
        return []

    def load_data(self, filename):
        cwd = os.path.abspath(os.path.dirname(__file__))
        data_folder = os.path.join(cwd, "data")

        with open(os.path.join(data_folder, filename), 'rb') as xml_file:
            xml_data = xml_file.read()

        return xml_data

    def load_model(self, filename):
        return modelroot.ModelRoot.from_xml(self.load_data(filename))

    def execute(self):
        return self._cluster_data_model or self.build_scenario_1()

    def build_scenario_1(self):
        instances = []

        model = modelroot.ModelRoot()
        # number of nodes
        node_count = 5
        # number max of instance per node
        node_instance_count = 7
        # total number of virtual machine
        instance_count = (node_count * node_instance_count)

        for id_ in range(0, node_count):
            node_uuid = "Node_{0}".format(id_)
            hostname = "hostname_{0}".format(id_)
            node_attributes = {
                "id": id_,
                "uuid": node_uuid,
                "hostname": hostname,
                "memory": 132,
                "disk": 250,
                "disk_capacity": 250,
                "vcpus": 40,
            }
            node = element.ComputeNode(**node_attributes)
            model.add_node(node)

        for i in range(0, instance_count):
            instance_uuid = "INSTANCE_{0}".format(i)
            instance_attributes = {
                "uuid": instance_uuid,
                "memory": 2,
                "disk": 20,
                "disk_capacity": 20,
                "vcpus": 10,
                "metadata":
                    '{"optimize": true,"top": "floor","nested": {"x": "y"}}'
            }

            instance = element.Instance(**instance_attributes)
            instances.append(instance)
            model.add_instance(instance)

        mappings = [
            ("INSTANCE_0", "Node_0"),
            ("INSTANCE_1", "Node_0"),
            ("INSTANCE_2", "Node_1"),
            ("INSTANCE_3", "Node_2"),
            ("INSTANCE_4", "Node_2"),
            ("INSTANCE_5", "Node_2"),
            ("INSTANCE_6", "Node_3"),
            ("INSTANCE_7", "Node_4"),
        ]
        for instance_uuid, node_uuid in mappings:
            model.map_instance(
                model.get_instance_by_uuid(instance_uuid),
                model.get_node_by_uuid(node_uuid),
            )

        return model

    def generate_scenario_1(self):
        return self.load_model('scenario_1.xml')

    def generate_scenario_3_with_2_nodes(self):
        return self.load_model('scenario_3_with_2_nodes.xml')

    def generate_scenario_4_with_1_node_no_instance(self):
        return self.load_model('scenario_4_with_1_node_no_instance.xml')

    def generate_scenario_5_with_instance_disk_0(self):
        return self.load_model('scenario_5_with_instance_disk_0.xml')

    def generate_scenario_6_with_2_nodes(self):
        return self.load_model('scenario_6_with_2_nodes.xml')

    def generate_scenario_7_with_2_nodes(self):
        return self.load_model('scenario_7_with_2_nodes.xml')

    def generate_scenario_8_with_4_nodes(self):
        return self.load_model('scenario_8_with_4_nodes.xml')

    def generate_scenario_9_with_3_active_plus_1_disabled_nodes(self):
        return self.load_model(
            'scenario_9_with_3_active_plus_1_disabled_nodes.xml')
