<?php

/*
|--------------------------------------------------------------------------
| Application Routes
|--------------------------------------------------------------------------
|
| Here is where you can register all of the routes for an application.
| It's a breeze. Simply tell Laravel the URIs it should respond to
| and give it the Closure to execute when that URI is requested.
|
*/

Route::pattern('id', '[0-9]+');
Route::pattern('uri_id', '[0-9]+');
Route::pattern('active', '(true|false)');
Route::pattern('hint', '(access-token|refresh-token)');
Route::pattern('scope_id', '[0-9]+');

Route::group(['namespace' => 'App\Http\Controllers', 'middleware' => 'web' ], function() {

    // openid endpoints
    Route::group(array('middleware' => ['ssl']), function () {

        Route::get('/', "HomeController@index");

        // OpenId endpoints

        Route::group(['namespace' => 'OpenId' ], function() {

            Route::get('/discovery', "DiscoveryController@idp");
            Route::get("/discovery/users/{identifier}","DiscoveryController@user")->where(array('identifier' => '[\d\w\.\#]+'));
            //op endpoint url
            Route::post('/accounts/openid2', 'OpenIdProviderController@endpoint');
            Route::get('/accounts/openid2', 'OpenIdProviderController@endpoint');
        });

        /*
        * If the Claimed Identifier was not previously discovered by the Relying Party
        * (the "openid.identity" in the request was "http://specs.openid.net/auth/2.0/identifier_select"
        * or a different Identifier, or if the OP is sending an unsolicited positive assertion),
        * the Relying Party MUST perform discovery on the Claimed Identifier in
        * the response to make sure that the OP is authorized to make assertions about the Claimed Identifier.
        */
        Route::get("/{identifier}", "UserController@getIdentity");
        //user interaction
        Route::get('/accounts/user/login', "UserController@getLogin");
        Route::post('/accounts/user/login', ['middleware' => 'csrf', 'uses' => 'UserController@postLogin']);
        Route::get('/accounts/user/login/cancel', "UserController@cancelLogin");
    });

    //oauth2 endpoints

    Route::group(['namespace'=> 'OAuth2', 'middleware' => ['ssl']], function () {
        Route::get('/.well-known/openid-configuration', "OAuth2ProviderController@discovery");
    });

    Route::group(['namespace' => 'OAuth2' , 'prefix' => 'oauth2', 'middleware' => ['ssl']], function () {
        Route::get('/check-session', "OAuth2ProviderController@checkSessionIFrame");
        Route::get('/end-session', "OAuth2ProviderController@endSession");
        Route::get('/end-session/cancel', "OAuth2ProviderController@cancelLogout");
        Route::post('/end-session', "OAuth2ProviderController@endSession");

        //authorization endpoint
        Route::any('/auth', "OAuth2ProviderController@auth");
        // OIDC
        // certificates
        Route::get('/certs', "OAuth2ProviderController@certs");
        // discovery document
        Route::get('/.well-known/openid-configuration', "OAuth2ProviderController@discovery");
        //token endpoint
        Route::group(array('prefix' => 'token'), function () {
            Route::post('/', "OAuth2ProviderController@token");
            Route::post('/revoke', "OAuth2ProviderController@revoke");
            Route::post('/introspection', "OAuth2ProviderController@introspection");
        });
    });

    Route::group(array('middleware' => ['ssl', 'auth']), function () {
        Route::get('/accounts/user/consent', "UserController@getConsent");
        Route::post('/accounts/user/consent', ['middleware' => 'csrf', 'uses' => 'UserController@postConsent']);
        Route::any("/accounts/user/logout", "UserController@logout");
        Route::any("/accounts/user/profile", "UserController@getProfile");
        Route::any("/accounts/user/profile/trusted_site/delete/{id}", "UserController@deleteTrustedSite");
        Route::post('/accounts/user/profile/update', 'UserController@postUserProfileOptions');
    });

    Route::group(['prefix' => 'admin', 'middleware' => ['ssl', 'auth']], function () {
        //client admin UI
        Route::get('clients/edit/{id}', ['middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'AdminController@editRegisteredClient']);
        Route::get('clients', 'AdminController@listOAuth2Clients');
        Route::get('/grants', 'AdminController@editIssuedGrants');

        //oauth2 server admin UI
        Route::group(array('middleware' => ['oauth2.currentuser.serveradmin']), function () {
            Route::get('/api-scope-groups', 'AdminController@listApiScopeGroups');
            Route::get('/api-scope-groups/{id}', 'AdminController@editApiScopeGroup');
            Route::get('/resource-servers', 'AdminController@listResourceServers');
            Route::get('/resource-server/{id}', 'AdminController@editResourceServer');
            Route::get('/api/{id}', 'AdminController@editApi');
            Route::get('/scope/{id}', 'AdminController@editScope');
            Route::get('/endpoint/{id}', 'AdminController@editEndpoint');
            Route::get('/locked-clients', 'AdminController@listLockedClients');
            // server private keys
            Route::get('/private-keys', 'AdminController@listServerPrivateKeys');
        });

        Route::group(array('middleware' => ['openstackid.currentuser.serveradmin']), function () {
            Route::get('/locked-users', 'AdminController@listLockedUsers');
            Route::get('/server-config', 'AdminController@listServerConfig');
            Route::post('/server-config', 'AdminController@saveServerConfig');
            Route::get('/banned-ips', 'AdminController@listBannedIPs');
        });
    });

    //Admin Backend API

    Route::group([
        'namespace'  =>  'Api',
        'prefix'     => 'admin/api/v1',
        'middleware' => ['ssl', 'auth']], function () {

        Route::group(array('prefix' => 'users'), function () {
            Route::delete('/{id}/locked', array('middleware' => ['openstackid.currentuser.serveradmin.json'], 'uses' => 'UserApiController@unlock'));
            Route::delete('/{id}/token/{value}', array('middleware' => ['currentuser.checkroute'], 'uses' => 'UserApiController@revokeToken'));
            Route::get('/fetch', array('uses' => "UserApiController@fetch"));
        });

        Route::group(array('prefix' => 'banned-ips', 'middleware' => ['openstackid.currentuser.serveradmin.json']), function () {
            Route::get('/{id}', "ApiBannedIPController@get");
            Route::get('/', "ApiBannedIPController@getByPage");
            Route::delete('/{id?}', "ApiBannedIPController@delete");
        });

        //client api
        Route::group(array('prefix' => 'clients'), function () {

            // public keys
            Route::post('/{id}/public_keys', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientPublicKeyApiController@create'));
            Route::get('/{id}/public_keys', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientPublicKeyApiController@getByPage'));
            Route::delete('/{id}/public_keys/{public_key_id}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientPublicKeyApiController@delete'));
            Route::put('/{id}/public_keys/{public_key_id}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientPublicKeyApiController@update'));

            Route::post('/', array('middleware' => ['currentuser.checkroute'], 'uses' => 'ClientApiController@create'));
            Route::put('/', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@update'));
            Route::get('/{id}', "ClientApiController@get");
            Route::get('/', array('middleware' => ['currentuser.checkroute'], 'uses' => 'ClientApiController@getByPage'));
            Route::delete('/{id}', array('middleware' => ['oauth2.currentuser.owns.client'], 'uses' => 'ClientApiController@delete'));
            //allowed redirect uris endpoints
            Route::get('/{id}/uris', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@getRegisteredUris'));
            Route::post('/{id}/uris', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@addAllowedRedirectUri'));
            Route::delete('/{id}/uris/{uri_id}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@deleteClientAllowedUri'));

            //allowedApiResourceServerControllert('/{id}/origins', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@geAllowedOrigins'));
            Route::post('/{id}/origins', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@addAllowedOrigin'));
            Route::delete('/{id}/origins/{origin_id}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@deleteClientAllowedOrigin'));
            Route::delete('/{id}/lock', array('middleware' => ['openstackid.currentuser.serveradmin.json'], 'uses' => 'ClientApiController@unlock'));
            Route::put('/{id}/secret', array('middleware' => ['oauth2.currentuser.owns.client'], 'uses' => 'ClientApiController@regenerateClientSecret'));
            Route::put('/{id}/use-refresh-token', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@setRefreshTokenClient'));
            Route::put('/{id}/rotate-refresh-token', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@setRotateRefreshTokenPolicy'));
            Route::get('/{id}/access-token', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@getAccessTokens'));
            Route::get('/{id}/refresh-token', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@getRefreshTokens'));
            Route::get('/me/access-tokens', array('middleware' => [], 'uses' => 'ClientApiController@getAccessTokensByCurrentUser'));
            Route::get('/me/refresh-tokens', array('middleware' => [], 'uses' => 'ClientApiController@getRefreshTokensByCurrentUser'));
            Route::delete('/{id}/token/{value}/{hint}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@revokeToken'));
            Route::put('/{id}/scopes/{scope_id}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@addAllowedScope'));
            Route::delete('/{id}/scopes/{scope_id}', array('middleware' => ['oauth2.currentuser.allow.client.edition'], 'uses' => 'ClientApiController@removeAllowedScope'));
            Route::put('/{id}/active', array('middleware' => ['oauth2.currentuser.owns.client'], 'uses' => 'ClientApiController@activate'));
            Route::delete('/{id}/active', array('middleware' => ['oauth2.currentuser.owns.client'], 'uses' => 'ClientApiController@deactivate'));

        });

        // resource servers
        Route::group(array('prefix' => 'resource-servers', 'middleware' => ['oauth2.currentuser.serveradmin.json']), function () {
            Route::get('/{id}', "ApiResourceServerController@get");
            Route::get('/', "ApiResourceServerController@getByPage");
            Route::post('/', "ApiResourceServerController@create");
            Route::delete('/{id}', "ApiResourceServerController@delete");
            Route::put('/', "ApiResourceServerController@update");
            Route::put('/{id}/client-secret', "ApiResourceServerController@regenerateClientSecret");
            Route::put('/{id}/active', "ApiResourceServerController@activate");
            Route::delete('/{id}/active', "ApiResourceServerController@deactivate");
        });

        // api scope groups
        Route::group(array('prefix' => 'api-scope-groups', 'middleware' => ['oauth2.currentuser.serveradmin.json']), function () {
            Route::get('/{id}', "ApiScopeGroupController@get");
            Route::get('/', "ApiScopeGroupController@getByPage");
            Route::put('/', "ApiScopeGroupController@update");
            Route::post('/', "ApiScopeGroupController@create");
            Route::delete('/{id}', "ApiScopeGroupController@delete");
            Route::put('/{id}/active', "ApiScopeGroupController@activate");
            Route::delete('/{id}/active', "ApiScopeGroupController@deactivate");
        });

        // apis
        Route::group(array('prefix' => 'apis', 'middleware' => ['oauth2.currentuser.serveradmin.json']), function () {
            Route::get('/{id}', "ApiController@get");
            Route::get('/', "ApiController@getByPage");
            Route::post('/', "ApiController@create");
            Route::delete('/{id}', "ApiController@delete");
            Route::put('/', "ApiController@update");
            Route::put('/{id}/active', "ApiController@activate");
            Route::delete('/{id}/active', "ApiController@deactivate");
        });

        // scopes
        Route::group(array('prefix' => 'scopes', 'middleware' => ['oauth2.currentuser.serveradmin.json']), function () {
            Route::get('/{id}', "ApiScopeController@get");
            Route::get('/', "ApiScopeController@getByPage");
            Route::post('/', "ApiScopeController@create");
            Route::delete('/{id}', "ApiScopeController@delete");
            Route::put('/', "ApiScopeController@update");
            Route::put('/{id}/active', "ApiScopeController@activate");
            Route::delete('/{id}/active', "ApiScopeController@deactivate");
        });

        // endpoints
        Route::group(array('prefix' => 'endpoints', 'middleware' => ['oauth2.currentuser.serveradmin.json']), function () {
            Route::get('/{id}', "ApiEndpointController@get");
            Route::get('/', "ApiEndpointController@getByPage");
            Route::post('/', "ApiEndpointController@create");
            Route::delete('/{id}', "ApiEndpointController@delete");
            Route::put('/', "ApiEndpointController@update");
            Route::put('/{id}/scope/{scope_id}', "ApiEndpointController@addRequiredScope");
            Route::delete('/{id}/scope/{scope_id}', "ApiEndpointController@removeRequiredScope");
            Route::put('/{id}/active', "ApiEndpointController@activate");
            Route::delete('/{id}/active', "ApiEndpointController@deactivate");
        });

        // private keys
        Route::group(array('prefix' => 'private-keys', 'middleware' => ['oauth2.currentuser.serveradmin.json']), function () {
            Route::get('/', "ServerPrivateKeyApiController@getByPage");
            Route::post('/', "ServerPrivateKeyApiController@create");
            Route::delete('/{id}', "ServerPrivateKeyApiController@delete");
            Route::put('/{id}', "ServerPrivateKeyApiController@update");
        });

    });
});

//OAuth2 Protected API
Route::group(
    [
        'namespace'  => 'App\Http\Controllers\Api\OAuth2',
        'prefix'     => 'api/v1',
        'middleware' => ['api']
    ], function () {

    Route::group(array('prefix' => 'users'), function () {
        Route::get('/me', 'OAuth2UserApiController@me');
        Route::get('/info', 'OAuth2UserApiController@userInfo');
        Route::post('/info', 'OAuth2UserApiController@userInfo');
    });
});